
#include "build.h"



/***************************************************************************/
/* Create an isotropic initial condition (molecules given random positions
   and orientations).

input:
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species)
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int isotropic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int graph_switch, 
      int n_mols, double length1, double length2, double skin, double **h, 
      double **h_inv, long *i_ran, int *mol_first_atm, int *n_atoms_per_mol, 
      double ***temp_atm_pos,  double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, int *n_bonds_per_mol,
      int **temp_bonds_1, int **temp_bonds_2, 
      int n_species, int *n_mols_per_species, int *mol_species, int *atom_rel,
      int *atom_type, int species_choice, int *comb_col, 
      int resph, double radius_sph, double radius_cyl) 
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, skip_j, i, j, k;
   double norm, s[2][3], u[2][3], s1[3], u1[3], norm1, norm2;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp, signe, **rel_atom_coords;
   short keep_going = 1;
   
   int i_species, j_species;

printf("side : %g %g %g\n", h[0][0], h[1][1], h[2][2]);
fflush(NULL);
   
   /* allocate memory for 'temporary' relative atomic coordinates. */
   rel_atom_coords = allocate_2d_array(3 * n_mols, 3, sizeof(double));

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
            }

         /* Generate scaled position. */
         for (j = 0; j < 3; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;

         /* Molecular position. */
         mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0];
         mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1];
         mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
           psi = TWO_PI * ran3(i_ran); 
           phi = TWO_PI * ran3(i_ran); 
         cos_psi = cos(psi);
         sin_psi = sin(psi);
         cos_theta = 2.0 * ran3(i_ran) - 1.0;
         sin_theta = sqrt(1.0 - SQR(cos_theta));
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i_species][i][0];
            y_rot = temp_atm_pos[i_species][i][1];
            z_rot = temp_atm_pos[i_species][i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
             mol_species, h_inv, atom_coords, scaled_atom_coords);

        for (i = 0; i < n_bonds_per_mol[i_species]; ++i) {
         norm1 = norm2 = 0.0;
            abs = skip +i;
         for (k = 0; k < 3; ++k) {
            s[i][k] = scaled_atom_coords[abs + 1][k] -  
                      scaled_atom_coords[skip][k];  
            s[i][k] -= NINT(s[i][k]);
         if ( i_species == 0)
            u[i][k] = s[i][k] * h[k][k] /length1;
         else
            u[i][k] = s[i][k] * h[k][k] /length2;
            s[i][k] = s[i][k] * 0.5 + scaled_atom_coords[skip][k];
            s[i][k] -= NINT(s[i][k]);

            norm1 += u[i][k] * u[i][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-04)){
            printf("pb with norm., first norm1 %g length1 %g\n", norm1, length1);
            exit(1);
            }
        }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           j_species = mol_species[j_mol];
           
           for (i = 0; i < n_bonds_per_mol[j_species]; ++i) { 
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + i][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
          if (j_species == 0)
               u1[k] = s1[k] * h[k][k] / length1;
           else
               u1[k] = s1[k] * h[k][k] / length2;
               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-04){
                printf("pb with norm.norm1 %g\n", norm1);
                exit(1);
             }
             for (j = 0; j < n_bonds_per_mol[i_species]; ++j)
             if (overlap(s[j], u[j], s1, u1, length1, length2, 0.0, h)) {
                 overlap_flag = 1;
                 break;
                 }
             if (overlap_flag) 
               break;
           }
           if (overlap_flag) 
             break;
         }
/*
if (overlap_flag){
   printf("i_mol = %d, j_mol = %d\n", i_mol, j_mol);
printf("  sx = %g, sy = %g, sz = %g\n", scaled_mol_coords[i_mol][0], scaled_mol_coords[i_mol][1], scaled_mol_coords[i_mol][2]); 
exit(1);
   }
*/
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Plot molecular configuration. */

#ifdef GRAPHICS
  if (graph_switch > 0){
          redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1,
                 mol_species, mol_first_atm, n_atoms_per_mol, n_bonds_per_mol,
                 mol_coords, rel_atom_coords, h, temp_bonds_1, temp_bonds_2,
                 species_choice, atom_type, comb_col,
                 resph, radius_sph, radius_cyl, &keep_going);
}
#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
/*printf("  sx = %g, sy = %g, sz = %g\n", scaled_mol_coords[i_mol][0], scaled_mol_coords[i_mol][1], scaled_mol_coords[i_mol][2]); 
      fflush(NULL);
*/
   }

   /* Free memory of temporary array. */
   free_2d_array(rel_atom_coords, 3 * n_mols);

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/* Create a smectic initial condition (molecules given random positions
   and orientations).

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int smectic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int graph_switch, int polar_switch, int n_mols, 
      int n_layers, 
      double length1, double length2, double skin, 
      double **h, double **h_inv, long *i_ran, int *mol_first_atm, 
      int *n_atoms_per_mol, double ***temp_atm_pos,  double **mol_coords, 
      double **scaled_mol_coords, double **atom_coords, 
      double **scaled_atom_coords, int *n_bonds_per_mol,
      int **temp_bonds_1, int **temp_bonds_2, 
      int n_species, int *n_mols_per_species, int *mol_species, int *atom_rel, 
      int *atom_type, int species_choice, int *comb_col,
      int resph, double radius_sph, double radius_cyl, double *length, 
      int *mol_first_sphero) 
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, skip_j, i, j, k,
      **layer_counter, *n_per_layer, i_layer, i_number, layer_tmp;
   double norm, s[2][3], u[2][3], s1[3], u1[3], norm1, norm2;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp, signe, scaled_layer_spacing,
      **rel_atom_coords;
   short keep_going = 1;
   int i_species, j_species, abs_j, skip_sphero, skip_sphero_j, abs_sphero;
  

   /* Allocate memory for temporary array. */
   layer_counter = allocate_2d_array(n_species, n_layers, sizeof(int));
   n_per_layer = allocate_1d_array(n_species, sizeof(int));
   rel_atom_coords = allocate_2d_array(3 * n_mols, 3, sizeof(double));

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {
       layer_tmp = n_mols_per_species[i_species] % n_layers;
       if (layer_tmp == 0)
          n_per_layer[i_species] = n_mols_per_species[i_species] /
                                                         (double)n_layers;
       else
          error_exit("Wrong number of mol per layer");
   }
   scaled_layer_spacing = 1.0 / n_layers;
   printf(" scaled_layer_spacing %g\n", scaled_layer_spacing);

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species and label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];
      skip_sphero = mol_first_sphero[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
            }

          /* Pick a layer at random. */
          do {
             i_layer = (int) (n_layers * ran3(i_ran));
             i_number = (i_layer % 2 == 0);
             if (i_layer == n_layers) i_layer = 0.0;
         } while (layer_counter[i_species][i_layer] == n_per_layer[i_species]);

         /* Generate scaled position. */
         if (i_species == 0){
         for (j = 0; j < NDIM - 1; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;
         scaled_mol_coords[i_mol][2] = (i_layer + 0.5) * scaled_layer_spacing
                                       - 0.5;
        } else{
         for (j = 0; j < NDIM - 1; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;
         scaled_mol_coords[i_mol][2] = (i_layer + 0.5) * scaled_layer_spacing
                                       - 0.5;
        }

         /* Molecular position. */
         mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0];
         mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1];
         mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         /* If polar_switch == 2, built a smectic antipolar order.
            If polar_switch == 1, built a smectic polar order.
            If polar_switch == 0, built a smectic azimutally isotropic order. */

         if (polar_switch == 2)
            if (i_number) {
               psi = 0.0;
               phi = 0.0;
            }
            else {
               psi = 0.0;
               phi = PI;
            }
         else if (polar_switch == 1) {
            psi = 0.0;
            phi = 0.0;
         }
         else{ 
            psi = TWO_PI * ran3(i_ran);
            phi = TWO_PI * ran3(i_ran);
         }


         cos_psi = cos(psi);
         sin_psi = sin(psi);
     /*    signe = ran3(i_ran) - 0.5;
         if (signe == 0.0) signe = 1.0;
         else signe = signe / fabs(signe);
*/
         signe = 1.0;
         cos_theta = signe;
         sin_theta = 0.0;
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i_species][i][0];
            y_rot = temp_atm_pos[i_species][i][1];
            z_rot = temp_atm_pos[i_species][i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
             mol_species, h_inv, atom_coords, scaled_atom_coords);

         for (i = 0; i < n_bonds_per_mol[i_species]; ++i) {
         norm1 = norm2 = 0.0;
            abs_sphero = skip_sphero +i;
            abs = skip +i;

         for (k = 0; k < 3; ++k) {
            s[i][k] = scaled_atom_coords[abs + 1][k] -
                      scaled_atom_coords[skip][k];
            s[i][k] -= NINT(s[i][k]);
            u[i][k] = s[i][k] * h[k][k] /length[abs_sphero];
            s[i][k] = s[i][k] * 0.5 + scaled_atom_coords[skip][k];
            s[i][k] -= NINT(s[i][k]);

            norm1 += u[i][k] * u[i][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-04)){
     printf("pb with norm., first norm1 %g length[%d] %g\n", norm1, abs_sphero,
                         length[abs_sphero]);
            exit(1);
            }
        }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           skip_sphero_j = mol_first_sphero[j_mol];
           
           j_species = mol_species[j_mol];

           for (j = 0; j < n_bonds_per_mol[j_species]; ++j) { 
               abs_j = skip_sphero_j + j;
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + j][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
               u1[k] = s1[k] * h[k][k] / length[abs_j];

               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-03){
                printf("pb with norm. norm1 %g\n", norm1);
                exit(1);
             }
             for (i = 0; i < n_bonds_per_mol[i_species]; ++i){
                 abs = skip_sphero + i;
             if (overlap(s[i], u[i], s1, u1, abs, abs_j, length, 0.0, h)) {
                 overlap_flag = 1;
                 break;
                 }
             }
             if (overlap_flag) 
               break;
           }
           if (overlap_flag) 
             break;
         }
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Increment counter. */
      ++layer_counter[i_species][i_layer];

      /* Plot molecular configuration. */

#ifdef GRAPHICS
  if (graph_switch > 0){
          redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1,
                 mol_species, mol_first_atm, n_atoms_per_mol, n_bonds_per_mol,
                 mol_coords, rel_atom_coords, h, temp_bonds_1, temp_bonds_2, 
                 species_choice, atom_type, comb_col,
                 resph, radius_sph, radius_cyl, &keep_going);
}

#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
      fflush(NULL);
   }

   /* Free memory of temporary array. */
   free_2d_array(rel_atom_coords, 3 * n_mols);

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/* Create an nematic initial condition (molecules given random positions
   and orientations).

input:
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species)
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int nematic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int graph_switch, int polar_switch,
      int n_mols, double length1, double length2, double skin, double **h, 
      double **h_inv, long *i_ran, int *mol_first_atm, int *n_atoms_per_mol, 
      double ***temp_atm_pos,  double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, int *n_bonds_per_mol,
      int **temp_bonds_1, int **temp_bonds_2, 
      int n_species, int *n_mols_per_species, int *mol_species, int *atom_rel,
      int *atom_type, int species_choice, int *comb_col, 
      int resph, double radius_sph, double radius_cyl) 
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, skip_j, i, j, k;
   double norm, s[2][3], u[2][3], s1[3], u1[3], norm1, norm2;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp, signe, **rel_atom_coords;
   short keep_going = 1;
   
   int i_species, j_species;

printf("side : %g %g %g\n", h[0][0], h[1][1], h[2][2]);
fflush(NULL);
   
   /* allocate memory for 'temporary' relative atomic coordinates. */
   rel_atom_coords = allocate_2d_array(3 * n_mols, 3, sizeof(double));

printf("polar_switch = %d\n", polar_switch);
   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
            }

         /* Generate scaled position. */
         for (j = 0; j < 3; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;

         /* Molecular position. */
         mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0];
         mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1];
         mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         if (polar_switch) {
           psi = 0.0; 
           psi = 0.0; 
         }
         else {
           psi = TWO_PI * ran3(i_ran); 
           phi = TWO_PI * ran3(i_ran); 
         }

         cos_psi = cos(psi);
         sin_psi = sin(psi);
/*         signe = ran3(i_ran) - 0.5;
         if (signe == 0.0) signe = 1.0;
         else signe = signe / fabs(signe);  
*/
         signe = 1.0;
         cos_theta = signe;
         sin_theta = 0.0;
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i_species][i][0];
            y_rot = temp_atm_pos[i_species][i][1];
            z_rot = temp_atm_pos[i_species][i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
             mol_species, h_inv, atom_coords, scaled_atom_coords);

        for (i = 0; i < n_bonds_per_mol[i_species]; ++i) {
         norm1 = norm2 = 0.0;
            abs = skip +i;
         for (k = 0; k < 3; ++k) {
            s[i][k] = scaled_atom_coords[abs + 1][k] -  
                      scaled_atom_coords[skip][k];  
/* printf("s %g nint %d\n", s[i][k], NINT(s[i][k])); */
            s[i][k] -= NINT(s[i][k]);
/* printf("coords1 %g coords2 %g s %g\n", scaled_atom_coords[abs + 1][k],
         scaled_atom_coords[skip][k], s[i][k]); */
         if ( i_species == 0)
            u[i][k] = s[i][k] * h[k][k] /length1;
         else
            u[i][k] = s[i][k] * h[k][k] /length2;
            s[i][k] = s[i][k] * 0.5 + scaled_atom_coords[skip][k];
            s[i][k] -= NINT(s[i][k]);

            norm1 += u[i][k] * u[i][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-04)){
            printf("pb with norm., first norm1 %g length1 %g\n", norm1, length1);
            exit(1);
            }
        }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           j_species = mol_species[j_mol];
           
           for (i = 0; i < n_bonds_per_mol[j_species]; ++i) { 
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + i][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
          if (j_species == 0)
               u1[k] = s1[k] * h[k][k] / length1;
           else
               u1[k] = s1[k] * h[k][k] / length2;
               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-04){
                printf("pb with norm.norm1 %g\n", norm1);
                exit(1);
             }
             for (j = 0; j < n_bonds_per_mol[i_species]; ++j)
             if (overlap(s[j], u[j], s1, u1, length1, length2, 0.0, h)) {
                 overlap_flag = 1;
                 break;
                 }
             if (overlap_flag) 
               break;
           }
           if (overlap_flag) 
             break;
         }
/*
if (overlap_flag){
   printf("i_mol = %d, j_mol = %d\n", i_mol, j_mol);
printf("  sx = %g, sy = %g, sz = %g\n", scaled_mol_coords[i_mol][0], scaled_mol_coords[i_mol][1], scaled_mol_coords[i_mol][2]); 
exit(1);
   }
*/
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Plot molecular configuration. */

#ifdef GRAPHICS
  if (graph_switch > 0){
          redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1,
                 mol_species, mol_first_atm, n_atoms_per_mol, n_bonds_per_mol,
                 mol_coords, rel_atom_coords, h, temp_bonds_1, temp_bonds_2,
                 species_choice, atom_type, comb_col,
                 resph, radius_sph, radius_cyl, &keep_going);
}
#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
/*printf("  sx = %g, sy = %g, sz = %g\n", scaled_mol_coords[i_mol][0], scaled_mol_coords[i_mol][1], scaled_mol_coords[i_mol][2]); 
      fflush(NULL);
*/
   }

   /* Free memory of temporary array. */
   free_2d_array(rel_atom_coords, 3 * n_mols);

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

