/* Input/output routines for builder module. */

#include "build.h"
#include "ff.h"

/*************************************************************************/
/* Read builder module input parameter file. */
/*
input :
    name of the file containing the parameters used in the builder (par_file)
output :
    Force field file for masses (mass_file)
    Force field file for LJ parameters (lj_mass)
    Force field file for exp-6 parameters (e6_mass)
    Switch for periodic system (period_switch - 1 means periodic)
    Number of molecular species (n_species)
    Array of number of molecules of each specie in the unit cell 
                                                       (n_mols_per_unit_cell)
    Array of files of the molecular structures (struct_file)
    Switch for the combination rule between atomic site (vdw_rad_switch)
    Overlap criterion between atomi sites (overlap_criterion)
    Switch to chose the initial state being built (init_switch)
    Number of smectic layers present in the initial state (n_layers)
    Layer spacing (layer_spacing)
    Number density for the molecules (rho_star)
    Number of unit cells (n_unit_cells)
    Aspect ratio for the box (aspect_xy, aspect_xz)
    Array of the molecular principal axe chosen to be along the z cartesian
                                                          axe (principal)
    Seed for the random generator (seed)
    Name of the output header file (header_file)
    Name of the output configuration file (config_file)
*/

void read_new_build_params(char *par_file, char **mass_file, char **pot_file, 
   char **color_file,
   int *period_switch, int *n_species, int **n_mols_per_unit_cell,
   char ***struct_file, int *trial_switch, int *vdw_rad_switch, 
   double *overlap_criterion, int *init_switch, int *n_layers, 
   double *layer_spacing, double *rho_star, int *n_unit_cells, 
   double *aspect_xy, double *aspect_xz, int **principal,  
   long *seed,
   double *length1, double *length2, int *polar_switch, double *skin, 
   double *radius_sph, double *radius_cyl, int *resph, int *antialias_switch,
   int *graph_switch, int *species_choice,
   double *dr_max, double *ratio_du_dr, double *dside_max,
   char **header_file, char **config_file)
{
   int i_species;
   FILE *f_param;
   char line[MAX_LINE];

   /* Open parameter file. */
   if ((f_param = fopen(par_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_new_build_params");

   /* Read in builder parameters. */
   *mass_file = allocate_1d_array(F_MAX, sizeof(char));
   *pot_file = allocate_1d_array(F_MAX, sizeof(char));
   *color_file = allocate_1d_array(F_MAX, sizeof(char));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%s %s %s\n", *mass_file, *pot_file, *color_file);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", &(*period_switch));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", &(*n_species));
   fgets(line, MAX_LINE, f_param);
   /* Allocate memory for some parameters */
   *n_mols_per_unit_cell = allocate_1d_array(*n_species, sizeof(int));
   *struct_file = allocate_2d_array(*n_species, F_MAX, sizeof(char));

   for (i_species = 0; i_species < *n_species; ++i_species) {
      fscanf(f_param, "%d", &(*n_mols_per_unit_cell)[i_species]);
      fscanf(f_param, "%s", (*struct_file)[i_species]);
   }
   fscanf(f_param, "\n");
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", trial_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", &(*vdw_rad_switch));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf\n", &(*overlap_criterion));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d %lf\n", &(*init_switch), &(*n_layers), 
     &(*layer_spacing));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf\n",&(*rho_star));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %lf %lf\n", &(*n_unit_cells),  &(*aspect_xy), 
     &(*aspect_xz));
   fgets(line, MAX_LINE, f_param);

   *principal = allocate_1d_array(*n_species, sizeof(int));
   for (i_species = 0; i_species < *n_species; ++i_species) 
      fscanf(f_param, "%d", &(*principal)[i_species]);
   fscanf(f_param, "\n");
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%ld\n", &(*seed));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf %d %lf\n", &(*length1), &(*length2), &(*polar_switch), &(*skin));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf %d\n", &(*radius_sph), &(*radius_cyl), &(*resph));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d %d\n", &(*antialias_switch), &(*graph_switch), 
                     &(*species_choice));
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf %lf\n", &(*dr_max), &(*ratio_du_dr), 
                     &(*dside_max));
   fgets(line, MAX_LINE, f_param);

/* Read in builder parameters. */
   *header_file = allocate_1d_array(F_MAX, sizeof(char));
   *config_file = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s %s\n", *header_file, *config_file); 

   /* Close parameter file. */
   fclose(f_param);
}

/*************************************************************************/

/* Write periodicity switch, number of molecular species, molecular structures,
   numbers of molecules of each species, and list of molecular species to
   header of output file. */
/*
input :
       Switch for periodic system (period_switch)
       Number of species (n_species)
       Array of number of atom in template molecule of each specie 
                                                           (n_atoms_per_mol)
       Array of template atom - atomic labels (temp_atm_lab)
       Array of template atom - atomic types (temp_atm_type)
       Array of template atom - atom branching number (temp_atm_nbr)
       Array of template atom - label of branched atoms (temp_atm_br)
       Array of template atom - order of the bonds branched to an atom
                                                            (temp_atm_ord)
       Array of template atom - realtive position of atoms in the template
                                molecule                    (temp_atm_pos)
       Array of template atom - atomic site charges (temp_atm_chg) 
       Arrays of template atom - LJ atomic site parameters (temp_atm_sigma,
                                                            temp_atm_eps)
       Array of number of bonds in a template molecule (n_bonds_per_mol)
       Arrays of template molecule - label of the 2 atoms sharing a bond
							(temp_bonds_1,
                                                         temp_bonds_2) 
       Array of template molecule - order of a bond (temp_bonds_ord)
       Total number of molecules (n_mols)
       Specie label of the molecules (mol_species)

output :
       Name of the output header file
*/

void write_header_direct(char *header_file, int period_switch, int n_species,
  int *n_atoms_per_mol, int **temp_atm_lab, char ***temp_atm_type, 
  int **temp_atm_nbr, int ***temp_atm_br, double ***temp_atm_ord, 
  double ***temp_atm_pos, double **temp_atm_chg, int *n_bonds_per_mol, 
  int **temp_bonds_1, int **temp_bonds_2, double **temp_bonds_ord, 
  int *n_mols_per_species, int n_mols, int *mol_species)
{
   int i, j, i_species;
   FILE *f_output;

   /* Open header file. */
   if ((f_output = fopen(header_file, "w")) == NULL)
      error_exit("Unable to open header file in write_config_direct");

   /* Write out periodicity switch. */
   fprintf(f_output,"%d\n",period_switch);

   /* Write out number of molecular species. */
   fprintf(f_output,"%d\n",n_species);

   /* Write out molecular structures and number of molecules for all species.*/
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Write out number of atoms per molecule. */
      fprintf(f_output,"%d\n", n_atoms_per_mol[i_species]);

      /* Write out template atom attributes. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i)
      {
      fprintf(f_output,"%d %s %d ", temp_atm_lab[i_species][i],
         temp_atm_type[i_species][i], temp_atm_nbr[i_species][i]);
         for (j = 0; j < temp_atm_nbr[i_species][i]; ++j)
          fprintf(f_output,"%d %g ",temp_atm_br[i_species][i][j],
            temp_atm_ord[i_species][i][j]);
         fprintf(f_output,"\n");
         fprintf(f_output,"%g %g %g %g\n", temp_atm_pos[i_species][i][0], 
           temp_atm_pos[i_species][i][1], temp_atm_pos[i_species][i][2], 
           temp_atm_chg[i_species][i]);
 /*     fprintf(f_output,"%g %g\n", temp_atm_sigma[i_species][i],
           temp_atm_eps[i_species][i]);  */
      }
      /* Write out number of bonds per molecule. */
      fprintf(f_output,"%d\n", n_bonds_per_mol[i_species]);

      /* Write out template bond attributes. */
      for (i = 0; i < n_bonds_per_mol[i_species]; ++i)
      fprintf(f_output, "%d %d %g\n", temp_bonds_1[i_species][i], 
           temp_bonds_2[i_species][i], temp_bonds_ord[i_species][i]);

      /* Write out number of molecules. */
      fprintf(f_output, "%d\n", n_mols_per_species[i_species]);
   }

   /* Write out list of molecular species. */
   for (i = 0; i < n_mols; ++i)
   fprintf(f_output, "%d ", mol_species[i]);
   fprintf(f_output, "\n");

   /* Close header file. */
   fclose(f_output);
}

/*************************************************************************/
/* Write atomic configuration to config file. */
/*
input :
      Switch about periodicity (period_switch)
      Array of box matrix (h)
      Total number of atoms (n_atoms)
      Array of cartesian coordinates of atoms (atom_coords)
      Array of velocities of atoms (atom_vels)
output :
      Name of the configuration output file (config_file)
*/

void write_config_direct(char *config_file, double **h,
   int n_atoms, double length1, double length2, double dr_max, double ratio_du_dr, 
   double dside_max, double skin, double **atom_coords)
{
   FILE *f_output;
   int i, j;

   /* Open config file. */
   if ((f_output = fopen(config_file, "w")) == NULL)
      error_exit("Unable to open configuration file in write_config_direct");

   /* Write out box matrix if params.period_switch == 1. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fprintf(f_output, "%g ", h[i][j]);
   fprintf(f_output, "\n");
   fprintf(f_output, "%g %g\n", length1, length2);
   fprintf(f_output, "%g %g %g %g\n", dr_max, ratio_du_dr, dside_max, skin);

   /* Write out atomic positions and velocities. */
   for (i = 0; i < n_atoms; ++i)
     {
     fprintf(f_output, "%g %g %g\n", atom_coords[i][0],
      atom_coords[i][1], atom_coords[i][2]);

     }
   /* Close config file. */
   fclose(f_output);
}

