/* Routine to check for overlap of two spherocylinders. */

#include "build.h"

#define SMALL 1.0e-12

/* overlap checks for overlap between two spherocylinders. If skin > 0.0, 
"near" overlaps are also detected (this feature is used in constructing 
the neighbor lists).

input: scaled position of first molecule (s1)
       director of first molecule (u1)
       scaled position of second molecule (s2)
       director of second molecule (u2)
       spherocylinder length (length)
       neighbor list skin, if applicable (skin)
       array of linear dimensions of orthorhombic unit cell (h)

output: flag indicating whether molecules are overlapping;
           1 if they are, 0 if not (return value) */

int overlap(double *s1, double *u1, double *s2, double *u2,
            int sphero1, int sphero2, double *length, double skin, double **h)
{
   int i;
   double half_length1, half_length2, dr2, ds[3], dr[3], dr_dot_u1, dr_dot_u2, u1_dot_u2, 
      denom, lambda, mu, lambda_mag, mu_mag, r_min1, r_min[3], r_min2, lambda1, 
      lambda2, mu1, mu2, r_min3, lambda_mag1, mu_mag1;

   /* Compute various constants. */
   half_length1 = 0.5 * length[sphero1];
   half_length2 = 0.5 * length[sphero2];

/* printf(" length1 %g length2 %g\n", length[sphero1], length[sphero2]); */
   /* Compute pair separation. */
   dr2 = 0.0;
   for (i = 0; i < 3; ++i) {
      ds[i] = s1[i] - s2[i];
      ds[i] -= NINT(ds[i]);
      dr[i] = h[i][i] * ds[i];
      dr2 += SQR(dr[i]);
   }

   /* If the pair separation exceeds the spherocylinder length plus diameter plus
      the skin thickness, the spherocylinders don't overlap. */
   if (dr2 > SQR(half_length1 + half_length2 + 1.0 + skin)){
      return 0;
   }

   /* Test for overlaps (see Allen et al., Adv. Chem. Phys. 86, 1 (1993)). */
   dr_dot_u1 = dr[0] * u1[0] + dr[1] * u1[1] + dr[2] * u1[2];
   dr_dot_u2 = dr[0] * u2[0] + dr[1] * u2[1] + dr[2] * u2[2];
   u1_dot_u2 = u1[0] * u2[0] + u1[1] * u2[1] + u1[2] * u2[2];
   denom = 1.0 - SQR(u1_dot_u2);
   if (denom < SMALL) {
      lambda = -dr_dot_u1 / 2.0;
      mu =  dr_dot_u2 / 2.0;
   }
   else {
      lambda = (-dr_dot_u1 + u1_dot_u2 * dr_dot_u2) / denom;
      mu = ( dr_dot_u2 - u1_dot_u2 * dr_dot_u1) / denom;
   }
   lambda_mag = ABS(lambda);
   mu_mag = ABS(mu);

/* Chek if lambda > L_1/2 or mu > L_2/2  */
 if (lambda_mag > half_length1 || mu_mag > half_length2){
   if (lambda_mag > half_length1) { 
/* Set lambda = L_1/2 and find the value of mu and check for overlaps */

         lambda1 = SIGN(half_length1, lambda);
         mu1 =  dr_dot_u2 + lambda1 * u1_dot_u2;
         mu_mag1 = ABS(mu1);

/* If mu > L_2/2 dont check for overlaps */

         if (mu_mag1 > half_length2)      
         mu1 = SIGN(half_length2, mu1);  


   /* Calculate minimum separation of two spherocylinders. */
   r_min1 = 0.0;
   for (i = 0; i < 3; ++i) {
      r_min[i] = dr[i] + lambda1 * u1[i] - mu1 * u2[i];
      r_min1 += SQR(r_min[i]);
   }

   /* Return 1 if spherocylinders overlap, 0 if not. */
if (r_min1 < SQR(1.0 + skin))
   return(1);

}
  if (mu_mag > half_length2) {

/* set mu = L_2/2 and find the value of lambda and check for overlaps */

         mu2 = SIGN(half_length2, mu);
         lambda2 = -dr_dot_u1 + mu2 * u1_dot_u2;
         lambda_mag1 = ABS(lambda2);

/* If lambda > L_1/2 dont check for overlaps */

        if ( lambda_mag1 > half_length1) 
         lambda2 = SIGN(half_length1, lambda2);  


   /* Calculate minimum separation of two spherocylinders. */
   r_min2 = 0.0;
   for (i = 0; i < 3; ++i) {
      r_min[i] = dr[i] + lambda2 * u1[i] - mu2 * u2[i];
      r_min2 += SQR(r_min[i]);
   }

   /* Return 1 if spherocylinders overlap, 0 if not. */
 if (r_min2 < SQR(1.0 + skin))
  return(1);

 }
} else{

   /* Otherwise calculate minimum separation of two spherocylinders. */
   r_min3 = 0.0;
   for (i = 0; i < 3; ++i) {
      r_min[i] = dr[i] + lambda * u1[i] - mu * u2[i];
      r_min3 += SQR(r_min[i]);
   }
   /* Return 1 if spherocylinders overlap, 0 if not. */
if (r_min3 < SQR(1.0 + skin))
   return(1);
}
return(0);

}

#undef SMALL
