/* Input/output routines for template module. */

#include "build.h"
#include "ff.h"

/****************************************************************************/
/* Read molecular structure from structure file. */
/*
input : 
    array of files containing molecular structure of each specie (struct_file)

output :
    array of number of atoms per molecule for each specie (n_atoms_per_mol)
    array of template atom labels (temp_atm_lab)
    array of template atom type (temp_atm_type)
    array of template atom position (temp_atm_pos)
    array of template atom charge (temp_atm_chg)
    array of template atom numbre of branching (temp_atm_nbr)
    array of template atom label of branched atoms (temp_atm_br)
    array of template atom label order (temp_atm_ord)
    array of the number of bonds per molecule for each specie (n_bonds_per_mol)
    arrays of template bond - the labels of the 2 atoms sharing a bond
                                                  (temp_bonds_1, temp_bonds_2) 
    array of template bond - the order of a template bond (temp_bonds_ord)
*/

void read_str(char *struct_file, int *n_atoms_per_mol, int **temp_atm_lab,
   char ***temp_atm_type, double ***temp_atm_pos, double **temp_atm_chg,
   int **temp_atm_nbr, int ***temp_atm_br, double ***temp_atm_ord,
   int *n_bonds_per_mol, int **temp_bonds_1, int **temp_bonds_2,
   double **temp_bonds_ord)
{
   FILE *f_input;
   char line[MAX_LINE];
   int i, atom_1, atom_2, order;

   /* Open structure file. */
   if ((f_input = fopen(struct_file, "r")) == NULL)
      error_exit("Unable to open structure file in read_str");

   /* Skip comments in header of structure file. */
   do {
      if (fgets(line, MAX_LINE, f_input) == NULL)
         error_exit("Molecular structure data missing in read_str");
   } while (line[0] == '#');

   /* Read in number of atoms per molecule. */
   sscanf(line, "%d", n_atoms_per_mol);

   /* Allocate memory for array of template atom attributes. */
   *temp_atm_lab = allocate_1d_array(*n_atoms_per_mol, sizeof(int));
   *temp_atm_type = allocate_2d_array(*n_atoms_per_mol, TYPE_MAX, sizeof(char));
   *temp_atm_pos = allocate_2d_array(*n_atoms_per_mol, NDIM, sizeof(double));
   *temp_atm_chg = allocate_1d_array(*n_atoms_per_mol, sizeof(double));
   *temp_atm_nbr = allocate_1d_array(*n_atoms_per_mol, sizeof(int));
   *temp_atm_br = allocate_2d_array(*n_atoms_per_mol, 6, sizeof(int));
   *temp_atm_ord = allocate_2d_array(*n_atoms_per_mol, 6, sizeof(double));

   for (i = 0; i < *n_atoms_per_mol; ++i)
      {
      fscanf(f_input, "%d %s %lf %lf %lf %lf",
         &(*temp_atm_lab)[i], (*temp_atm_type)[i], &(*temp_atm_pos)[i][0],
         &(*temp_atm_pos)[i][1], &(*temp_atm_pos)[i][2], &(*temp_atm_chg)[i]);
      }
   /* Read in number of bonds per molecule. */
   fscanf(f_input, "%d", n_bonds_per_mol);

   /* Allocate memory for array of template bond attributes. */
   if (*n_bonds_per_mol > 0) {
   *temp_bonds_1 = allocate_1d_array(*n_bonds_per_mol, sizeof(int));
   *temp_bonds_2 = allocate_1d_array(*n_bonds_per_mol, sizeof(int));
   *temp_bonds_ord = allocate_1d_array(*n_bonds_per_mol, sizeof(double));
   }

   /* Read in template bond attributes. */
   for (i = 0; i < *n_bonds_per_mol; ++i)
      {
      fscanf(f_input, "%d %d %lf",
         &(*temp_bonds_1)[i], &(*temp_bonds_2)[i], &(*temp_bonds_ord)[i]);
      }

   /* Close structure file. */
   fclose(f_input);

   /* Add connectivity information to template atom array. */
   for (i = 0; i < *n_atoms_per_mol; ++i)
      (*temp_atm_nbr)[i] = 0;
   for (i = 0; i < *n_bonds_per_mol; ++i) {
      atom_1 = (*temp_bonds_1)[i];
      atom_2 = (*temp_bonds_2)[i];
      order = (*temp_bonds_ord)[i];
      (*temp_atm_br)[atom_1][(*temp_atm_nbr)[atom_1]] = atom_2;
      (*temp_atm_br)[atom_2][(*temp_atm_nbr)[atom_2]] = atom_1;
      (*temp_atm_ord)[atom_1][(*temp_atm_nbr)[atom_1]] = order;
      (*temp_atm_ord)[atom_2][(*temp_atm_nbr)[atom_2]] = order;
      ++(*temp_atm_nbr)[atom_1];
      ++(*temp_atm_nbr)[atom_2];
   }
}

/****************************************************************************/
/* Read mass parameters from specified data file. */
/*input :  
      force field file name (mass_file)

output : 
      the FF mass structure (mass)
      the number of entries in FF mass structure (n_mass)
*/

void read_mass_params(char *mass_file, mass_entry **mass, int *n_mass)
{
   FILE *f_input;
   char line[MAX_LINE];
   mass_entry tmp_mass;

   /* Open parameter file. */
   if ((f_input = fopen(mass_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_mass_params");

   /* Read in mass entries. */
   while (fgets(line, MAX_LINE, f_input) != NULL) {
      sscanf(line, "%s %lf %lf",
         tmp_mass.type, &tmp_mass.true_mass, &tmp_mass.eff_mass);
      if (((*mass) = realloc((*mass), 
        ((*n_mass) + 1) * sizeof(mass_entry))) == NULL)
         error_exit("Out of memory in read_mass_params");
      (*mass)[(*n_mass)] = tmp_mass;
      ++(*n_mass);
   }
 
   /* Close input file. */
   fclose(f_input);
}

/****************************************************************************/
/* Read and Set up potential parameters for nonbonded interactions. */
/*
input :
     Combination rule switch between atomic sites (vdw_rad_switch)
     Number of molecular species (n_species)
     Array of number of atomic sites per template molecule (n_atoms_per_mol)
     Number of LJ parameters (n_lj)
     Array of LJ parameters (lj)
     Array of template atom - type of atomic sites (temp_atm_type)
     Array of template atom - sigma LJ parameter (temp_atm_sigma) 
     Array of template atom - epsilon LJ parameter (temp_atm_eps)
     Array of template atom - charge of atomic sites (temp_atm_chg) 
output :
     Array of sigma combination between 2 atomic sites (comb_sigma)
     Array of sigma2 combination between 2 atomic sites (comb_sigma2)
     Array of r_min2 combination between 2 atomic sites (comb_r_min2)
     Array of epsilon combination between 2 atomic sites (comb_eps)
     Array of 4*epsilon combination between 2 atomic sites (comb_four_eps)
     Array of pair interaction  combination between 2 atomic sites (comb_pot)
     Array of charge combination between 2 atomic sites (comb_chg)
*/

void read_set_up_pot(char *pot_file, int vdw_rad_switch,  
     type_entry *atm_type, int n_type, int ***comb_pot, double ****comb_par)
{
   FILE *f_input;
   char line[MAX_LINE], **token = NULL;
   char *type_1, *type_2, *type_i, *type_j;
   int i_type, j_type, id_0, id_1, identifier[2], test_1, test_2, 
      found, same, pot_form, pot_form_1, pot_form_2, n_tokens, **found_entry;
   double cube_root_of_two, max_min;

   /* Allocate memory for combination array. */
   *comb_pot = allocate_2d_array(n_type, n_type, sizeof(int));
   *comb_par = allocate_3d_array(n_type, n_type, MAX_PAR, sizeof(double));

   /* Allocate memory for temporary array. */
   found_entry = allocate_2d_array(n_type, n_type, sizeof(int)); 

   /* Look only for pair interactions. */
   /* Initialisation. */
   for (i_type = 0; i_type < n_type; ++i_type)
     for (j_type = 0; j_type < n_type; ++j_type)
       found_entry[i_type][j_type] = 0;

   /* Open potential file. */
   if ((f_input = fopen(pot_file, "r")) == NULL)
      error_exit("Unable to open potential file in combination_array2");

   /* Read and parse pot_file line. */ 
   while (fgets(line, MAX_LINE, f_input) != NULL) {
     if (strncmp(line, "#", 1) != 0) {
       n_tokens = parse_tokens(line, &token);
       pot_form = atoi(token[0]); 
       type_i = token[1];
       type_j = token[2];

       /* Check if the atoms are of the same type. */ 
       same = 0;  
       if (strcmp(type_i, type_j) == 0) 
         same = 1;
        
       /* Search for the integer identifiers corresponding 
          to the atom types. */
       if (same) {
         found = 0;
         for (i_type = 0; i_type < n_type; ++i_type) {
            type_1 = atm_type[i_type].type;
            test_1 = strcmp(type_i, type_1) == 0;
            if (test_1) {
              found = 1;
              break;
            }
         }
                  
         if (found) {
           /* Get integer identifier. */
           id_0 = atm_type[i_type].id;
           /* Assign potential form. */
           (*comb_pot)[id_0][id_0] = pot_form;
           /* Update found entry counter. */
           found_entry[id_0][id_0] = 1;          
           /* Select the form of the potential between the 2 atom types. */
           select_pot(id_0, id_0, pot_form, token, n_tokens, vdw_rad_switch,
             0, *comb_par);
         }
       }
       else {
         found = 0;
         for (i_type = 0; i_type < n_type; ++i_type) {
            type_1 = atm_type[i_type].type;
            test_1 = strcmp(type_i, type_1) == 0;
            test_2 = strcmp(type_j, type_1) == 0;
            if (test_1 || test_2) {
              identifier[found] = atm_type[i_type].id;
              ++found;
              if (found == 2) break;
            }
         }
                  
         if (found == 2) {
           /* Assign potential form. */
           id_0 = identifier[0];
           id_1 = identifier[1];
           (*comb_pot)[id_0][id_1] = pot_form;
           (*comb_pot)[id_1][id_0] = pot_form;
           /* Update found entry counter. */
           found_entry[id_0][id_1] = 1;
           found_entry[id_1][id_0] = 1;
           /* Select the form of the potential between the 2 atom types. */
           select_pot(id_0, id_1, pot_form, token, n_tokens, vdw_rad_switch,
             0, *comb_par);
        }
       }
     }
   } /* end while */

   /* Apply combination rules between atoms of unlike type. */
   /* We are looking for the off-diagonal combinations which were 
   not explicitely defined in pot_file. */

   for (i_type = 0; i_type < n_type - 1; ++i_type)
     for (j_type = i_type + 1; j_type < n_type; ++j_type) {
       /* Check if off-diagonal combination have not been found yet. */
       if (found_entry[i_type][j_type] == 0) {

         /* Check if combination rules may apply. */
         pot_form_1 = (*comb_pot)[i_type][i_type];
         pot_form_2 = (*comb_pot)[j_type][j_type];
         if (pot_form_1 == pot_form_2) {
           (*comb_pot)[i_type][j_type] = pot_form_1;
           (*comb_pot)[j_type][i_type] = pot_form_1;
           /* Update found_entry counter. */
           found_entry[i_type][j_type] = 1;
           found_entry[j_type][i_type] = 1;
           /* Select the form of the potential between the 2 atom types. */
           select_pot(i_type, j_type, pot_form_1, token, n_tokens, 
             vdw_rad_switch, 1, *comb_par);
         }
         else {
           printf("i_type = %d, pot_form = %d\n", i_type, pot_form_1); 
           printf("j_type = %d, pot_form = %d\n", j_type, pot_form_2);
           error_exit("No possible combination rule. Missing entry\n");
         }
         
       }
     }
   /* Close potential file. */
   fclose(f_input); 
}
/*****************************************************************************/
