/* Graphics routines for molecular modeling package. */

#include "build.h"

/* Initialize graphics. */
void initialize_graphics(double **h)
{
   char *remote_host;
   int fovy, twist, i, c_map_start = 128, xwinsize = 800;
   long e_code, w_x0, w_y0, w_dx, w_dy, znear, zfar;
   double aspect, near, far, vx, vy, vz, px, py, pz;
   int graph_switch = 1;

   /* Open a remote host connection if appropriate. */
   remote_host = getenv("REMOTEHOST");
   if (remote_host == NULL) {
      remote_host = getenv("DISPLAY");
      e_code = dglopen(remote_host, DGLLOCAL);
      if (e_code < 0)
         error_exit("Cannot open DGL connection in initialize_graphics");
   }
   else {
      e_code = dglopen(remote_host, DGLTSOCKET);
      if (e_code < 0)
         error_exit("Cannot open DGL connection in initialize_graphics");
   }

   /* Open graphics window. */
   foreground();
   w_x0 = 50;
   w_y0 = 50;
   if (h[1][1] < h[0][0]) {
      w_dx = xwinsize;
      w_dy = (int) (xwinsize * h[1][1] / h[0][0]);
   }
   else {
      w_dy = xwinsize;
      w_dx = (int) (xwinsize * h[0][0] / h[1][1]);
   }
   prefposition(w_x0, w_x0 + w_dx, w_y0, w_y0 + w_dy);
   e_code = winopen("Hybrid MC");
   if (e_code == -1)
      error_exit("Cannot open graphics window in initialize_graphics");
   doublebuffer();
   gconfig();
   reshapeviewport();
   fovy = 600;
   aspect = 1.0;
   near = 1.5 * h[2][2] / 2.0;
   far = 3.0 * 1.5 * h[2][2] / 2.0;
   perspective(fovy, aspect, near, far);
   vx = 0.0;
   vy = 0.0;
   vz = 1.5 * h[2][2];
   px = 0.0;
   py = 0.0;
   pz = 0.0;
   twist = 0;
   lookat(vx, vy, vz, px, py, pz, twist);

   /* If params.graph_switch == 1, load the color map with a rainbow ramp.
      Otherwise, use a grey scale. */
   if (graph_switch == 1)
      for (i = 0; i < 16; ++i) {
         mapcolor(c_map_start + i, 16 * i, 0, 255);
         mapcolor(c_map_start + 16 + i, 255, 0, 255 - 16 * i);
         mapcolor(c_map_start + 32 + i, 255, 16 * i, 0);
         mapcolor(c_map_start + 48 + i, 255 - 16 * i, 255, 0);
         mapcolor(c_map_start + 64 + i, 0, 255, 16 * i);
         mapcolor(c_map_start + 80 + i, 0, 255 - 16 * i, 255);
         mapcolor(c_map_start + 96 + i, 16 * i, 0, 255);
         mapcolor(c_map_start + 112 + i, 255, 0, 255 - 16 * i);
      }
   else
      for (i = 0; i < 256; ++i)
         mapcolor(c_map_start + i, 255 - i, 255 - i, 255 - i);

   /* Set up mapping of z values to color map indices. */
   glcompat(GLC_FORCECIMAP, 1);
   glcompat(GLC_ZRANGEMAP, 0);
   znear = getgdesc(GD_ZMIN);
   zfar = getgdesc(GD_ZMAX);
   /* The following line should be included for new hardware, but omitted
      for old hardware. */
   lsetdepth(znear, zfar);
   if (graph_switch == 1)
      lshaderange(c_map_start, c_map_start + 127, znear, zfar);
   else
      lshaderange(c_map_start, c_map_start + 255, znear, zfar);

   /* Turn on depthcue mode. */
   depthcue(TRUE);

   /* Set linewidth and anti-aliasing. */
   linewidth(2);
/*   linesmooth(SML_ON);*/
}

/* Plot molecules. */
void plot_molecules(int n_mols, int *mol_species, int *mol_first_atm,
    int *n_atoms_per_mol, int *n_bonds_per_mol, double **atom_coords,
    double **h, int **temp_bonds_1, int **temp_bonds_2)
{
   long m_x, m_y;
   int i_species, i_bond, i_mol, skip, rel_1, rel_2, abs_1, abs_2,
      v_skip, v_rel, v_abs, k;
   double x_bond, y_bond, z_bond, vec[3];
   static int rot_x = 900, rot_y = 0;
/*   static float factor = 1.0; */
   static float factor = 0.7;
   int graph_switch = 1;

   /* Calculate bond vectors if necessary. */
/*   if (vars.bond_flag) {
      if (params.period_switch)
         calculate_bond_vectors_period();
      else
         calculate_bond_vectors_free();
   }
*/
   /* Clear window, poll mouse, etc. */
   pushmatrix();
   if (graph_switch == 1)
      color(0);
   else
      color(7);
   clear();
   if (getbutton(MOUSE1)) {
      m_x = getvaluator(MOUSEX);
      m_y = getvaluator(MOUSEY);
      rot_x = 4 * m_x;
      rot_y = 4 * m_y;
   }
   if (getbutton(MOUSE3) && getbutton(MOUSE2)) {
      factor = 1.0;
      rot_x = 0;
      rot_y = 0;
   }
   else if (getbutton(MOUSE3))
      factor += 0.05;
   else if (getbutton(MOUSE2))
      factor -= 0.05;
   scale(factor, factor, factor);
   rotate(rot_x, 'x');
   rotate(rot_y, 'y');

   /* Draw molecular backbones. */
      
   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom and bond vector label offsets. */
      skip = mol_first_atm[i_mol];
/*      v_skip = mol_props[i_mol].first_vector;  */

      /* Loop over bonds. */
      for (i_bond = 0; i_bond < n_bonds_per_mol[i_species]; ++i_bond) {

         /* Get relative labels of atoms that share bond. */
         rel_1 = temp_bonds_1[i_species][i_bond];
         rel_2 = temp_bonds_2[i_species][i_bond];

         /* Get relative label of bond vectors. */
    /*     v_rel = vector_index[i_species][rel_1][rel_2]; */

         /* Compute absolute labels of atoms that share bond. */
         abs_1 = skip + rel_1;
         abs_2 = skip + rel_2;

         /* Compute absolute labels of bond vector. */
   /*      v_abs = v_skip + v_rel;  */

         /* Get bond vector. */
   /*    x_bond = bond_vectors[v_abs].x;
         y_bond = bond_vectors[v_abs].y;
         z_bond = bond_vectors[v_abs].z;
*/
    /*     bgnline();
         vec[0] = atom_coords[abs_1].x;
         vec[1] = atom_coords[abs_1].y;
         vec[2] = atom_coords[abs_1].z;
         v3d(vec);
         vec[0] = atom_coords[abs_1].x + x_bond;
         vec[1] = atom_coords[abs_1].y + y_bond;
         vec[2] = atom_coords[abs_1].z + z_bond;
         v3d(vec);
         endline();

         bgnline();
         vec[0] = atom_coords[abs_2].x;
         vec[1] = atom_coords[abs_2].y;
         vec[2] = atom_coords[abs_2].z;
         v3d(vec);
         vec[0] = atom_coords[abs_2].x - x_bond;
         vec[1] = atom_coords[abs_2].y - y_bond;
         vec[2] = atom_coords[abs_2].z - z_bond;
         v3d(vec);
         endline();
*/
	 bgnline();
         for (k = 0; k < NDIM; ++k)
           vec[k] = atom_coords[abs_1][k];
         v3d(vec);
         for (k = 0; k < NDIM; ++k)
           vec[k] = atom_coords[abs_2][k];
         v3d(vec);
         endline();

      }
   }

   /* Draw bounding box. */
   bgnline();
   vec[0] = 0.5 * (- h[0][0] - h[0][1] - h[0][2]);
   vec[1] = 0.5 * (- h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (h[0][0] - h[0][1] - h[0][2]);
   vec[1] = 0.5 * (- h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (h[0][0] + h[0][1] - h[0][2]);
   vec[1] = 0.5 * (h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (- h[0][0] + h[0][1] - h[0][2]);
   vec[1] = 0.5 * (h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (- h[0][0] - h[0][1] - h[0][2]);
   vec[1] = 0.5 * (- h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (- h[0][0] - h[0][1] + h[0][2]);
   vec[1] = 0.5 * (- h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (h[0][0] - h[0][1] + h[0][2]);
   vec[1] = 0.5 * (- h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (h[0][0] - h[0][1] - h[0][2]);
   vec[1] = 0.5 * (- h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   endline();
   bgnline();
   vec[0] = 0.5 * (- h[0][0] + h[0][1] - h[0][2]);
   vec[1] = 0.5 * (h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (- h[0][0] + h[0][1] + h[0][2]);
   vec[1] = 0.5 * (h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (h[0][0] + h[0][1] + h[0][2]);
   vec[1] = 0.5 * (h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (h[0][0] + h[0][1] - h[0][2]);
   vec[1] = 0.5 * (h[1][1] - h[1][2]);
   vec[2] = - 0.5 * h[2][2];
   v3d(vec);
   endline();
   bgnline();
   vec[0] = 0.5 * (- h[0][0] - h[0][1] + h[0][2]);
   vec[1] = 0.5 * (- h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (- h[0][0] + h[0][1] + h[0][2]);
   vec[1] = 0.5 * (h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   endline();
   bgnline();
   vec[0] = 0.5 * (h[0][0] - h[0][1] + h[0][2]);
   vec[1] = 0.5 * (- h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   vec[0] = 0.5 * (h[0][0] + h[0][1] + h[0][2]);
   vec[1] = 0.5 * (h[1][1] + h[1][2]);
   vec[2] = 0.5 * h[2][2];
   v3d(vec);
   endline();

   /* Swap buffers. */
   popmatrix();
   swapbuffers();
}
