/* Periodic boundary condition nonbonded force routines for molecular
   modeling package, with LJ form for vdW interactions. */

#include "build.h"

/* Calculate short-range LJ interactions using neighbor lists, and using a
   switching function to truncate the pair potential and pair force smoothly. */
void lj_s_nl_period_phantom( int *mol_species, int *atom_rel, int *atom_mol, 
   int n_atoms,
   int *n_atoms_per_mol, int ***exclusions, double ****comb_sigma2,
   double ****comb_four_eps, double **h, double r2_on, double r2_off,
   double **scaled_atom_coords, double gamma, double three_gamma_over_two,
   double two_over_gamma_cubed, double *pe_vdw_s,
   int period_switch, int *first, int *last, phantom_cell *phantoms,
   int noffsets, int *nc_p, int kc, int *phantom_skip, int *nc,
   atom_cell *atom_cells, int *offsets, 
   double **atom_coords, nl_entry **nl_head, nl_entry **nl_tail, double r2_nl,
   double **rel_atom_coords,
   double **at_stress_vdw_s, double **mol_stress_vdw_s, double **vdw_s, 
   double **atom_move)
{
   int i, j, i_species, j_species, i_mol, j_mol, i_rel, j_rel, k, j_real;
   double s_sep[NDIM], sep[NDIM], s_i[NDIM], 
      r2_sep, one_over_r2_sep,
      rho_2, rho_6, rho_12, four_epsilon,
      u_vdw, sw1, sw2, sw3, sw4;
   double p_norm, volume, rel_dist[NDIM], f_i_vdw[NDIM], fvdw, f_vdw[NDIM];
   nl_entry *p, *p_term;

    /* Initialisation. */
   *pe_vdw_s = 0.0;

   /* Zero short-range vdW forces. */
    for (i = 0; i < n_atoms; ++i)
       for ( k = 0; k < NDIM; ++k)
          vdw_s[i][k] = 0.0;

   for (i = 0; i < NDIM; ++i)
      for (k = 0; k < NDIM; ++k)
         mol_stress_vdw_s[i][k] = 0.0;

   /* Loop over atoms. */
   for (i = 0; i < n_atoms; ++i) {
      
      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k) {
        s_i[k] = scaled_atom_coords[i][k];
/*        f_i_vdw[k] = vdw_s[i][k];  */
        f_i_vdw[k] = 0.0;
      }

      i_mol = atom_mol[i];
      i_rel = atom_rel[i];
      i_species = mol_species[i_mol];

      /* loop over entries in neighbor list for atom i. */
      if (nl_tail[i] != NULL) {
         p = nl_head[i];
         p_term = nl_tail[i] -> next;
         while (p != p_term) {

            /* Get atomic label. */
            j = p -> atom;

            /* Calculate pair separation. */
            for (k = 0; k < NDIM; ++k) {
                 s_sep[k] = scaled_atom_coords[j][k] - s_i[k];
                 s_sep[k] -= NINT(s_sep[k]);
                 }
            sep[0] = h[0][0] * s_sep[0] + h[0][1] * s_sep[1]
                   + h[0][2] * s_sep[2];
            sep[1] = h[1][1] * s_sep[1] + h[1][2] * s_sep[2];
            sep[2] = h[2][2] * s_sep[2];

            r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

            /* If the pair separation is less than r_off, calculate vdW
               interaction. */
            if (r2_sep < r2_off) {

               /* Calculate pair interaction. The vdW pair potential
                  is switched off smoothly between params.r_on
                  and params.r_off. */
               j_mol = atom_mol[j];
               j_rel = atom_rel[j];
               j_species = mol_species[j_mol];
               one_over_r2_sep = 1.0 / r2_sep;
               rho_2 = comb_sigma2[i_species][i_rel][j_species][j_rel]
                          * one_over_r2_sep;
               rho_6 = CUBE(rho_2);
               rho_12 = SQR(rho_6);
               u_vdw = rho_12 - rho_6;
               fvdw = rho_12 + u_vdw;
               four_epsilon =  
                         comb_four_eps[i_species][i_rel][j_species][j_rel];
               u_vdw *= four_epsilon;
               fvdw *= 6.0 * one_over_r2_sep * four_epsilon;
               if (r2_sep > r2_on) {
                  sw1 = r2_off - r2_sep;
                  sw2 = two_over_gamma_cubed * sw1;
                  sw3 = sw1 * sw2 * (three_gamma_over_two - sw1);
                  sw4 = 6.0 * sw2 * (gamma - sw1);
                  fvdw = sw4 * u_vdw + sw3 * fvdw;
                  u_vdw *= sw3;
               }
               /* Add contributions to force. */
                  for ( k = 0; k<NDIM; ++k){
                  f_vdw[k] = sep[k] * fvdw;
                  f_i_vdw[k] -= f_vdw[k];
            /*      vdw_s[j][k] +=f_vdw[k];   */
                  }

               /* Add contributions to potential energy and atomic and
                  molecular pressure tensors. */
                  *pe_vdw_s += u_vdw;
                  mol_stress_vdw_s[0][0] += sep[0] * f_vdw[0];
                  mol_stress_vdw_s[1][1] += sep[1] * f_vdw[1];
                  mol_stress_vdw_s[2][2] += sep[2] * f_vdw[2];
                  mol_stress_vdw_s[0][1] += sep[0] * f_vdw[1];
                  mol_stress_vdw_s[0][2] += sep[0] * f_vdw[2];
                  mol_stress_vdw_s[1][2] += sep[1] * f_vdw[2];
               /* Add contribution coming from the intramolecular rigidity 
                  constraint */
                    for ( k =0; k<NDIM; ++k)
                       rel_dist[k] = (rel_atom_coords[j][k] - 
                                                  rel_atom_coords[i][k]);

                     mol_stress_vdw_s[0][0] -= rel_dist[0] * f_vdw[0];
                     mol_stress_vdw_s[1][1] -= rel_dist[1] * f_vdw[1];
                     mol_stress_vdw_s[2][2] -= rel_dist[2] * f_vdw[2];
                     mol_stress_vdw_s[0][1] -= rel_dist[0] * f_vdw[1];
                     mol_stress_vdw_s[0][2] -= rel_dist[0] * f_vdw[2];
                     mol_stress_vdw_s[1][2] -= rel_dist[1] * f_vdw[2];
            }
            p = p -> next;
         }
      }
     for ( k = 0; k < NDIM; ++k)
      vdw_s[i][k] = f_i_vdw[k];
   }

/* The contributions to the potential and to the stress tensor are half
   the previous quantities (we have computed i on j and j on i). */

*pe_vdw_s /= 2.0;

for (i = 0; i < NDIM; ++i)
  for (j = i; j < NDIM; ++j)
    mol_stress_vdw_s[i][j] /= 2.0;
/*
printf("--------------------------------------\n");
fflush(NULL);
*/
  /* normalize contributions to potential
     energy and atomic stress tensor, and calculate contributions
     to molecular stress tensor. */
/*      p_norm = 1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_vdw_s[i][j] *= p_norm;
      for (i = 0; i < 3; ++i)
      for (j = 0; j < 3; ++j)
         mol_stress_vdw_s[i][j] *= p_norm;
      for (i = 0; i < 3; ++i)
           for (j = i; j < 3; ++j)
              mol_stress_vdw_s[i][j] += at_stress_vdw_s[i][j];
        for (i = 1; i < 3; ++i)
           for (j = 0; j < i; ++j)
              mol_stress_vdw_s[i][j] += at_stress_vdw_s[j][i];
 */ 
/* for (i = 0; i < 3; ++i)
      for (j = 0; j < 3; ++j)
printf("mol_stress %g\n", mol_stress_vdw_s[i][j]);
*/
}
/****************************************************************************/
/* Calculate short-range LJ interactions for single molecules using neighbor 
   lists, and using a switching function to truncate the pair potential and 
   pair force smoothly. */

void lj_s_nl_period_phantom_single( int skip, int mol,
   int *mol_species, int *atom_rel, int *atom_mol, int n_atoms,
   int *n_atoms_per_mol, int ***exclusions, double ****comb_sigma2,
   double ****comb_four_eps, double **h, double r2_on, double r2_off,
   double **scaled_atom_coords, double gamma, double three_gamma_over_two,
   double two_over_gamma_cubed, double *pe_vdw_s_single,
   int period_switch, int *first, int *last, phantom_cell *phantoms,
   int noffsets, int *nc_p, int kc, int *phantom_skip, int *nc,
   atom_cell *atom_cells, int *offsets, 
   double **atom_coords, nl_entry **nl_head, nl_entry **nl_tail, double r2_nl,
   double **rel_atom_coords, int *mol_first_atm)
{
   int i, j, i_species, j_species, i_mol, j_mol, i_rel, j_rel, k, j_real;
   double s_sep[NDIM], sep[NDIM], s_i[NDIM], r2_sep, one_over_r2_sep,
      rho_2, rho_6, rho_12, four_epsilon, u_vdw, sw1, sw2, sw3, sw4;
   nl_entry *p, *p_term;

    /* Initialisation. */
   *pe_vdw_s_single = 0.0;

   /* Update neighbor lists if necessary. */
/*   neighbor_check_single( skip, mol, mol_species, atom_rel, atom_mol, n_atoms,
   n_atoms_per_mol, exclusions, h, r2_on, r2_off,
   scaled_atom_coords, period_switch, first, last, phantoms,
   noffsets, nc_p, kc, phantom_skip, nc,
   atom_cells, offsets, atom_coords,
   mol_first_atm, atom_move, nl_head, nl_tail, r2_nl);
*/

   /* Get molecular specie. */
   i_species = mol_species[mol];

   /* Loop over atoms in the molecule i_mol. */
   for (i = skip; i < skip + n_atoms_per_mol[i_species] ; ++i) {

      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k)
        s_i[k] = scaled_atom_coords[i][k];
      i_mol = atom_mol[i];
      i_rel = atom_rel[i];

      /* loop over entries in neighbor list for atom i. */
      if (nl_tail[i] != NULL) {
         p = nl_head[i];
         p_term = nl_tail[i] -> next;
         while (p != p_term) {

            /* Get atomic label. */
            j = p -> atom;

            /* Calculate pair separation. */
            for (k = 0; k < NDIM; ++k) {
                 s_sep[k] = scaled_atom_coords[j][k] - s_i[k];
                 s_sep[k] -= NINT(s_sep[k]);
                 }
               sep[0] = h[0][0] * s_sep[0] + h[0][1] * s_sep[1]
                        + h[0][2] * s_sep[2];
               sep[1] = h[1][1] * s_sep[1] + h[1][2] * s_sep[2];
               sep[2] = h[2][2] * s_sep[2];

                r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

            /* If the pair separation is less than r_off, calculate vdW
               interaction. */
            if (r2_sep < r2_off) {

               /* Calculate pair interaction. The vdW pair potential
                  is switched off smoothly between r_on
                  and r_off. */
               j_mol = atom_mol[j];
               j_rel = atom_rel[j];
               j_species = mol_species[j_mol];
               one_over_r2_sep = 1.0 / r2_sep;
               rho_2 = comb_sigma2[i_species][i_rel][j_species][j_rel]
                          * one_over_r2_sep;
               rho_6 = CUBE(rho_2);
               rho_12 = SQR(rho_6);
               u_vdw = rho_12 - rho_6;
               four_epsilon =  
                         comb_four_eps[i_species][i_rel][j_species][j_rel];
               u_vdw *= four_epsilon;
               if (r2_sep > r2_on) {
                  sw1 = r2_off - r2_sep;
                  sw2 = two_over_gamma_cubed * sw1;
                  sw3 = sw1 * sw2 * (three_gamma_over_two - sw1);
                  sw4 = 6.0 * sw2 * (gamma - sw1);
                  u_vdw *= sw3;
               }
               *pe_vdw_s_single += u_vdw;
            }
            p = p -> next;
         }
      }
   }
}
