/* Integration routines for molecular modeling package. */

#include "build.h"
#include "ff.h"
#include "proto.h"

/* Integrate the equations of motion using the velocity Verlet algorithm,
   which (in the absence of constraints) is reversible and area preserving
   and thus suitable for MD. */

void one_step(int *mol_species, int *atom_rel, int *atom_mol, int n_type,
   int *atom_type, int n_atoms, int *n_atoms_per_mol, int ***exclusions,
   int **comb_pot, double ***comb_par, double r2_on, double r2_off,
   double **h, double **scaled_atom_coords, double gamma,
   double three_gamma_over_two, double two_over_gamma_cubed,
   double *pe_vdw_s, int period_switch, int *first, int *last,
   phantom_cell *phantoms, int noffsets, int *nc, int *nc_p, 
   int *nc_p_total, int kc, int *phantom_skip,
   atom_cell *atom_cells, int *offsets, double ***corr,
   double **atom_coords,
   double **rel_atom_coords, double **at_stress_vdw_s,
   double **mol_stress_vdw_s, double **f_vdw_s, double **comp_forces, 
   double **atom_vels, double delta, double *atom_mass, double **atom_move,
   double **h_inv, int neigh_switch, double *pe_stretch, double **f_stretch,
   bond_prop **bond_props, int vdw_switch, int bend_switch,
   nl_entry **nl_head, nl_entry **nl_tail,
   double r2_nl, int *mol_first_atm, int *n_bonds_per_mol,
   double **at_stress_stretch, int eval_flag,int **temp_bonds_1, 
   int **temp_bonds_2, int n_mols, double r_nl, double r_off, 
   angle_prop **angle_props, double *pe_bend, double **f_bend, 
   double **at_stress_bend, int *n_angles_per_mol,
   int pore_switch, double r0, double x0, double y0, double z0, 
   double *pe_pore_s, double **f_pore_s, double **at_stress_pore, 
   double **mol_stress_pore, double *mol_mass,
   int **scan_atm_1, int **scan_atm_2, double **mol_coords, 
   double **scaled_mol_coords, int n_species, 

   double *pe_coul_s, int coul_switch, double alpha,
   double alpha2, double one_over_erf_tab_incr, double *erfc_sqrt_tab,
   double two_alpha_over_sqrt_pi, double ****prod_chg, int ****comb_chg,
   double **f_coul_s, double **at_stress_coul_s, double *atom_chg,
   double *b1, double *b2, double *b3,
   int m1_max, int m2_max, int m3_max, double g2_cutoff,
   double **cos_G_1_dot_r, double **sin_G_1_dot_r,
   double **cos_G_2_dot_r, double **sin_G_2_dot_r,
   double **cos_G_3_dot_r, double **sin_G_3_dot_r,
   double *q_cos_G_12_dot_r, double *q_sin_G_12_dot_r,
   double *q_cos_G_123_dot_r, double *q_sin_G_123_dot_r,
   double *pe_coul_recip, 
   double **f_coul_recip, int n_grid_x, int n_grid_y, int n_grid_z,
   double **gridded_atom_coords,
   int B_spline_order, double **M_x, double **M_y, double **M_z,
   double **dM_du_x, double **dM_du_y, double **dM_du_z,
   fft elec, double volume, double *b_1_un, double *b_2_un, double *b_3_un,
   double *b_mod2_x, double *b_mod2_y, double *b_mod2_z, 
   double *erf_sqrt_tab, double pe_coul_self, double **at_stress_coul_recip,
   double **mol_stress_coul_s, double **mol_stress_coul_recip, double *p_normal_vdw, 
   double *p_tangent_vdw, double *p_normal_coul,
   double *p_tangent_coul, double slab_width, int ift_switch, int n_slab,
   double *p_tangent_stretch, double *p_normal_stretch, double *p_tangent_bend,
   double *p_normal_bend)
{
  int i, thermo_flag, k, j;
  double p_norm;

/*memory("     enter integrate");*/

   /* Carry out half-timestep update of atomic velocities using old forces. */
   velocity_step(n_atoms, atom_vels, atom_mass, delta, comp_forces);

   /* Carry out full-timestep update of atomic positions using
      half-timestep velocities. */

   position_step(n_atoms, atom_coords, atom_vels, atom_move, 
                 scaled_atom_coords, h_inv, h, delta, neigh_switch);

   /* Compute center of mass of the molecules and relative atom positions
      within the molecules. */
  center_of_mass_positions(period_switch, n_mols, n_species,
      n_atoms_per_mol,  mol_species, atom_mass, mol_first_atm,
      corr, atom_coords, scaled_atom_coords,  scan_atm_1, scan_atm_2,
      mol_mass, h, h_inv, mol_coords, scaled_mol_coords,  rel_atom_coords);


   /* Zero force accumulators. */
   for (i = 0; i < n_atoms; ++i) {
       for (k = 0; k < NDIM; ++k)
        comp_forces[i][k] = 0.0;
   }

         wca_period_phantom(mol_species, atom_rel, atom_mol, atom_type, n_atoms,
          n_atoms_per_mol, exclusions, comb_pot, comb_par,
          r2_on, r2_off, h, scaled_atom_coords, gamma, three_gamma_over_two,
          two_over_gamma_cubed, pe_vdw_s, pe_coul_s, coul_switch,
          alpha, alpha2, one_over_erf_tab_incr, erfc_sqrt_tab,
          two_alpha_over_sqrt_pi, prod_chg, comb_chg, period_switch,
          first, last, phantoms, noffsets, nc_p, kc,
          phantom_skip, nc, atom_cells, offsets, atom_coords,
          rel_atom_coords, at_stress_vdw_s, mol_stress_vdw_s,
          f_vdw_s, f_coul_s, at_stress_coul_s, mol_stress_coul_s,
          p_normal_vdw, p_tangent_vdw, p_normal_coul, p_tangent_coul,
          slab_width, ift_switch, n_slab);

      /* Add short-range nonbonded forces to force accumulators. */
      for (i = 0; i < n_atoms; ++i) {
         for ( k = 0; k < NDIM; ++k)
         comp_forces[i][k] += f_vdw_s[i][k] + f_coul_s[i][k];
      }

   /* Calculate bond stretching interactions and add contributions to force
      /* Calculate reciprocal part of the non bonded electrostatic
         interactions (if coul_switch == 1, uses conventional Ewald,
         if coul_switch == 2, uses PME). */
  if (coul_switch > 0){

      if (coul_switch == 1)
         ewald_recip(n_atoms, volume, scaled_atom_coords, atom_chg, b1, b2, b3,
                  m1_max, m2_max, m3_max, alpha, g2_cutoff, cos_G_1_dot_r,
                  sin_G_1_dot_r, cos_G_2_dot_r, sin_G_2_dot_r, cos_G_3_dot_r,
                  sin_G_3_dot_r, q_cos_G_12_dot_r, q_sin_G_12_dot_r,
                  q_cos_G_123_dot_r, q_sin_G_123_dot_r, pe_coul_recip,
                  at_stress_coul_recip, f_coul_recip);
      else 
         pme_recip(n_atoms, n_grid_x, n_grid_y, n_grid_z, scaled_atom_coords,
                   gridded_atom_coords, B_spline_order, M_x, M_y, M_z,
                   dM_du_x, dM_du_y, dM_du_z, atom_chg, elec, volume,
                   alpha, b1, b2, b3, b_mod2_x, b_mod2_y, b_mod2_z,
                   pe_coul_recip, at_stress_coul_recip, f_coul_recip);

      /* Subtract contributions to long-range Coulomb interactions from
         nonbonded exclusions, which are implicitly contained in the
         reciprocal-space Ewald sum. */

      subtract_nonbonded_exclusions(n_mols, mol_species, mol_first_atm,
       n_atoms_per_mol, exclusions, h, volume, scaled_atom_coords, prod_chg, 
       comb_chg, alpha, alpha2, two_alpha_over_sqrt_pi, one_over_erf_tab_incr,
       erf_sqrt_tab, pe_coul_self, pe_coul_recip, at_stress_coul_recip, 
       f_coul_recip, p_normal_coul, p_tangent_coul, ift_switch, n_slab);

      /* Add long-range nonbonded Coulombic forces to force accumulators. */
      for (i = 0; i < n_atoms; ++i) {
         for ( k = 0; k < NDIM; ++k)
            comp_forces[i][k] += f_coul_recip[i][k];
      }

      /* Compute the reciprocal part of the molecular stress after removing 
         the non-bonded part of the coulomb interaction. */

      mol_stress_cal(n_atoms, rel_atom_coords, f_coul_recip,
                      at_stress_coul_recip, mol_stress_coul_recip, volume);
   }


   /* Calculate bond stretching interactions. */
       stretch_harmonic(period_switch, eval_flag, 
             f_stretch, mol_first_atm, n_bonds_per_mol,at_stress_stretch,
             pe_stretch, n_atoms, n_mols, mol_species,
             temp_bonds_1, temp_bonds_2, h, scaled_atom_coords, bond_props,
             ift_switch, n_slab, p_tangent_stretch, p_normal_stretch);

      /* Add bond stretching forces to force accumulators. */
      for (i = 0; i < n_atoms; ++i) {
         for ( k = 0; k < NDIM; ++k)
         comp_forces[i][k] += f_stretch[i][k];
      }

   /* Calculate bond angle bending interactions and add contributions to force
      accumulators. */

   if(bend_switch){
    bend_cos_harmonic(period_switch, eval_flag,
             f_bend, mol_first_atm, n_angles_per_mol,
             at_stress_bend, pe_bend, n_atoms, n_mols, mol_species,
             angle_props, h, scaled_atom_coords, ift_switch, n_slab,
             p_tangent_bend, p_normal_bend); 

    /* Add bond angle bending  forces to force accumulators. */
      for (i = 0; i < n_atoms; ++i) {
         for ( k = 0; k < NDIM; ++k)
         comp_forces[i][k] += f_bend[i][k]; 
      }
   }

   /* Calculate interactions with the pore walls. */
   if (pore_switch){
      force_pore(n_atoms, n_type, atom_type, comb_pot, comb_par, r2_on,
                 r2_off, gamma, three_gamma_over_two, two_over_gamma_cubed,
                 pe_pore_s, atom_coords, r0, x0, y0, z0, at_stress_pore,
                 mol_stress_pore, f_pore_s);

      for (i = 0; i < n_atoms; ++i) {
         for ( k = 0; k < NDIM; ++k)
         comp_forces[i][k] += f_pore_s[i][k];
      }
   }

   /* Carry out half-timestep update of atomic velocities using new forces. */
   velocity_step(n_atoms, atom_vels, atom_mass, delta, comp_forces);

/*memory("     exit integrate");*/
}

/* Integrate the equations of motion using a velocity
   Verlet algorithm */

void velocity_step(int n_atoms, double **atom_vels, 
                   double *atom_mass, double delta, double **comp_forces)
{
   int i, k;
   double delta_over_2m;

   /* Update velocities. */
   for (i = 0; i < n_atoms; ++i) {
      delta_over_2m = delta /(2*atom_mass[i]) ;
      for ( k =  0; k < NDIM; ++k)
      atom_vels[i][k] += delta_over_2m * comp_forces[i][k];
   }
}

/* Carry out full-timestep update of atomic positions using
   half-timestep velocities. */
void position_step(int n_atoms, double **atom_coords, 
                   double **atom_vels,
                   double **atom_move, double **scaled_atom_coords, 
                   double **h_inv, double **h, double delta, int neigh_switch)
{
   int i, k;
   double dr[NDIM];


   /* Update positions. */
   for (i = 0; i < n_atoms; ++i) {
       for (k = 0; k < NDIM;  ++k) {
      dr[k] = delta * atom_vels[i][k];
      atom_coords[i][k] += dr[k];

      /* If neigh_switch == 2, add atomic displacements to movement
         accumulators. */
      if (neigh_switch ==2) 
         atom_move[i][k] += dr[k];
      }
   }

   /* If we are using periodic boundary conditions, calculate scaled
      atomic coordinates. */
      scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);
      periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, atom_coords);

}

/* Scale velocities to maintain constant temperature, using the algorithm
   of Berendsen et al. (J. Chem. Phys. 81, 3684 (1984)). */
void nvt_berendsen(int n_atoms, double temperature, double **atom_vels,
                   double temp_scale, double at_temp, int n_mols, 
                   int *mol_species, double *mol_mass,
                   double **mol_vels, double boltzmann, int n_free_atom, 
                   int n_free_mol, double *atom_mass)
{
   int i, k;
   double scale_vel;

   /* Update atomic velocities, using scaling factor derived from
      atomic temperature. */

   scale_vel = 1.0 + temp_scale * (temperature / at_temp - 1.0);
   for (i = 0; i < n_atoms; ++i)
      for (k = 0; k < NDIM; k++)
         atom_vels[i][k] *= scale_vel;
}

/* Regulate pressure using the Berendsen manostat. */
void manostat(int n_mols, int n_atoms, int *mol_species, int *n_atoms_per_mol,
              double **mol_coords, double **atom_coords, 
              double **scaled_atom_coords,
              double **h, double **h_inv, double **at_stress, int scale_con,
              double r_off, double r2_nl, double r_nl, 
              double press, int neigh_switch, int phant_switch,
              int **first, int **last, phantom_cell **phantoms, 
              int *nc, int *nc_p, int *nc_p_total, int kc, int *phantom_skip,
              atom_cell *atom_cells, int *offsets, double press_scale, 
              int period_switch, int **scale, int *mol_first_atm,
              double **mol_stress)
{
   int i_species, i, j, k, i_mol, i_atom, skip, abs;
   double scale_pos[3][3], h_tmp[3][3], x_tmp, y_tmp, z_tmp;


   /* Calculate components of scaling tensor, using symmetrized stress
      tensor. */
   for (i = 0; i < 3; ++i) {
      scale_pos[i][i] = 1.0;
      if (scale[i][i])
         scale_pos[i][i] -= press_scale
            * (press - mol_stress[i][i]);
  
     for (j = i + 1; j < 3; ++j) {
         scale_pos[i][j] = 0.0;
         if (scale[i][j]) {
            scale_pos[i][j] += 0.5 * press_scale
               * (mol_stress[i][j] + mol_stress[j][i]);
         }
      }

   }

   if (scale_con == 1)
      scale_pos[0][0] = scale_pos[1][1]
         = 0.5 * (scale_pos[0][0] + scale_pos[1][1]);
   else if (scale_con == 2)
      scale_pos[0][0] = scale_pos[2][2]
         = 0.5 * (scale_pos[0][0] + scale_pos[2][2]);
   else if (scale_con == 3)
      scale_pos[1][1] = scale_pos[2][2]
         = 0.5 * (scale_pos[1][1] + scale_pos[2][2]);
   else if (scale_con == 4)  /* volume change in all direction */
      scale_pos[0][0] = scale_pos[1][1] = scale_pos[2][2]
         = (scale_pos[0][0] + scale_pos[1][1] + scale_pos[2][2]) / 3.0;
   else if (scale_con == 5) /* If a pore is present, use this option ONLY. */
      scale_pos[0][0] = scale_pos[1][1] =  1.0;

   /* Calculate new box matrix and box vectors. */
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j) {
         h_tmp[i][j] = 0.0;
         for (k = i; k <= j; ++k){
            h_tmp[i][j] += scale_pos[i][k] * h[k][j];
      }
      }

   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         h[i][j] = h_tmp[i][j];

   /* Calculate quantities that depend on box dimensions and exit if
      system is too small. */

    box_dimensions(h, h_inv, period_switch, 0.0, 0.0);

   /* Scale molecular center of mass coordinates. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {
      x_tmp = mol_coords[i_mol][0];
      y_tmp = mol_coords[i_mol][1];
      z_tmp = mol_coords[i_mol][2];

      mol_coords[i_mol][0] = scale_pos[0][0] * x_tmp
         + scale_pos[0][1] * y_tmp
         + scale_pos[0][2] * z_tmp;
      mol_coords[i_mol][1] = scale_pos[1][1] * y_tmp
         + scale_pos[1][2] * z_tmp;
      mol_coords[i_mol][2] = scale_pos[2][2] * z_tmp;

      x_tmp = mol_coords[i_mol][0] - x_tmp;
      y_tmp = mol_coords[i_mol][1] - y_tmp;
      z_tmp = mol_coords[i_mol][2] - z_tmp;

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Shift atomic positions. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {
         abs = skip + i_atom;
         atom_coords[abs][0] += x_tmp;
         atom_coords[abs][1] += y_tmp;
         atom_coords[abs][2] += z_tmp;

         }
      }

   /* Calculate scaled atomic coordinates. */
    scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);

   /* Re-initialize cell-search data structures if necessary. */
    set_up_cells_phantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
         period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
         n_atoms, scaled_atom_coords, atom_coords, phantoms, phantom_skip);

       update_cells(period_switch, n_atoms, h, scaled_atom_coords,
         atom_coords, atom_cells, *first, *last, *phantoms, kc,
         phantom_skip, nc, nc_p);

       update_phantoms(n_atoms, atom_cells, phantom_skip, *phantoms,
         atom_coords);
}
