#include "build.h"
#include "ff.h"
#include "proto.h"

/* Initialize quantities needed for PME. */
void init_pme(double **h, double volume, double *a1, double *a2, 
              double *a3, double delta_grid, int *n_grid_x, int *n_grid_y, 
              int *n_grid_z, double **b_mod2_x, double **b_mod2_y,
              double **b_mod2_z, int b_spline_order, double *b1, double *b2,
              double *b3, fft *elec) 
{
   double a_1_mag, a_2_mag, a_3_mag, a_1_dot_a_2, a_2_dot_a_3, a_3_dot_a_1,
       area_1_2, area_2_3, area_3_1, a_1_perp, a_2_perp, a_3_perp, atemp[3],
       norm;
   int pow_2, pow_3, pow_5;
   int n_grid, n_grid_old, n_grid_x_old, n_grid_y_old, n_grid_z_old;
#ifdef COMPAQ
   int status, stride_flag;
#endif

   a1[0] = h[0][0];
   a1[1] = h[0][1];
   a1[2] = h[0][2];

   a2[0] = h[1][0];
   a2[1] = h[1][1];
   a2[2] = h[1][2];

   a3[0] = h[2][0];
   a3[1] = h[2][1];
   a3[2] = h[2][2];

   /* Calculate variables that depend on dimensions of computational box. */
   a_1_mag = magnitude(a1, 3);
   a_2_mag = magnitude(a2, 3);
   a_3_mag = magnitude(a3, 3);
   a_1_dot_a_2 = dot_product(a1, a2, 3);
   a_2_dot_a_3 = dot_product(a2, a3, 3);
   a_3_dot_a_1 = dot_product(a3, a1, 3);
   area_1_2 = sqrt(SQR(a_1_mag) * SQR(a_2_mag) - SQR(a_1_dot_a_2));
   area_2_3 = sqrt(SQR(a_2_mag) * SQR(a_3_mag) - SQR(a_2_dot_a_3));
   area_3_1 = sqrt(SQR(a_3_mag) * SQR(a_1_mag) - SQR(a_3_dot_a_1));

/*   volume = a_1.x * a_2.y * a_3.z; */
   a_1_perp = volume / area_2_3;
   a_2_perp = volume / area_3_1;
   a_3_perp = volume / area_1_2;

   /* Calculate unconventional primitive reciprocal lattice vectors. */
   norm = 1.0 / volume;
   cross_product(a2, a3, atemp, 3);
   scale_vector(atemp, b1, 3, norm);
   cross_product(a3, a1, atemp, 3);
   scale_vector(atemp, b2, 3, norm);
   cross_product(a1, a2, atemp, 3);
   scale_vector(atemp, b3, 3, norm);

   n_grid_x_old = (*n_grid_x);
   n_grid_y_old = (*n_grid_y);
   n_grid_z_old = (*n_grid_z);
   n_grid_old = n_grid_x_old + n_grid_y_old + n_grid_z_old;
   
   *n_grid_x = (int) (a_1_perp / delta_grid);
   while (!product_of_powers_of_2_3_5(*n_grid_x, &pow_2, &pow_3, &pow_5))
     ++(*n_grid_x);
   *n_grid_y = (int) (a_2_perp / delta_grid);
   while (!product_of_powers_of_2_3_5(*n_grid_y, &pow_2, &pow_3, &pow_5))
     ++(*n_grid_y);
   *n_grid_z = (int) (a_3_perp / delta_grid);
   while (!product_of_powers_of_2_3_5(*n_grid_z, &pow_2, &pow_3, &pow_5))
     ++(*n_grid_z);

   n_grid = (*n_grid_x) * (*n_grid_y) * (*n_grid_z);


   /* Allocate or reallocate memory for arrays used in PME routines
      if necessary. */
   if (*n_grid_x != n_grid_x_old)
      *b_mod2_x = realloc(*b_mod2_x, (*n_grid_x) * sizeof(double));
   if (*n_grid_y != n_grid_y_old)
      *b_mod2_y = realloc(*b_mod2_y, (*n_grid_y) * sizeof(double));
   if (*n_grid_z != n_grid_z_old)
      *b_mod2_z = realloc(*b_mod2_z, (*n_grid_z) * sizeof(double));

  if (n_grid != n_grid_old){

#if defined(COMPAQ)
    /* Reallocate memory for interpolation arrays. */
    /* elec->Q is equivalent to (*elec).Q, i.e this is a pointer on the
       address of the element array Q. */
    elec->Q_linear_re = realloc(elec->Q_linear_re, n_grid * sizeof(double));
    elec->Q_linear_im = realloc(elec->Q_linear_im, n_grid * sizeof(double));

    /* Calculate squared moduli of 1d discrete Fourier transforms of 
       B-spline coefficient array and initialize coefficient array for FFT 
       routine (we make use of the COMPAQ FFT library). */
    B_spline_dft_moduli(b_spline_order, *n_grid_x, *n_grid_y, *n_grid_z,
                        *b_mod2_x, *b_mod2_y, *b_mod2_z);

    /* We want to pass the address of coeff and not its value which is
       elec->coeff (equivalent to (*elec).coeff). */
    status = zfft_exit_3d_(&(elec->coeff)); 
    stride_flag = TRUE;
    status = zfft_init_3d_(n_grid_x, n_grid_y, n_grid_z, &(elec->coeff), 
                           &stride_flag);


#elif defined(SGI)
    elec->Q_linear = realloc(elec->Q_linear, n_grid * sizeof(zomplex));

    /* Calculate squared moduli of 1d discrete Fourier transforms of
       B-spline coefficient array and initialize coefficient array for FFT
       routine (we make use of the SGI FFT library). */
    B_spline_dft_moduli(b_spline_order, *n_grid_x, *n_grid_y, *n_grid_z,
                        *b_mod2_x, *b_mod2_y, *b_mod2_z);

    /* Here coeff is the address of the element array *coeff. */ 
    if (elec->coeff != NULL)
       free(elec->coeff);
    elec->coeff = zfft3di(*n_grid_x, *n_grid_y, *n_grid_z, NULL);

#elif defined(FFTW)

    elec->Q_linear = realloc(elec->Q_linear, n_grid * sizeof(fftw_complex));

if(elec->pfw != NULL) fftwnd_destroy_plan(elec->pfw);
if(elec->pbw != NULL) fftwnd_destroy_plan(elec->pbw);

    elec->pfw = fftw3d_create_plan(*n_grid_x, *n_grid_y, *n_grid_z,
                FFTW_FORWARD, FFTW_MEASURE | FFTW_IN_PLACE);
    elec->pbw = fftw3d_create_plan(*n_grid_x, *n_grid_y, *n_grid_z,
                FFTW_BACKWARD, FFTW_MEASURE | FFTW_IN_PLACE);

    /* Calculate squared moduli of 1d discrete Fourier transforms of
       B-spline coefficient array and initialize coefficient array for FFT
       routine (we make use of the SGI FFT library). */
    B_spline_dft_moduli(b_spline_order, *n_grid_x, *n_grid_y, *n_grid_z,
                        *b_mod2_x, *b_mod2_y, *b_mod2_z);

#endif
  }

}
