/* Reciprocal-space Ewald summation routines for molecular
   simulation programs. */

#include "build.h"

/* Compute reciprocal-space contributions to electrostatic energy,
   virial, and forces, using the conventianal Ewald method. */
void ewald_recip(int n_atoms, double volume,
   double **scaled_atom_coords, double *atom_charges,
   double *b1, double *b2, double *b3,
   int m1_max, int m2_max, int m3_max,
   double alpha, double G2_cutoff,
   double **cos_G_1_dot_r, double **sin_G_1_dot_r,
   double **cos_G_2_dot_r, double **sin_G_2_dot_r,
   double **cos_G_3_dot_r, double **sin_G_3_dot_r,
   double *q_cos_G_12_dot_r, double *q_sin_G_12_dot_r,
   double *q_cos_G_123_dot_r, double *q_sin_G_123_dot_r,
   double *pe_coul_recip, double **vir_coul_recip,
   double **atom_forces_coul_recip)
{
   int i, j, m1, m2, m3, m2_min, m2_abs, m2_sign, m3_min, m3_abs, m3_sign;
   double b_1_dot_r, b_2_dot_r, b_3_dot_r,
      G_x, G_y, G_z, G_x2, G_y2, G_z2, G_y_m1, G_z_m1, G_z_m2, G2,
      one_over_four_alpha2, G2_over_four_alpha2,
      exp_minus_G2_over_four_alpha2,
      sin_G_2_dot_r_m2, sin_G_3_dot_r_m3,
      sum_q_cos_G_123_dot_r, sum_q_sin_G_123_dot_r,
      four_pi_over_volume, u_coul, kernel, struct2, v_coul, f_coul;

   /* Compute various constants. */
   four_pi_over_volume = 4.0 * PI / volume;
   one_over_four_alpha2 = 1.0 / SQR(2.0 * alpha);

   /* Zero accumulators. */
   *pe_coul_recip = 0.0;
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         vir_coul_recip[i][j] = 0.0;
   for (i = 0; i < n_atoms; ++i) {
      atom_forces_coul_recip[i][0] = 0.0;
      atom_forces_coul_recip[i][1] = 0.0;
      atom_forces_coul_recip[i][2] = 0.0;
   }

   /* Calculate and store sines and cosines of dot products of multiples
      of reciprocal lattice basis vectors and atomic positions, using
      recursion to build up higher multiples. */
   for (i = 0; i < n_atoms; ++i) {

     if (atom_charges[i] != 0.0) {
      cos_G_1_dot_r[i][0] = cos_G_2_dot_r[i][0] = cos_G_3_dot_r[i][0] = 1.0;
      sin_G_1_dot_r[i][0] = sin_G_2_dot_r[i][0] = sin_G_3_dot_r[i][0] = 0.0;
      b_1_dot_r = TWO_PI * scaled_atom_coords[i][0];
      b_2_dot_r = TWO_PI * scaled_atom_coords[i][1];
      b_3_dot_r = TWO_PI * scaled_atom_coords[i][2];
      cos_G_1_dot_r[i][1] = cos(b_1_dot_r);
      sin_G_1_dot_r[i][1] = sin(b_1_dot_r);
      cos_G_2_dot_r[i][1] = cos(b_2_dot_r);
      sin_G_2_dot_r[i][1] = sin(b_2_dot_r);
      cos_G_3_dot_r[i][1] = cos(b_3_dot_r);
      sin_G_3_dot_r[i][1] = sin(b_3_dot_r);
      for (m1 = 2; m1 <= m1_max; ++m1) {
         cos_G_1_dot_r[i][m1]
            = cos_G_1_dot_r[i][m1-1] * cos_G_1_dot_r[i][1]
            - sin_G_1_dot_r[i][m1-1] * sin_G_1_dot_r[i][1];
         sin_G_1_dot_r[i][m1]
            = cos_G_1_dot_r[i][m1-1] * sin_G_1_dot_r[i][1]
            + sin_G_1_dot_r[i][m1-1] * cos_G_1_dot_r[i][1];
      }
      for (m2 = 2; m2 <= m2_max; ++m2) {
         cos_G_2_dot_r[i][m2]
            = cos_G_2_dot_r[i][m2-1] * cos_G_2_dot_r[i][1]
            - sin_G_2_dot_r[i][m2-1] * sin_G_2_dot_r[i][1];
         sin_G_2_dot_r[i][m2]
            = cos_G_2_dot_r[i][m2-1] * sin_G_2_dot_r[i][1]
            + sin_G_2_dot_r[i][m2-1] * cos_G_2_dot_r[i][1];
      }
      for (m3 = 2; m3 <= m3_max; ++m3) {
         cos_G_3_dot_r[i][m3]
            = cos_G_3_dot_r[i][m3-1] * cos_G_3_dot_r[i][1]
            - sin_G_3_dot_r[i][m3-1] * sin_G_3_dot_r[i][1];
         sin_G_3_dot_r[i][m3]
            = cos_G_3_dot_r[i][m3-1] * sin_G_3_dot_r[i][1]
            + sin_G_3_dot_r[i][m3-1] * cos_G_3_dot_r[i][1];
      }
    }
   }

   /* Loop over reciprocal lattice vectors, covering one half of the
      reciprocal-space domain. */
   m2_min = 0;
   m3_min = 1;
   for (m1 = 0; m1 <= m1_max; ++m1) {
      G_x = m1 * b1[0];
      G_x2 = SQR(G_x);
      G_y_m1 = m1 * b1[1];
      G_z_m1 = m1 * b1[2];
      for (m2 = m2_min; m2 <= m2_max; ++m2) {
         m2_abs = (m2 < 0) ? - m2 : m2;
         m2_sign = (m2 < 0) ? - 1 : 1;
         G_y = G_y_m1 + m2 * b2[1];
         G_y2 = SQR(G_y);
         G_z_m2 = m2 * b2[2];
         for (i = 0; i < n_atoms; ++i) {
           if (atom_charges[i] != 0.0){
            sin_G_2_dot_r_m2 = m2_sign * sin_G_2_dot_r[i][m2_abs];
            q_cos_G_12_dot_r[i] = atom_charges[i]
               * (cos_G_1_dot_r[i][m1] * cos_G_2_dot_r[i][m2_abs]
               - sin_G_1_dot_r[i][m1] * sin_G_2_dot_r_m2);
            q_sin_G_12_dot_r[i] = atom_charges[i]
               * (cos_G_1_dot_r[i][m1] * sin_G_2_dot_r_m2
               + sin_G_1_dot_r[i][m1] * cos_G_2_dot_r[i][m2_abs]);
           }
         }
         for (m3 = m3_min; m3 <= m3_max; ++m3) {
            m3_abs = (m3 < 0) ? - m3 : m3;
            m3_sign = (m3 < 0) ? - 1 : 1;
            G_z = G_z_m1 + G_z_m2 + m3 * b3[2];
            G_z2 = SQR(G_z);
            G2 = G_x2 + G_y2 + G_z2;

            /* If reciprocal lattice vector is within cutoff, calculate
               contribution to Coulomb interactions. */
            if (G2 < G2_cutoff) {

               /* Calculate reciprocal-space Coulomb kernel. */
               G2_over_four_alpha2 = G2 * one_over_four_alpha2;
               exp_minus_G2_over_four_alpha2 = exp(- G2_over_four_alpha2);
               kernel = four_pi_over_volume * exp_minus_G2_over_four_alpha2 / G2;
               
               /* Calculate Fourier components of charge distribution. */
               sum_q_cos_G_123_dot_r = sum_q_sin_G_123_dot_r = 0.0;
               for (i = 0; i < n_atoms; ++i) {

                if (atom_charges[i] != 0.0){
                  sin_G_3_dot_r_m3 = m3_sign * sin_G_3_dot_r[i][m3_abs];
                  q_cos_G_123_dot_r[i]
                     = q_cos_G_12_dot_r[i] * cos_G_3_dot_r[i][m3_abs]
                     - q_sin_G_12_dot_r[i] * sin_G_3_dot_r_m3;
                  q_sin_G_123_dot_r[i]
                     = q_cos_G_12_dot_r[i] * sin_G_3_dot_r_m3
                     + q_sin_G_12_dot_r[i] * cos_G_3_dot_r[i][m3_abs];
                  sum_q_cos_G_123_dot_r += q_cos_G_123_dot_r[i];
                  sum_q_sin_G_123_dot_r += q_sin_G_123_dot_r[i];
                }
               }

               /* Add contributions to forces. */
               for (i = 0; i < n_atoms; ++i) {
                if (atom_charges[i] != 0.0) {
                  f_coul = 2.0 * kernel
                     * (q_sin_G_123_dot_r[i] * sum_q_cos_G_123_dot_r
                     - q_cos_G_123_dot_r[i] * sum_q_sin_G_123_dot_r);
                  atom_forces_coul_recip[i][0] += G_x * f_coul;
                  atom_forces_coul_recip[i][1] += G_y * f_coul;
                  atom_forces_coul_recip[i][2] += G_z * f_coul;
                }
               }

               struct2 = SQR(sum_q_cos_G_123_dot_r)
                  + SQR(sum_q_sin_G_123_dot_r);
               u_coul = kernel * struct2;
               *pe_coul_recip += u_coul;
               v_coul = 2.0 * u_coul * (1.0 / G2 + one_over_four_alpha2);
               vir_coul_recip[0][0] += v_coul * SQR(G_x) - u_coul;
               vir_coul_recip[1][1] += v_coul * SQR(G_y) - u_coul;
               vir_coul_recip[2][2] += v_coul * SQR(G_z) - u_coul;
               vir_coul_recip[0][1] += v_coul * G_x * G_y;
               vir_coul_recip[0][2] += v_coul * G_x * G_z;
               vir_coul_recip[1][2] += v_coul * G_y * G_z;
            }
         }
         m3_min = - m3_max;
      }
      m2_min = - m2_max;
   }
}
