/* Bonded force routines for molecular modeling package. */

#include "build.h"
#include "ff.h"
#include "proto.h"

/* Calculate bond stretching interactions. */
void stretch_harmonic(int period_switch, int eval_flag, 
             double **f_stretch, int *mol_first_atm, int *n_bonds_per_mol,
             double **at_stress_stretch, double *pe_stretch, 
             int n_atoms, int n_mols, int *mol_species,
             int **temp_bonds_1, int **temp_bonds_2, double **h, 
             double **scaled_atom_coords, bond_prop **bond_props)
{
   int i_species, i, j, i_bond, i_mol, rel_1, rel_2, abs_1, abs_2, skip,
      v_rel, v_abs, v_skip, k;
   double x_bond, y_bond, z_bond, r2_bond, r_bond, fx_bond, fy_bond, fz_bond,
      elongation, dum1, dum2, p_norm, s_bond[NDIM], f_bond[NDIM], volume,
      bond[NDIM], spring, r_equil;

   volume = h[0][0]*h[1][1]*h[2][2];

   /* Zero forces. */
   for (i = 0; i < n_atoms; ++i)
      for (k = 0; k < NDIM; ++k)
         f_stretch[i][k] = 0.0;

   /* If eval_flag == 1, zero bond stretching contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_stretch = 0.0;
      for (i = 0; i < NDIM; ++i)
          for (k = 0; k < NDIM; ++k)
             at_stress_stretch[i][k] = 0.0;
   }


   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom and bond vector label offsets. */
      skip = mol_first_atm[i_mol];

      /* Loop over bonds. */
    if(i_species == 0)
      for (i_bond = 0; i_bond < n_bonds_per_mol[i_species]; ++i_bond) {

         /* Get relative labels of atoms that share bond. */
         rel_1 = temp_bonds_1[i_species][i_bond];
         rel_2 = temp_bonds_2[i_species][i_bond];


         /* Calculate absolute labels of atoms that share bond. */
         abs_1 = skip + rel_1;
         abs_2 = skip + rel_2;

         /* Get spring constant and equilibrium bond length. */
         spring = bond_props[i_species][i_bond].spring;
         r_equil = bond_props[i_species][i_bond].r_equil;


         /* Calculate bond vector and bond length. */
         for (k = 0; k < NDIM; ++k)
         {
            s_bond[k] = scaled_atom_coords[abs_2][k]
                      - scaled_atom_coords[abs_1][k];
            s_bond[k] -= NINT(s_bond[k]);
         }
         bond[0] = h[0][0] * s_bond[0] + h[0][1] * s_bond[1]
                  + h[0][2] * s_bond[2];
         bond[1] = h[1][1] * s_bond[1] + h[1][2] * s_bond[2];
         bond[2] = h[2][2] * s_bond[2];
         r2_bond = SQR(bond[0]) + SQR(bond[1]) + SQR(bond[2]);
         r_bond = sqrt(r2_bond);

         /* Calculate forces. */
         elongation = r_bond - r_equil;
         dum1 = spring * elongation;
         dum2 = dum1 / r_bond;
         for (k = 0; k < NDIM; ++k)
         f_bond[k] = dum2 * bond[k];
         for (k = 0; k < NDIM; ++k){
             f_stretch[abs_1][k] += f_bond[k];
             f_stretch[abs_2][k] -= f_bond[k];
         }
/* printf(" bond atom 1 %d 2 %d force x %g y %g z %g\n", abs_1,abs_2,f_stretch[abs_1][0],
        f_stretch[abs_1][1], f_stretch[abs_1][2]); */

         /* Add contributions to potential energy and atomic pressure
            tensor. */
         if (eval_flag) {
            *pe_stretch += dum1 * elongation;
            at_stress_stretch[0][0] -= bond[0] * f_bond[0];
            at_stress_stretch[1][1] -= bond[1] * f_bond[1];
            at_stress_stretch[2][2] -= bond[2] * f_bond[2];
            at_stress_stretch[0][1] -= bond[0] * f_bond[1];
            at_stress_stretch[0][2] -= bond[0] * f_bond[2];
            at_stress_stretch[1][2] -= bond[1] * f_bond[2];
         }
      }
   }

   /* If eval_flag == 1, normalize bond stretching contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_stretch *= 0.5;
      p_norm =  1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_stretch[i][j] *= p_norm;
   }
}

/* Calculate bond angle bending interactions, using a cosine harmonic form
   for the bond angle bending potential. */

void bend_cos_harmonic(int period_switch, int eval_flag,
             double **f_bend, int *mol_first_atm, int *n_angles_per_mol,
             double **at_stress_bend, double *pe_bend, 
             int n_atoms, int n_mols, int *mol_species,
             angle_prop **angle_props, double **h,
             double **scaled_atom_coords)
{
   int i_species, i, j, k, i_angle, i_mol, rel_1, rel_2, rel_3,
      skip, abs_1, abs_2, abs_3, v_skip, v_abs_1, v_abs_2;
   double bond_1[NDIM], c_11, s_bond_1[NDIM], s_bond_2[NDIM],
      bond_2[NDIM], c_22, c_12, norm_1, norm_2,
      norm_3, cos_theta, delta_cos_theta, d_pe_d_cos_theta,
      d_cos_theta_d_b1[NDIM], d_cos_theta_d_b2[NDIM], 
      f_bond_1[NDIM], f_bond_2[NDIM], p_norm, volume, spring_bend,
      cos_theta_equil;


    volume = h[0][0] * h[1][1] * h[2][2] ;

   /* Zero forces. */
   for (i = 0; i < n_atoms; ++i) {
       for (k = 0; k < NDIM; ++k)
       f_bend[i][k] = 0.0;
   }

   /* If vars.eval_flag == 1, zero bond angle bending contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_bend = 0.0;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_bend[i][j] = 0.0;
   }
 
   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom and bond vector label offsets. */
      skip = mol_first_atm[i_mol];

      /* Loop over bond angles. */
     if(i_species == 0)
      for (i_angle = 0; i_angle < n_angles_per_mol[i_species]; ++i_angle) {

         /* Get relative labels of atoms that share angle. */
         rel_1 = angle_props[i_species][i_angle].atom_1;
         rel_2 = angle_props[i_species][i_angle].atom_2;
         rel_3 = angle_props[i_species][i_angle].atom_3;

         /* Calculate absolute labels of atoms that share angle. */
         abs_1 = skip + rel_1;
         abs_2 = skip + rel_2;
         abs_3 = skip + rel_3;

         /* Get spring constant and cosine of equilibrium bond angle. */
         spring_bend =  angle_props[i_species][i_angle].spring_bend;
         cos_theta_equil = angle_props[i_species][i_angle].cos_theta_equil;


         /* Calculate bond vector and bond length. */
         for (k = 0; k < NDIM; ++k)
         {
            s_bond_1[k] = scaled_atom_coords[abs_2][k]
                      - scaled_atom_coords[abs_1][k];
            s_bond_2[k] = scaled_atom_coords[abs_3][k]
                      - scaled_atom_coords[abs_2][k];
            s_bond_1[k] -= NINT(s_bond_1[k]);
            s_bond_2[k] -= NINT(s_bond_2[k]);
         }
         bond_1[0] = h[0][0] * s_bond_1[0] + h[0][1] * s_bond_1[1]
                  + h[0][2] * s_bond_1[2];
         bond_1[1] = h[1][1] * s_bond_1[1] + h[1][2] * s_bond_1[2];
         bond_1[2] = h[2][2] * s_bond_1[2];
         bond_2[0] = h[0][0] * s_bond_2[0] + h[0][1] * s_bond_2[1]
                  + h[0][2] * s_bond_2[2];
         bond_2[1] = h[1][1] * s_bond_2[1] + h[1][2] * s_bond_2[2];
         bond_2[2] = h[2][2] * s_bond_2[2];
         c_11 = SQR(bond_1[0]) + SQR(bond_1[1]) + SQR(bond_1[2]);
         c_22 = SQR(bond_2[0]) + SQR(bond_2[1]) + SQR(bond_2[2]);

         /* Calculate forces. */
         c_12 = bond_1[0] * bond_2[0] + bond_1[1] * bond_2[1] 
                + bond_1[2] * bond_2[2];
         norm_1 = 1.0 / sqrt(c_11 * c_22);
         cos_theta = - norm_1 * c_12;
         delta_cos_theta = cos_theta - cos_theta_equil;
         d_pe_d_cos_theta = spring_bend * delta_cos_theta;
         norm_2 = c_12 / c_11;
         norm_3 = c_12 / c_22;
         d_cos_theta_d_b1[0] = norm_1 * (norm_2 * bond_1[0] - bond_2[0]);
         d_cos_theta_d_b1[1] = norm_1 * (norm_2 * bond_1[1] - bond_2[1]);
         d_cos_theta_d_b1[2] = norm_1 * (norm_2 * bond_1[2] - bond_2[2]);
         d_cos_theta_d_b2[0] = norm_1 * (norm_3 * bond_2[0] - bond_1[0]);
         d_cos_theta_d_b2[1] = norm_1 * (norm_3 * bond_2[1] - bond_1[1]);
         d_cos_theta_d_b2[2] = norm_1 * (norm_3 * bond_2[2] - bond_1[2]);
         for (k = 0; k < NDIM; ++k) {
             f_bond_1[k] = d_pe_d_cos_theta * d_cos_theta_d_b1[k];
             f_bond_2[k] = d_pe_d_cos_theta * d_cos_theta_d_b2[k];
         }
         for ( k = 0; k < NDIM; ++k) {
            f_bend[abs_1][k] += f_bond_1[k];
            f_bend[abs_2][k] += f_bond_2[k] - f_bond_1[k];
            f_bend[abs_3][k] -= f_bond_2[k];
             
         }

   /*    printf(" d_pe_d_cos_theta %lf \n", d_pe_d_cos_theta); */
         /* Add contributions to potential energy and atomic pressure tensor. */
         if (eval_flag) {
            *pe_bend += spring_bend * SQR(delta_cos_theta);
            at_stress_bend[0][0]
               -= bond_1[0] * f_bond_1[0] + bond_2[0] * f_bond_2[0];
            at_stress_bend[1][1]
               -= bond_1[1] * f_bond_1[1] + bond_2[1] * f_bond_2[1];
            at_stress_bend[2][2]
               -= bond_1[2] * f_bond_1[2] + bond_2[2] * f_bond_2[2];
            at_stress_bend[0][1]
               -= bond_1[0] * f_bond_1[1] + bond_2[0] * f_bond_2[1];
            at_stress_bend[0][2]
               -= bond_1[0] * f_bond_1[2] + bond_2[0] * f_bond_2[2];
            at_stress_bend[1][2]
               -= bond_1[1] * f_bond_1[2] + bond_2[1] * f_bond_2[2];
         }
      }
   }

   /* If vars.eval_flag == 1, normalize bond angle bending contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_bend *= 0.5;
      p_norm = 1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_bend[i][j] *= p_norm;
   }
}


/* Calculate torsional interactions, using a torsional potential that is
   a power series in the cosine of the dihedral angle (most commonly used
   torsional potentials can be cast in this form). If vars.min_flag >= 1,
   selected torsions are restrained using a harmonic restraining potential
   instead of a cosine expansion and/or are set to zero (see below). */
void torsion(void)
{
   int i_species, i_mol, i, j, i_dihed, skip, v_skip;
   double p_norm;


   /* Zero forces. */
   for (i = 0; i < n_atoms; ++i) {
       for ( k = 0; k < NDIM; ++k)
          f_torsion[i][k] = 0.0;
   }

   /* If vars.eval_flag == 1, zero dihedral torsion contributions to
      potential energy and pressure. */
   if (eval_flag) {
       *pe_tors = 0.0;
       *pe_restraint = 0.0;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_tors[i][j] = 0.0;
   }
     
   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom and bond vector label offsets. */
      skip = mol_first_atm[i_mol];

      /* Loop over dihedral angles. */
      for (i_dihed = 0; i_dihed < n_diheds_per_mol[i_species]; ++i_dihed) {

         /* If vars.min_flag == 0, a cosine expansion is used for all torsional
            potentials, unless the torsional potential is to be modified, in
            which case a full Fourier expansion is used. If vars.min_flag == 1,
            a harmonic potential is used for selected (driven or restrained)
            torsions, the potentials for other torsions around driven or
            restrained bonds are set to zero, and a cosine expansion is used
            for the remaining torsions. If vars.min_flag == 2, the potentials
            for torsions around a driven bond are set to zero, and a cosine
            expansion is used for the remaining torsions. If vars.min_flag == 3,
            the potential for the driver torsion is set to zero, and a cosine
            expansion is used for the remaining torsions. */
         if (vars.min_flag == 0) {
            if (dihed_props[i_species][i_dihed].mod_flag && vars.mod_flag)
               torsion_fourier_expansion(i_species, i_dihed, skip, v_skip);
            else
               torsion_cosine_expansion(i_species, i_dihed, skip, v_skip);
         }
         else if (vars.min_flag == 1) {
            if (dihed_props[i_species][i_dihed].mask_2)
               torsion_harmonic(i_species, i_dihed, skip, v_skip);
            else if (!dihed_props[i_species][i_dihed].mask_1)
               torsion_cosine_expansion(i_species, i_dihed, skip, v_skip);
         }
         else if (vars.min_flag == 2) {
            if (!dihed_props[i_species][i_dihed].mask_3)
               torsion_cosine_expansion(i_species, i_dihed, skip, v_skip);
         }
         else {
            if (!dihed_props[i_species][i_dihed].mask_4)
               torsion_cosine_expansion(i_species, i_dihed, skip, v_skip);
         }
      }
   }

   /* If vars.eval_flag == 1, normalize dihedral torsion contributions to
      potential energy and pressure. */
   if (eval_flag) {
      p_norm =  1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_tors[i][j] *= p_norm;
   }
}

/* Add contributions to dihedral torsion interactions for a given
   dihedral angle, using a cosine expansion form for the dihedral
   potential. */
void torsion_cosine_expansion(int i_species, int i_dihed, int skip, int v_skip)
{
   int rel_1, rel_2, rel_3, rel_4,
      v_rel_1, v_rel_2, v_rel_3, n_order,
      abs_1, abs_2, abs_3, abs_4,
      v_abs_1, v_abs_2, v_abs_3, n;
   double a[MAX_ORD+1], a_prime[MAX_ORD+1],
      x_bond_1, y_bond_1, z_bond_1, c_11,
      x_bond_2, y_bond_2, z_bond_2, c_22,
      x_bond_3, y_bond_3, z_bond_3, c_33,
      c_12, c_13, c_23, d_12, d_23, e_1, e_2, e_3,
      norm_1, norm_2, norm_3, norm_4, norm_5, norm_6, norm_7,
      cos_phi, cosines[MAX_ORD+1],
      d_pe_d_cos_phi,
      d_cos_phi_d_xb1, d_cos_phi_d_yb1, d_cos_phi_d_zb1,
      d_cos_phi_d_xb2, d_cos_phi_d_yb2, d_cos_phi_d_zb2,
      d_cos_phi_d_xb3, d_cos_phi_d_yb3, d_cos_phi_d_zb3,
      fx_bond_1, fy_bond_1, fz_bond_1,
      fx_bond_2, fy_bond_2, fz_bond_2,
      fx_bond_3, fy_bond_3, fz_bond_3;

   /* Get relative labels of atoms that share dihedral angle. */
   rel_1 = dihed_props[i_species][i_dihed].atom_1;
   rel_2 = dihed_props[i_species][i_dihed].atom_2;
   rel_3 = dihed_props[i_species][i_dihed].atom_3;
   rel_4 = dihed_props[i_species][i_dihed].atom_4;

   /* Get relative labels of bond vectors. */
   v_rel_1 = vector_index[i_species][rel_1][rel_2];
   v_rel_2 = vector_index[i_species][rel_2][rel_3];
   v_rel_3 = vector_index[i_species][rel_3][rel_4];

   /* Get coefficients for cosine expansion. */
   n_order = dihed_props[i_species][i_dihed].n_order;
   for (n = 0; n <= n_order; ++n) {
      a[n] = dihed_props[i_species][i_dihed].c_a[n];
      a_prime[n] = n * a[n];
   }

   /* Calculate absolute labels of atoms that share dihedral angle. */
   abs_1 = skip + rel_1;
   abs_2 = skip + rel_2;
   abs_3 = skip + rel_3;
   abs_4 = skip + rel_4;

   /* Calculate absolute labels of bond vectors. */
   v_abs_1 = v_skip + v_rel_1;
   v_abs_2 = v_skip + v_rel_2;
   v_abs_3 = v_skip + v_rel_3;

   /* Get bond vectors. */
   x_bond_1 = bond_vectors[v_abs_1].x;
   y_bond_1 = bond_vectors[v_abs_1].y;
   z_bond_1 = bond_vectors[v_abs_1].z;
   c_11 = bond_vectors[v_abs_1].r2;
   x_bond_2 = bond_vectors[v_abs_2].x;
   y_bond_2 = bond_vectors[v_abs_2].y;
   z_bond_2 = bond_vectors[v_abs_2].z;
   c_22 = bond_vectors[v_abs_2].r2;
   x_bond_3 = bond_vectors[v_abs_3].x;
   y_bond_3 = bond_vectors[v_abs_3].y;
   z_bond_3 = bond_vectors[v_abs_3].z;
   c_33 = bond_vectors[v_abs_3].r2;

   /* Calculate cosine of dihedral angle. */
   c_12 = x_bond_1 * x_bond_2 + y_bond_1 * y_bond_2 + z_bond_1 * z_bond_2;
   c_13 = x_bond_1 * x_bond_3 + y_bond_1 * y_bond_3 + z_bond_1 * z_bond_3;
   c_23 = x_bond_2 * x_bond_3 + y_bond_2 * y_bond_3 + z_bond_2 * z_bond_3;
   d_12 = c_11 * c_22 - SQR(c_12);
   d_23 = c_22 * c_33 - SQR(c_23);
   e_1 = c_12 * c_13 - c_23 * c_11;
   e_2 = c_12 * c_23 - c_13 * c_22;
   e_3 = c_13 * c_23 - c_12 * c_33;
   norm_1 = 1.0 / sqrt(d_12 * d_23);
   norm_2 = norm_1 / d_12;
   norm_3 = norm_1 / d_23;
   norm_4 = norm_2 * c_22;
   norm_5 = norm_3 * c_22;
   norm_6 = norm_2 * e_1;
   norm_7 = norm_3 * e_3;
   cos_phi = e_2 * norm_1;

   /* Calculate forces. */
   cosines[1] = cos_phi;
   d_pe_d_cos_phi = a_prime[1];
   for (n = 2; n <= n_order; ++n) {
      cosines[n] = cosines[n - 1] * cos_phi;
      d_pe_d_cos_phi += a_prime[n] * cosines[n - 1];
   }
   d_cos_phi_d_xb1 = norm_4
      * (- e_2 * x_bond_1 - e_1 * x_bond_2 - d_12 * x_bond_3);
   d_cos_phi_d_yb1 = norm_4
      * (- e_2 * y_bond_1 - e_1 * y_bond_2 - d_12 * y_bond_3);
   d_cos_phi_d_zb1 = norm_4
      * (- e_2 * z_bond_1 - e_1 * z_bond_2 - d_12 * z_bond_3);
   d_cos_phi_d_xb2 = norm_6 * (c_12 * x_bond_2 - c_22 * x_bond_1)
      + norm_7 * (c_23 * x_bond_2 - c_22 * x_bond_3);
   d_cos_phi_d_yb2 = norm_6 * (c_12 * y_bond_2 - c_22 * y_bond_1)
      + norm_7 * (c_23 * y_bond_2 - c_22 * y_bond_3);
   d_cos_phi_d_zb2 = norm_6 * (c_12 * z_bond_2 - c_22 * z_bond_1)
      + norm_7 * (c_23 * z_bond_2 - c_22 * z_bond_3);
   d_cos_phi_d_xb3 = norm_5
      * (- e_2 * x_bond_3 - e_3 * x_bond_2 - d_23 * x_bond_1);
   d_cos_phi_d_yb3 = norm_5
      * (- e_2 * y_bond_3 - e_3 * y_bond_2 - d_23 * y_bond_1);
   d_cos_phi_d_zb3 = norm_5
      * (- e_2 * z_bond_3 - e_3 * z_bond_2 - d_23 * z_bond_1);
   fx_bond_1 = d_pe_d_cos_phi * d_cos_phi_d_xb1;
   fy_bond_1 = d_pe_d_cos_phi * d_cos_phi_d_yb1;
   fz_bond_1 = d_pe_d_cos_phi * d_cos_phi_d_zb1;
   fx_bond_2 = d_pe_d_cos_phi * d_cos_phi_d_xb2;
   fy_bond_2 = d_pe_d_cos_phi * d_cos_phi_d_yb2;
   fz_bond_2 = d_pe_d_cos_phi * d_cos_phi_d_zb2;
   fx_bond_3 = d_pe_d_cos_phi * d_cos_phi_d_xb3;
   fy_bond_3 = d_pe_d_cos_phi * d_cos_phi_d_yb3;
   fz_bond_3 = d_pe_d_cos_phi * d_cos_phi_d_zb3;
   atom_forces[abs_1].tors.x += fx_bond_1;
   atom_forces[abs_1].tors.y += fy_bond_1;
   atom_forces[abs_1].tors.z += fz_bond_1;
   atom_forces[abs_2].tors.x += fx_bond_2 - fx_bond_1;
   atom_forces[abs_2].tors.y += fy_bond_2 - fy_bond_1;
   atom_forces[abs_2].tors.z += fz_bond_2 - fz_bond_1;
   atom_forces[abs_3].tors.x += fx_bond_3 - fx_bond_2;
   atom_forces[abs_3].tors.y += fy_bond_3 - fy_bond_2;
   atom_forces[abs_3].tors.z += fz_bond_3 - fz_bond_2;
   atom_forces[abs_4].tors.x -= fx_bond_3;
   atom_forces[abs_4].tors.y -= fy_bond_3;
   atom_forces[abs_4].tors.z -= fz_bond_3;

   /* Add contributions to potential energy and atomic pressure tensor. */
   if (vars.eval_flag) {
      inst.pe_tors += a[0];
      for (n = 1; n <= n_order; ++n)
         inst.pe_tors += a[n] * cosines[n];
      inst.at_stress_tors[0][0] -= x_bond_1 * fx_bond_1
         + x_bond_2 * fx_bond_2 + x_bond_3 * fx_bond_3;
      inst.at_stress_tors[1][1] -= y_bond_1 * fy_bond_1
         + y_bond_2 * fy_bond_2 + y_bond_3 * fy_bond_3;
      inst.at_stress_tors[2][2] -= z_bond_1 * fz_bond_1
         + z_bond_2 * fz_bond_2 + z_bond_3 * fz_bond_3;
      inst.at_stress_tors[0][1] -= x_bond_1 * fy_bond_1
         + x_bond_2 * fy_bond_2 + x_bond_3 * fy_bond_3;
      inst.at_stress_tors[0][2] -= x_bond_1 * fz_bond_1
         + x_bond_2 * fz_bond_2 + x_bond_3 * fz_bond_3;
      inst.at_stress_tors[1][2] -= y_bond_1 * fz_bond_1
         + y_bond_2 * fz_bond_2 + y_bond_3 * fz_bond_3;
   }
}
