/* Thermodynamics routines for molecular modeling package */

#include "build.h"
#include "ff.h"
#include "proto.h"


/* Calculate thermodynamic quantities for MD simulation. */
void thermodynamics(int ke_flag, int n_mols, double **mol_vels,
                    double **atom_vels, double *atom_mass, double *mol_mass,
                    int *mol_first_atm, int *n_atoms_per_mol, 
                    int n_free_atom, int n_free_mol, double boltzmann,
                    double temperature, double **h, 
                    double *pe_vdw_s, double *pe_stretch, int n_atoms, 
                    int *mol_species, double *at_temp, double *ke, double *pe,
                    double *te, double **at_stress_vdw_s,
                    double **at_stress_stretch, double **mol_stress_kin,
                    double **at_stress_kin, 
                    double **mol_stress, double **at_stress,
                    double **mol_stress_vdw, double **mol_stress_vdw_s, 
                    double *pe_bend, double **at_stress_bend, 
                    int bend_switch, double *pe_coul_s, double *pe_coul_recip,
                    double **at_stress_coul_s, double **at_stress_coul_recip,
                    double **mol_stress_coul_s, double **mol_stress_coul_recip)
{
double volume, mol_ke, mol_temp, att_temp;

   /* Initialize potential and total energy. */
    *pe = 0.0;
    *te = 0.0;

   /* Compute volume. */
   volume = h[0][0]*h[1][1]*h[2][2];

   /* Calculate atomic and molecular kinetic energies and temperatures.
      If ke_flag == 1, these are calculated from the atomic and molecular
      velocities. Otherwise, their canonical values are used. */
    if (ke_flag) {
      center_of_mass_velocities(n_mols, mol_vels,
                 atom_vels, atom_mass, mol_mass,
                 mol_first_atm, n_atoms_per_mol, mol_species);
      kinetic_energy_and_temperature(n_mols, n_atoms, mol_species,
         mol_mass, atom_vels, mol_vels, boltzmann, n_free_atom, 
                       n_free_mol, atom_mass, at_temp, ke);
   }
   else {
      *ke = 0.5 * n_free_atom * boltzmann * temperature;
      *at_temp = temperature;
      mol_ke = 0.5 * n_free_mol * boltzmann * temperature;
      mol_temp = temperature;
   }

   /* Calculate total and intermolecular potential energies. */
   if(bend_switch)
      *pe = *pe_vdw_s + *pe_stretch + *pe_bend + *pe_coul_s + *pe_coul_recip;
   else
      *pe = *pe_vdw_s + *pe_stretch  + *pe_coul_s + *pe_coul_recip;

   /* Calculate total energy. */
   *te = *pe + *ke;

   /* Calculate total atomic and molecular stress tensors. */
   stress_tensors(ke_flag, at_stress_kin, mol_stress_kin,
                  atom_mass, atom_vels, mol_vels,
                  n_atoms, n_mols, mol_species, h,
                  mol_stress, at_stress, at_stress_vdw_s,
                  mol_stress_vdw, n_free_atom, n_free_mol,
                  boltzmann, temperature, at_stress_stretch, 
                  mol_stress_vdw_s, mol_mass, at_stress_bend,
                  at_stress_coul_s, at_stress_coul_recip,
                  mol_stress_coul_s, mol_stress_coul_recip);

}

/* Calculate center of mass velocities of molecules. */
void center_of_mass_velocities(int n_mols, double **mol_vels,
                 double **atom_vels, double *atom_mass, double *mol_mass,
                 int *mol_first_atm, int *n_atoms_per_mol, int *mol_species)
{
   int i_species, i_mol, i_atom, skip, abs, k;

   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Zero center of mass velocities. */
      for ( k = 0; k < NDIM; ++k)
      mol_vels[i_mol][k] = 0.0;

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Loop over atoms within molecule i_mol. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

         /* Calculate absolute label of atom. */
         abs = skip + i_atom;

         /* Add mass-weighted atomic velocities to center of mass velocities. */
        for (k = 0; k < NDIM;  ++k)
            mol_vels[i_mol][k] += atom_mass[abs] * atom_vels[abs][k];
      }

      /* Normalize center of mass velocities. */
      for (k = 0; k < NDIM;  ++k)
          mol_vels[i_mol][k] /= mol_mass[i_species];
   }
}

/* Calculate atomic and molecular kinetic energies and temperatures. */
  /* Number of Atomic degrees of freedom (n_free_atom = 3*n_atoms) */
  /* Number of molecular degrees of freedom (n_free_mol = 3*n_mols) */

void kinetic_energy_and_temperature(int n_mols, int n_atoms, int *mol_species,
            double *mol_mass, double **atom_vels, double **mol_vels,
            double boltzmann, int n_free_atom, int n_free_mol, double *atom_mass, 
            double *at_temp, double *ke)
{
   int i_species, i, i_mol;
   double mol_ke, mol_temp;

   *at_temp = 0.0;

   /* Calculate atomic kinetic energy and temperature. */
   *ke = 0.0;

   for (i = 0; i < n_atoms; ++i)
      *ke += atom_mass[i] * (SQR(atom_vels[i][0])
         + SQR(atom_vels[i][1]) + SQR(atom_vels[i][2]));
   *ke *= 0.5;
   *at_temp = 2.0 * *ke / (boltzmann * n_free_atom);

   /* Calculate molecular translational kinetic energy and molecular
      temperature. */
   if (n_mols == 1) {
      mol_ke = 0.0;
      mol_temp = 0.0;
   }
   else {
      mol_ke = 0.0;
      for (i_mol = 0; i_mol < n_mols; ++i_mol) {
         i_species = mol_species[i_mol];
         mol_ke += mol_mass[i_species]
            * (SQR(mol_vels[i_mol][0]) + SQR(mol_vels[i_mol][1]) 
                   + SQR(mol_vels[i_mol][2]));
      }
      mol_ke *= 0.5;
      mol_temp = 2.0 * mol_ke / (boltzmann * n_free_mol);
   }

}


/* Calculate total atomic and molecular stress tensors. */
void stress_tensors(int ke_flag, double **at_stress_kin, 
                    double **mol_stress_kin, double *atom_mass, 
                    double **atom_vels, double **mol_vels, int n_atoms, 
                    int n_mols, int *mol_species, double **h,
                    double **mol_stress, double **at_stress, 
                    double **at_stress_vdw_s, double **mol_stress_vdw, 
                    int n_free_atom, int n_free_mol, double boltzmann, 
                    double temperature, double **at_stress_stretch, 
                    double **mol_stress_vdw_s, double *mol_mass, 
                    double **at_stress_bend, double **at_stress_coul_s, 
                    double **at_stress_coul_recip, double **mol_stress_coul_s, 
                    double **mol_stress_coul_recip)
{
   int i_species, i, j, i_mol;
   double mass, p_norm, press_kin, volume;
   double at_press_kin, mol_press_kin;
   
   volume = h[0][0]*h[1][1]*h[2][2];

   /* Calculate kinetic contributions to atomic and molecular stress tensors.
      If ke_flag == 1, these are calculated from the atomic and molecular
      velocities. Otherwise, their canonical values are used. */
   if (ke_flag) {
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            at_stress_kin[i][j] = 0.0;
            mol_stress_kin[i][j] = 0.0;
         }
      for (i = 0; i < n_atoms; ++i) {
         mass = atom_mass[i];
         at_stress_kin[0][0] += mass * SQR(atom_vels[i][0]);
         at_stress_kin[1][1] += mass * SQR(atom_vels[i][1]);
         at_stress_kin[2][2] += mass * SQR(atom_vels[i][2]);
         at_stress_kin[0][1] += mass * atom_vels[i][0]*atom_vels[i][1];
         at_stress_kin[0][2] += mass * atom_vels[i][0]*atom_vels[i][2];
         at_stress_kin[1][2] += mass * atom_vels[i][1]*atom_vels[i][2];
      }
      for (i_mol = 0; i_mol < n_mols; ++i_mol) {
         i_species = mol_species[i_mol];
         mass = mol_mass[i_species];
         mol_stress_kin[0][0] += mass * SQR(mol_vels[i_mol][0]);
         mol_stress_kin[1][1] += mass * SQR(mol_vels[i_mol][1]);
         mol_stress_kin[2][2] += mass * SQR(mol_vels[i_mol][2]);
         mol_stress_kin[0][1] += mass * mol_vels[i_mol][0]*mol_vels[i_mol][1];
         mol_stress_kin[0][2] += mass * mol_vels[i_mol][0]*mol_vels[i_mol][2];
         mol_stress_kin[1][2] += mass * mol_vels[i_mol][1]*mol_vels[i_mol][2];
      }
      p_norm = 1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            at_stress_kin[i][j] *= p_norm;
            mol_stress_kin[i][j] *= p_norm;
         }
   }
   else {
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            at_stress_kin[i][j] = 0.0;
            mol_stress_kin[i][j] = 0.0;
         }
      press_kin = n_free_atom * boltzmann
         * temperature / (3.0 * volume);
      for (i = 0; i < 3; ++i)
         at_stress_kin[i][i] = press_kin;
      press_kin = n_free_mol * boltzmann
         * temperature / (3.0 * volume);
      for (i = 0; i < 3; ++i)
         mol_stress_kin[i][i] = press_kin;
   }

   /* Calculate total atomic stress tensor. */
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         at_stress[i][j] = at_stress_kin[i][j] + at_stress_vdw_s[i][j] 
            + at_stress_stretch[i][j] + at_stress_bend[i][j] 
             + at_stress_coul_s[i][j] + at_stress_coul_recip[i][j];

   for (i = 1; i < 3; ++i)
      for (j = 0; j < i; ++j)
         at_stress[i][j] = at_stress_kin[j][i]
            + at_stress_vdw_s[j][i] 
            + at_stress_stretch[j][i]+ at_stress_bend[j][i] 
             + at_stress_coul_s[j][i] + at_stress_coul_recip[j][i];

   /* Calculate total molecular stress tensor. */
   for (i = 0; i < 3; ++i)
      for (j = 0; j < 3; ++j) {
         mol_stress_vdw[i][j] = mol_stress_vdw_s[i][j];
      }
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         mol_stress[i][j] = mol_stress_kin[i][j] + mol_stress_vdw[i][j]
                            + mol_stress_coul_s[i][j] + mol_stress_coul_recip[i][j];
   for (i = 1; i < 3; ++i)
      for (j = 0; j < i; ++j)
         mol_stress[i][j] = mol_stress_kin[j][i] + mol_stress_vdw[i][j]
                           + mol_stress_coul_s[i][j] + mol_stress_coul_recip[i][j];;

}

/****************************************************************************/
/* Compute contributions to hydrostatic atomic and molecular pressure. */
void compute_hydrostatic(double **at_stress_vdw_s, double **at_stress_coul_s,
                         double **at_stress_coul_recip, 
                         double **at_stress_stretch, double **at_stress_bend,
                         double **at_stress, double **at_stress_kin, 
                         double **mol_stress_vdw_s, double **mol_stress_coul_s,
                         double **mol_stress_coul_recip, double **mol_stress,
                         double **mol_stress_kin, double *at_press_vdw_s,
                         double *at_press_coul_s, 
                         double *at_press_coul_recip, 
                         double *at_press_stretch, double *at_press_bend,
                         double *at_press, double *at_press_kin,
                         double *mol_press_vdw_s, double *mol_press_coul_s,
                         double *mol_press_coul_recip, double *mol_press,
                         double *mol_press_kin)
{
int i;

       /* Compute atomic and molecular contributions to hydrostatic 
          pressure. */
       *at_press = *at_press_vdw_s = *at_press_coul_s = *at_press_coul_recip
           = *at_press_stretch = *at_press_bend = *at_press_kin = 0.0;

       *mol_press = *mol_press_vdw_s = *mol_press_coul_s 
                  = *mol_press_coul_recip = *mol_press_kin = 0.0;

        for (i = 0; i < 3; ++i) {
          *at_press += at_stress[i][i];
          *at_press_vdw_s += at_stress_vdw_s[i][i];
          *at_press_coul_s += at_stress_coul_s[i][i];
          *at_press_coul_recip += at_stress_coul_recip[i][i];
          *at_press_stretch += at_stress_stretch[i][i];
          *at_press_bend += at_stress_coul_recip[i][i];
          *at_press_kin += at_stress_kin[i][i];

          *mol_press += mol_stress[i][i];
          *mol_press_vdw_s += mol_stress_vdw_s[i][i];
          *mol_press_coul_s += mol_stress_coul_s[i][i];
          *mol_press_coul_recip += mol_stress_coul_recip[i][i];
          *mol_press_kin += mol_stress_kin[i][i];
        }

       /* Normalize and write the contribution to hydrostatic pressure
          to output file. */
       *at_press /= 3.0;
       *at_press_vdw_s /= 3.0;
       *at_press_coul_s /= 3.0;
       *at_press_coul_recip /= 3.0;
       *at_press_stretch /= 3.0;
       *at_press_bend /= 3.0;
       *at_press_kin /= 3.0;

       *mol_press /= 3.0;
       *mol_press_vdw_s /= 3.0;
       *mol_press_coul_s /= 3.0;
       *mol_press_coul_recip /= 3.0;
       *mol_press_kin /= 3.0;

}
/****************************************************************************/
/* Compute contributions to hydrostatic atomic and molecular pressure. */
void accu_block(int i_block, double **h, double volume, double rho, 
                double at_temp, double pe, double ke, double te,
                double pe_vdw_s, double pe_coul_s, double pe_coul_recip,
                double pe_stretch, double pe_bend,  
                double **at_stress, double **at_stress_vdw_s,
                double **at_stress_coul_s, double **at_stress_coul_recip,
                double **at_stress_stretch, double **at_stress_bend, 
                double **at_stress_kin,  
                double **mol_stress, double **mol_stress_vdw_s,
                double **mol_stress_coul_s, double **mol_stress_coul_recip,
                double **mol_stress_kin,
                double at_press, double at_press_vdw_s, double at_press_coul_s, 
                double at_press_coul_recip, double at_press_stretch, 
                double at_press_bend, double at_press_kin, double mol_press,
                double mol_press_vdw_s, double mol_press_coul_s,
                double mol_press_coul_recip, double mol_press_kin, 
                double **h_block, double *vol_block, double *rho_block,
                double *at_temp_block, double *pe_block, double *ke_block,
                double *te_block, double *pe_vdw_s_block, 
                double *pe_coul_s_block, double *pe_coul_recip_block, 
                double *pe_stretch_block, double *pe_bend_block,
                double ***at_stress_block, double ***at_stress_vdw_s_block,
                double ***at_stress_coul_s_block, 
                double ***at_stress_coul_recip_block,
                double ***at_stress_stretch_block, 
                double ***at_stress_bend_block, double ***at_stress_kin_block, 
                double ***mol_stress_block, double ***mol_stress_vdw_s_block,
                double ***mol_stress_coul_s_block,
                double ***mol_stress_coul_recip_block,
                double ***mol_stress_kin_block,
                double *at_press_block, double *at_press_vdw_s_block, 
                double *at_press_coul_s_block, 
                double *at_press_coul_recip_block, 
                double *at_press_stretch_block, double *at_press_bend_block,
                double *at_press_kin_block, double *mol_press_block,
                double *mol_press_vdw_s_block, double *mol_press_coul_s_block,
                double *mol_press_coul_recip_block, 
                double *mol_press_kin_block)
{
int i, j;

    /* Add contributions to block accumulators. */
    for (i = 0; i < 3; ++i)
      h_block[i][i_block] += h[i][i];

    vol_block[i_block] += volume;
    rho_block[i_block] += rho;

    at_temp_block[i_block] += at_temp;

    pe_block[i_block] += pe;
    ke_block[i_block] += ke;
    te_block[i_block] += te;
   
    pe_vdw_s_block[i_block] += pe_vdw_s;
    pe_coul_s_block[i_block] += pe_coul_s;
    pe_coul_recip_block[i_block] += pe_coul_recip;
    pe_stretch_block[i_block] += pe_stretch;
    pe_bend_block[i_block] += pe_bend;

    for (i = 0; i < 3; ++i)
       for (j = i; j < 3; ++j){
          at_stress_block[i][j][i_block] += at_stress[i][j];
          at_stress_vdw_s_block[i][j][i_block] += at_stress_vdw_s[i][j];
          at_stress_coul_s_block[i][j][i_block] += at_stress_coul_s[i][j];
          at_stress_coul_recip_block[i][j][i_block] += 
                                                    at_stress_coul_recip[i][j];
          at_stress_stretch_block[i][j][i_block] += at_stress_stretch[i][j];
          at_stress_bend_block[i][j][i_block] += at_stress_bend[i][j];
          at_stress_kin_block[i][j][i_block] += at_stress_kin[i][j];

          mol_stress_block[i][j][i_block] += mol_stress[i][j];
          mol_stress_vdw_s_block[i][j][i_block] += mol_stress_vdw_s[i][j];
          mol_stress_coul_s_block[i][j][i_block] += mol_stress_coul_s[i][j];
          mol_stress_coul_recip_block[i][j][i_block] += 
                                                  mol_stress_coul_recip[i][j];
          mol_stress_kin_block[i][j][i_block] += mol_stress_kin[i][j];
       }

    at_press_block[i_block] += at_press;
    at_press_vdw_s_block[i_block] += at_press_vdw_s;
    at_press_coul_s_block[i_block] += at_press_coul_s;
    at_press_coul_recip_block[i_block] += at_press_coul_recip;
    at_press_stretch_block[i_block] += at_press_stretch;
    at_press_bend_block[i_block] += at_press_bend;
    at_press_kin_block[i_block] += at_press_kin;

    mol_press_block[i_block] += mol_press;
    mol_press_vdw_s_block[i_block] += mol_press_vdw_s;
    mol_press_coul_s_block[i_block] += mol_press_coul_s;
    mol_press_coul_recip_block[i_block] += mol_press_coul_recip;
    mol_press_kin_block[i_block] += mol_press_kin;

} 
/****************************************************************************/
/* Normalize block average and write to the standard output. */
void write_block(int i_block, int n_block, 
                double **h_block, double *vol_block, double *rho_block,
                double *at_temp_block, double *pe_block, double *ke_block,
                double *te_block, double *pe_vdw_s_block, 
                double *pe_coul_s_block, double *pe_coul_recip_block, 
                double *pe_stretch_block, double *pe_bend_block,
                double ***at_stress_block, double ***at_stress_vdw_s_block,
                double ***at_stress_coul_s_block, 
                double ***at_stress_coul_recip_block,
                double ***at_stress_stretch_block, 
                double ***at_stress_bend_block, double ***at_stress_kin_block, 
                double ***mol_stress_block, double ***mol_stress_vdw_s_block,
                double ***mol_stress_coul_s_block,
                double ***mol_stress_coul_recip_block,
                double ***mol_stress_kin_block,
                double *at_press_block, double *at_press_vdw_s_block, 
                double *at_press_coul_s_block, 
                double *at_press_coul_recip_block, 
                double *at_press_stretch_block, double *at_press_bend_block,
                double *at_press_kin_block, double *mol_press_block,
                double *mol_press_vdw_s_block, double *mol_press_coul_s_block,
                double *mol_press_coul_recip_block, 
                double *mol_press_kin_block)
{
int i, j;
double norm;

    /* Compute normalization. */
    norm = 1.0 / n_block;

    /* Add contributions to block accumulators. */
    for (i = 0; i < 3; ++i)
      h_block[i][i_block] *= norm;

    vol_block[i_block] *= norm;
    rho_block[i_block] *= norm;

    at_temp_block[i_block] *= norm;

    pe_block[i_block] *= norm;
    ke_block[i_block] *= norm;
    te_block[i_block] *= norm;
   
    pe_vdw_s_block[i_block] *= norm;
    pe_coul_s_block[i_block] *= norm;
    pe_coul_recip_block[i_block] *= norm;
    pe_stretch_block[i_block] *= norm;
    pe_bend_block[i_block] *= norm;

    for (i = 0; i < 3; ++i)
       for (j = i; j < 3; ++j){
          at_stress_block[i][j][i_block] *= norm;
          at_stress_vdw_s_block[i][j][i_block] *= norm;
          at_stress_coul_s_block[i][j][i_block] *= norm;
          at_stress_coul_recip_block[i][j][i_block] *= norm; 
          at_stress_stretch_block[i][j][i_block] *= norm;
          at_stress_bend_block[i][j][i_block] *= norm;
          at_stress_kin_block[i][j][i_block] *= norm;

          mol_stress_block[i][j][i_block] *= norm;
          mol_stress_vdw_s_block[i][j][i_block] *= norm;
          mol_stress_coul_s_block[i][j][i_block] *= norm;
          mol_stress_coul_recip_block[i][j][i_block] *= norm;
          mol_stress_kin_block[i][j][i_block] *= norm;
       }

    at_press_block[i_block] *= norm;
    at_press_vdw_s_block[i_block] *= norm;
    at_press_coul_s_block[i_block] *= norm;
    at_press_coul_recip_block[i_block] *= norm;
    at_press_stretch_block[i_block] *= norm;
    at_press_bend_block[i_block] *= norm;
    at_press_kin_block[i_block] *= norm;

    mol_press_block[i_block] *= norm;
    mol_press_vdw_s_block[i_block] *= norm;
    mol_press_coul_s_block[i_block] *= norm;
    mol_press_coul_recip_block[i_block] *= norm;
    mol_press_kin_block[i_block] *= norm;

    /* Print block averages to standard output. */
    printf("\n-----------------------------------------------------\n");
    printf("\nBlock averages for block %d, MD timesteps %d - %d\n",
          i_block, i_block * n_block + 1, (i_block + 1) * n_block);
    printf("\n-----------------------------------------------------\n");
      for (i = 0; i < 3; ++i)
        printf("   h[%d][%d] = %g\n", i, i, h_block[i][i_block]);
      
      printf("\n");

      printf("   volume = %g\n", vol_block[i_block]);
      printf("   rho = %g\n\n", rho_block[i_block]);

      printf("   at_temp = %g\n\n", at_temp_block[i_block]);

      printf("   pe = %g\n", pe_block[i_block]);
      printf("   ke = %g\n", ke_block[i_block]);
      printf("   te = %g\n\n", te_block[i_block]);

      printf("      pe_vdw_s = %g\n", pe_vdw_s_block[i_block]);
      printf("      pe_coul_s = %g\n", pe_coul_s_block[i_block]);
      printf("      pe_coul_recip = %g\n", pe_coul_recip_block[i_block]);
      printf("      pe_stretch = %g\n", pe_stretch_block[i_block]);
      printf("      pe_bend = %g\n\n", pe_bend_block[i_block]);

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("   at_stress[%d][%d] = %g\n", i, j,
                                       at_stress_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_vdw_s[%d][%d] = %g\n", i, j,
                                       at_stress_vdw_s_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_coul_s[%d][%d] = %g\n", i, j,
                                       at_stress_coul_s_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_coul_recip[%d][%d] = %g\n", i, j,
                                    at_stress_coul_recip_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_stretch[%d][%d] = %g\n", i, j,
                                    at_stress_stretch_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_bend[%d][%d] = %g\n", i, j,
                                    at_stress_bend_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_kin[%d][%d] = %g\n", i, j,
                                    at_stress_kin_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("   mol_stress[%d][%d] = %g\n", i, j,
                                       mol_stress_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_vdw_s[%d][%d] = %g\n", i, j,
                                       mol_stress_vdw_s_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_coul_s[%d][%d] = %g\n", i, j,
                                       mol_stress_coul_s_block[i][j][i_block]);
      printf("\n");
      
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_coul_recip[%d][%d] = %g\n", i, j,
                                   mol_stress_coul_recip_block[i][j][i_block]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_kin[%d][%d] = %g\n", i, j,
                                   mol_stress_kin_block[i][j][i_block]);
      printf("\n");

      printf("   at_press = %g\n", at_press_block[i_block]);
      printf("      at_press_vdw_s = %g\n", at_press_vdw_s_block[i_block]);
      printf("      at_press_coul_s = %g\n", at_press_coul_s_block[i_block]);
      printf("      at_press_coul_recip = %g\n", 
                                      at_press_coul_recip_block[i_block]);
      printf("      at_press_stretch = %g\n", at_press_stretch_block[i_block]);
      printf("      at_press_bend = %g\n", at_press_bend_block[i_block]);
      printf("      at_press_kin = %g\n\n", at_press_kin_block[i_block]);

      printf("   mol_press = %g\n", mol_press_block[i_block]);
      printf("      mol_press_vdw_s = %g\n", mol_press_vdw_s_block[i_block]);
      printf("      mol_press_coul_s = %g\n", mol_press_coul_s_block[i_block]);
      printf("      mol_press_coul_recip = %g\n", 
                                       mol_press_coul_recip_block[i_block]);
      printf("      mol_press_kin = %g\n", mol_press_kin_block[i_block]);

      fflush(NULL);
} 
/****************************************************************************/
/* Compute and write to the standard output the run averages 
   and uncertainties. */
void write_run(int n_blocks, 
               double **h_block, double *vol_block, double *rho_block,
               double *at_temp_block, double *pe_block, double *ke_block,
               double *te_block, double *pe_vdw_s_block, 
               double *pe_coul_s_block, double *pe_coul_recip_block, 
               double *pe_stretch_block, double *pe_bend_block,
               double ***at_stress_block, double ***at_stress_vdw_s_block,
               double ***at_stress_coul_s_block, 
               double ***at_stress_coul_recip_block,
               double ***at_stress_stretch_block, 
               double ***at_stress_bend_block, double ***at_stress_kin_block, 
               double ***mol_stress_block, double ***mol_stress_vdw_s_block,
               double ***mol_stress_coul_s_block,
               double ***mol_stress_coul_recip_block, 
               double ***mol_stress_kin_block,
               double *at_press_block, double *at_press_vdw_s_block, 
               double *at_press_coul_s_block, 
               double *at_press_coul_recip_block, 
               double *at_press_stretch_block, double *at_press_bend_block,
               double *at_press_kin_block, double *mol_press_block,
               double *mol_press_vdw_s_block, double *mol_press_coul_s_block,
               double *mol_press_coul_recip_block, 
               double *mol_press_kin_block, double **p_tangent_block, double **p_normal_block,
               double **dens_prof_block, double *ift_block, 
               int n_slab, int ift_switch, double ***dens_prof_species_block, int n_species)

{

int i, j, i_bin;
double norm, d_bin, z_bin;
double *h_ave, *h_unc, vol_ave, vol_unc, rho_ave, rho_unc, at_temp_ave, 
       at_temp_unc, pe_ave, pe_unc, ke_ave, ke_unc, te_ave, te_unc,
       pe_vdw_s_ave, pe_vdw_s_unc, pe_coul_s_ave, pe_coul_s_unc, 
       pe_coul_recip_ave, pe_coul_recip_unc, pe_stretch_ave, pe_stretch_unc, 
       pe_bend_ave, pe_bend_unc, **at_stress_ave, **at_stress_unc, 
       **at_stress_vdw_s_ave, **at_stress_vdw_s_unc, **at_stress_coul_s_ave, 
       **at_stress_coul_s_unc, **at_stress_coul_recip_ave, 
       **at_stress_coul_recip_unc, **at_stress_stretch_ave, 
       **at_stress_stretch_unc, **at_stress_bend_ave, **at_stress_bend_unc,
       **at_stress_kin_ave, **at_stress_kin_unc, **mol_stress_ave, 
       **mol_stress_unc, **mol_stress_vdw_s_ave, **mol_stress_vdw_s_unc,
       **mol_stress_coul_s_ave, **mol_stress_coul_s_unc, 
       **mol_stress_coul_recip_ave, **mol_stress_coul_recip_unc, 
       **mol_stress_kin_ave, **mol_stress_kin_unc, 
       at_press_ave, at_press_unc, at_press_vdw_s_ave, at_press_vdw_s_unc, 
       at_press_coul_s_ave, at_press_coul_s_unc, at_press_coul_recip_ave, 
       at_press_coul_recip_unc, at_press_stretch_ave, at_press_stretch_unc, 
       at_press_bend_ave, at_press_bend_unc, at_press_kin_ave, at_press_kin_unc, 
       mol_press_ave, mol_press_unc, mol_press_vdw_s_ave, mol_press_vdw_s_unc, 
       mol_press_coul_s_ave, mol_press_coul_s_unc, mol_press_coul_recip_ave, 
       mol_press_coul_recip_unc, mol_press_kin_ave, mol_press_kin_unc;


double *p_tangent_ave, *p_normal_ave, *dens_prof_ave, 
       *p_tangent_unc, *p_normal_unc, *dens_prof_unc, ift_ave, ift_unc, ift_run_average = 0.0; 
double **dens_prof_species_ave, **dens_prof_species_unc;
int i_species;

FILE *fp;

  /* Allocate memory for run average accumulators. */
  h_ave = (double *)allocate_1d_array(3, sizeof(double));
  h_unc = (double *)allocate_1d_array(3, sizeof(double));
  at_stress_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_vdw_s_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_vdw_s_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_coul_s_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_coul_s_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_coul_recip_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_coul_recip_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_stretch_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_stretch_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_bend_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_bend_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_kin_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  at_stress_kin_unc = (double **)allocate_2d_array(3, 3, sizeof(double));

  mol_stress_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_vdw_s_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_vdw_s_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_coul_s_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_coul_s_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_coul_recip_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_coul_recip_unc = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_kin_ave = (double **)allocate_2d_array(3, 3, sizeof(double));
  mol_stress_kin_unc = (double **)allocate_2d_array(3, 3, sizeof(double));

 /* Allocate memory for stress profile and ift calculation */
  p_tangent_ave = (double *) allocate_1d_array(n_slab, sizeof(double));
  p_normal_ave = (double *) allocate_1d_array(n_slab, sizeof(double));
  dens_prof_ave = (double *) allocate_1d_array(n_slab, sizeof(double));
  p_tangent_unc = (double *) allocate_1d_array(n_slab, sizeof(double));
  p_normal_unc = (double *) allocate_1d_array(n_slab, sizeof(double));
  dens_prof_unc = (double *) allocate_1d_array(n_slab, sizeof(double));
  dens_prof_species_ave = (double **) allocate_2d_array(n_species, n_slab, sizeof(double));
  dens_prof_species_unc = (double **) allocate_2d_array(n_species, n_slab, sizeof(double));

    
  /* Calculate run averages and uncertainties. */
  for (i = 0; i < 3; ++i)
    statistics(n_blocks, h_block[i], &h_ave[i], &h_unc[i]);

  statistics(n_blocks, vol_block, &vol_ave, &vol_unc);
  statistics(n_blocks, rho_block, &rho_ave, &rho_unc);
  statistics(n_blocks, at_temp_block, &at_temp_ave, &at_temp_unc);
  statistics(n_blocks, pe_block, &pe_ave, &pe_unc);
  statistics(n_blocks, ke_block, &ke_ave, &ke_unc);
  statistics(n_blocks, te_block, &te_ave, &te_unc);
  statistics(n_blocks, pe_vdw_s_block, &pe_vdw_s_ave, &pe_vdw_s_unc);
  statistics(n_blocks, pe_coul_s_block, &pe_coul_s_ave, &pe_coul_s_unc);
  statistics(n_blocks, pe_coul_recip_block, &pe_coul_recip_ave, 
             &pe_coul_recip_unc);
  statistics(n_blocks, pe_stretch_block, &pe_stretch_ave, &pe_stretch_unc);
  statistics(n_blocks, pe_bend_block, &pe_bend_ave, &pe_bend_unc);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, at_stress_block[i][j],
          &at_stress_ave[i][j], &at_stress_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, at_stress_vdw_s_block[i][j],
          &at_stress_vdw_s_ave[i][j], &at_stress_vdw_s_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, at_stress_coul_s_block[i][j],
          &at_stress_coul_s_ave[i][j], &at_stress_coul_s_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, at_stress_coul_recip_block[i][j],
          &at_stress_coul_recip_ave[i][j], &at_stress_coul_recip_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, at_stress_stretch_block[i][j],
          &at_stress_stretch_ave[i][j], &at_stress_stretch_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, at_stress_bend_block[i][j],
          &at_stress_bend_ave[i][j], &at_stress_bend_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, at_stress_kin_block[i][j],
          &at_stress_kin_ave[i][j], &at_stress_kin_unc[i][j]);

  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, mol_stress_block[i][j],
          &mol_stress_ave[i][j], &mol_stress_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, mol_stress_vdw_s_block[i][j],
          &mol_stress_vdw_s_ave[i][j], &mol_stress_vdw_s_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, mol_stress_coul_s_block[i][j],
          &mol_stress_coul_s_ave[i][j], &mol_stress_coul_s_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, mol_stress_coul_recip_block[i][j],
          &mol_stress_coul_recip_ave[i][j], &mol_stress_coul_recip_unc[i][j]);
  for (i = 0; i < 3; ++i)
    for (j = 0; j < 3; ++j)
      statistics(n_blocks, mol_stress_kin_block[i][j],
          &mol_stress_kin_ave[i][j], &mol_stress_kin_unc[i][j]);

  statistics(n_blocks, at_press_block, &at_press_ave, &at_press_unc);
  statistics(n_blocks, at_press_vdw_s_block, &at_press_vdw_s_ave, 
             &at_press_vdw_s_unc);
  statistics(n_blocks, at_press_coul_s_block, &at_press_coul_s_ave, 
             &at_press_coul_s_unc);
  statistics(n_blocks, at_press_coul_recip_block, &at_press_coul_recip_ave, 
             &at_press_coul_recip_unc);
  statistics(n_blocks, at_press_stretch_block, &at_press_stretch_ave, 
             &at_press_stretch_unc);
  statistics(n_blocks, at_press_bend_block, &at_press_bend_ave, 
             &at_press_bend_unc);
  statistics(n_blocks, at_press_kin_block, &at_press_kin_ave, 
             &at_press_kin_unc);

  statistics(n_blocks, mol_press_block, &mol_press_ave, &mol_press_unc);
  statistics(n_blocks, mol_press_vdw_s_block, &mol_press_vdw_s_ave, 
             &mol_press_vdw_s_unc);
  statistics(n_blocks, mol_press_coul_s_block, &mol_press_coul_s_ave, 
             &mol_press_coul_s_unc);
  statistics(n_blocks, mol_press_coul_recip_block, &mol_press_coul_recip_ave, 
             &mol_press_coul_recip_unc);
  statistics(n_blocks, mol_press_kin_block, &mol_press_kin_ave, 
             &mol_press_kin_unc);

  /* Calculate run averages and uncertainties for stress profile and ift. */

  if (ift_switch > 0){
  for (i = 0; i < n_slab; ++i){
      statistics(n_blocks, p_tangent_block[i], &p_tangent_ave[i], &p_tangent_unc[i]);
      statistics(n_blocks, p_normal_block[i], &p_normal_ave[i], &p_normal_unc[i]);
      statistics(n_blocks, dens_prof_block[i], &dens_prof_ave[i], &dens_prof_unc[i]);
   }
      statistics(n_blocks, ift_block, &ift_ave, &ift_unc);

   for (i_species=0; i_species <n_species; ++i_species)
       for (i = 0; i < n_slab; ++i)
       statistics(n_blocks, dens_prof_species_block[i_species][i], 
                   &dens_prof_species_ave[i_species][i], &dens_prof_species_unc[i_species][i]);
  }

    /* Print run averages to standard output. */
    printf("\n*****************************************************\n");
    printf("\n              Run averages\n");
    printf("\n*****************************************************\n");
      for (i = 0; i < 3; ++i)
        printf("   h[%d][%d] = %g +/- %g\n", i, i, h_ave[i], h_unc[i]);
      
      printf("\n");

      printf("   volume = %g +/- %g\n", vol_ave, vol_unc);
      printf("   rho = %g +/- %g\n\n", rho_ave, rho_unc);

      printf("   at_temp = %g +/- %g\n\n", at_temp_ave, at_temp_unc);

      printf("   pe = %g +/- %g\n", pe_ave, pe_unc);
      printf("   ke = %g +/- %g\n", ke_ave, ke_unc);
      printf("   te = %g +/- %g\n\n", te_ave, te_unc);

      printf("      pe_vdw_s = %g +/- %g\n", pe_vdw_s_ave, pe_vdw_s_unc);
      printf("      pe_coul_s = %g +/- %g\n", pe_coul_s_ave, pe_coul_s_unc);
      printf("      pe_coul_recip = %g +/- %g\n", pe_coul_recip_ave, 
             pe_coul_recip_unc);
      printf("      pe_stretch = %g +/- %g\n", pe_stretch_ave, pe_stretch_unc);
      printf("      pe_bend = %g +/- %g\n\n", pe_bend_ave, pe_bend_unc);

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("   at_stress[%d][%d] = %g +/- %g\n", i, j,
                   at_stress_ave[i][j], at_stress_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_vdw_s[%d][%d] = %g +/- %g\n", i, j,
                   at_stress_vdw_s_ave[i][j], at_stress_vdw_s_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_coul_s[%d][%d] = %g +/- %g\n", i, j,
                   at_stress_coul_s_ave[i][j], at_stress_coul_s_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_coul_recip[%d][%d] = %g +/- %g\n", i, j,
                   at_stress_coul_recip_ave[i][j], 
                   at_stress_coul_recip_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_stretch[%d][%d] = %g +/- %g\n", i, j,
                   at_stress_stretch_ave[i][j], at_stress_stretch_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_bend[%d][%d] = %g +/- %g\n", i, j,
                   at_stress_bend_ave[i][j], at_stress_bend_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      at_stress_kin[%d][%d] = %g +/- %g\n", i, j,
                   at_stress_kin_ave[i][j], at_stress_kin_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("   mol_stress[%d][%d] = %g +/- %g\n", i, j,
                   mol_stress_ave[i][j], mol_stress_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_vdw_s[%d][%d] = %g +/- %g\n", i, j,
                   mol_stress_vdw_s_ave[i][j], mol_stress_vdw_s_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_coul_s[%d][%d] = %g +/- %g\n", i, j,
                   mol_stress_coul_s_ave[i][j], mol_stress_coul_s_unc[i][j]);
      printf("\n");
      
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_coul_recip[%d][%d] = %g +/- %g\n", i, j,
                   mol_stress_coul_recip_ave[i][j], 
                   mol_stress_coul_recip_unc[i][j]);
      printf("\n");

      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            printf("      mol_stress_kin[%d][%d] = %g +/- %g\n", i, j,
                   mol_stress_kin_ave[i][j], mol_stress_kin_unc[i][j]);
      printf("\n");

      printf("   at_press = %g +/- %g\n", at_press_ave, at_press_unc);
      printf("      at_press_vdw_s = %g +/- %g\n", at_press_vdw_s_ave, 
             at_press_vdw_s_unc);
      printf("      at_press_coul_s = %g +/- %g\n", at_press_coul_s_ave,
             at_press_coul_s_unc);
      printf("      at_press_coul_recip = %g +/- %g\n", 
             at_press_coul_recip_ave, at_press_coul_recip_unc);
      printf("      at_press_stretch = %g +/- %g\n", at_press_stretch_ave, 
             at_press_stretch_unc);
      printf("      at_press_bend = %g +/- %g\n", at_press_bend_ave, 
             at_press_bend_unc);
      printf("      at_press_kin = %g +/- %g\n\n", at_press_kin_ave, 
             at_press_kin_unc);

      printf("   mol_press = %g +/- %g\n", mol_press_ave, mol_press_unc);
      printf("      mol_press_vdw_s = %g +/- %g\n", mol_press_vdw_s_ave,
             mol_press_vdw_s_unc);
      printf("      mol_press_coul_s = %g +/- %g\n", mol_press_coul_s_ave,
             mol_press_coul_s_unc);
      printf("      mol_press_coul_recip = %g +/- %g\n", 
             mol_press_coul_recip_ave, mol_press_coul_recip_unc);
      printf("      mol_press_kin = %g +/- %g\n", mol_press_kin_ave, 
             mol_press_kin_unc);

     /* Write out the stress profile and surface tension */
        if (ift_switch >0 ) {
             /* Open file */
        fp = fopen("p_prof.dat", "w");
        if (fp == NULL)
        {
                printf("Unable to open p_prof.dat file. \n");
                exit(0);
        }
       fprintf(fp,"#z\t\tden_ave\t\tden_unc\t\tp_tan_ave\tp_tan_unc\tp_norm_ave\tp_norm_unc\n");
       d_bin = h_ave[2]/n_slab;
       for (i_bin = 0; i_bin < n_slab; ++i_bin){
            z_bin = i_bin * d_bin;
    /* Calculate ift from run-averaged profiles */
           ift_run_average += (p_normal_ave[i_bin] - p_tangent_ave[i_bin])*d_bin;
       fprintf(fp, "%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\n", z_bin, dens_prof_ave[i_bin], dens_prof_unc[i_bin],
                    p_tangent_ave[i_bin], p_tangent_unc[i_bin], p_normal_ave[i_bin],
                     p_normal_unc[i_bin]);
       }
       ift_run_average = ift_run_average/2.0;
       fprintf(fp, "#Surface tension: %lf +/- %lf\n", ift_ave, ift_unc);
       fprintf(fp, "#Surface tension from run averaged profile: %lf\n", ift_run_average);
       fclose(fp);
        fp = fopen("dens_prof.dat", "w");
        if (fp == NULL)
        {
                printf("Unable to open dens_prof.dat file. \n");
                exit(0);
        }

          for (i_bin = 0; i_bin < n_slab; ++i_bin){
            z_bin = i_bin * d_bin;
            if (n_species==2)
            fprintf(fp, "%lf\t%lf\t%lf\t%lf\t%lf\n", z_bin, 
                    dens_prof_species_ave[0][i_bin], dens_prof_species_unc[0][i_bin],
                    dens_prof_species_ave[1][i_bin], dens_prof_species_unc[1][i_bin]);
           if (n_species==3)
            fprintf(fp, "%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\n", z_bin,
                    dens_prof_species_ave[0][i_bin], dens_prof_species_unc[0][i_bin],
                    dens_prof_species_ave[1][i_bin], dens_prof_species_unc[1][i_bin],
                    dens_prof_species_ave[2][i_bin], dens_prof_species_unc[2][i_bin]);
          }
      fclose(fp);
    }

      fflush(NULL);
} 
