/* Periodic boundary condition nonbonded force routines for molecular
   modeling package, with LJ form for vdW interactions. */

#include "build.h"

/* Calculate short-range LJ interactions using a N^2 search, and using a
   switching function to truncate the pair potential and pair force smoothly. */

void lj_s_period(int *mol_species, int *atom_rel, int *atom_mol, 
   int *atom_type, int n_atoms, 
   int *n_atoms_per_mol, int ***exclusions, int **comb_pot, 
   double ***comb_par, double **h, double r2_on, double r2_off, 
   double **scaled_atom_coords, double **scaled_mol_coords, 
   double **rel_atom_coords, double gamma, double three_gamma_over_two, 
   double two_over_gamma_cubed, double *pe_vdw_s, double **f_vdw_s, 
   double **mol_stress_vdw_s)
{
   int i, j, k, i_species, j_species, i_mol, j_mol, i_rel, j_rel, inter_flag,
      i_type, j_type, pot_form, compute_flag;
   double s_sep[NDIM], sep[NDIM], r2_sep, u_vdw, f_vdw, f_x, f_y, f_z, 
      s_i[NDIM], f_i[NDIM], s_mol_i[NDIM], s_sep_mol[NDIM], sep_mol[NDIM],
      rel_dist[NDIM];

   /* Initialisation. */
   *pe_vdw_s = 0.0;
   
   for (i = 0; i < n_atoms; ++i)
      for (k = 0; k < NDIM; ++k)
         f_vdw_s[i][k] = 0.0;
   
   for (i = 0; i < NDIM; ++i)
      for (k = 0; k < NDIM; ++k)
         mol_stress_vdw_s[i][k] = 0.0;

   /* Loop over atoms. */
   for (i = 0; i < n_atoms - 1; ++i) {

      /* Get attributes of atom i. */
      i_mol = atom_mol[i];
      i_rel = atom_rel[i];
      i_type = atom_type[i];
      i_species = mol_species[i_mol];
      for (k = 0; k < NDIM; ++k) {
        s_mol_i[k] = scaled_mol_coords[i_mol][k];
        s_i[k] = scaled_atom_coords[i][k];
        f_i[k] = f_vdw_s[i][k];
        }

      /* Loop over atoms. */ 
      for (j = i + 1; j < n_atoms; ++j) {

         /* Get attributes of atom j. */
         j_mol = atom_mol[j];
         j_rel = atom_rel[j];
         j_type = atom_type[j];

         /* Determine whether interaction should be computed. */
         inter_flag = i_mol != j_mol;
         if (inter_flag || exclusions[i_species][i_rel][j_rel]) {

               /* Calculate scaled pair separation. */
               for (k = 0; k < NDIM; ++k) {
                 s_sep[k] = scaled_atom_coords[j][k] - s_i[k];
                 s_sep[k] -= NINT(s_sep[k]);
                 }
               sep[0] = h[0][0] * s_sep[0] + h[0][1] * s_sep[1]
                        + h[0][2] * s_sep[2];
               sep[1] = h[1][1] * s_sep[1] + h[1][2] * s_sep[2];
               sep[2] = h[2][2] * s_sep[2];
               r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

               /* If the pair separation is less than r_off, calculate
                  vdW interaction. */

               /* Get the pair interaction form. */
               pot_form = comb_pot[i_type][j_type];
               compute_flag = 0;

/****************************************************************************/
/********************** Begin of pair interaction menu. *********************/
/****************************************************************************/
               /* Compute pair interaction. */
               compute_pot(i_type, j_type, pot_form, r2_off, r2_on, r2_sep,
                  comb_par, gamma, three_gamma_over_two, two_over_gamma_cubed,
                  &u_vdw, &f_vdw, &compute_flag);

               /* Add contributions to potential energy. */
               if (compute_flag) {
               *pe_vdw_s += u_vdw;

               /* Add contributions to forces. */
               f_x = f_vdw * sep[0];
               f_y = f_vdw * sep[1];
               f_z = f_vdw * sep[2];
               f_i[0] -= f_x;
               f_i[1] -= f_y;
               f_i[2] -= f_z;
               f_vdw_s[j][0] += f_x;
               f_vdw_s[j][1] += f_y;
               f_vdw_s[j][2] += f_z;
                  
               /* Compute stress tensor. */
               mol_stress_vdw_s[0][0] += sep[0] * f_x;
               mol_stress_vdw_s[1][1] += sep[1] * f_y;
               mol_stress_vdw_s[2][2] += sep[2] * f_z;
               mol_stress_vdw_s[0][1] += sep[0] * f_y;
               mol_stress_vdw_s[0][2] += sep[0] * f_z;
               mol_stress_vdw_s[1][2] += sep[1] * f_z;

               /* Add contribution coming from the intramolecular
                  rigidity constraint */
               for ( k =0; k<NDIM; ++k)
                  rel_dist[k] = (rel_atom_coords[j][k] -
                                                 rel_atom_coords[i][k]);

               mol_stress_vdw_s[0][0] -= rel_dist[0] * f_x;
               mol_stress_vdw_s[1][1] -= rel_dist[1] * f_y;
               mol_stress_vdw_s[2][2] -= rel_dist[2] * f_z;
               mol_stress_vdw_s[0][1] -= rel_dist[0] * f_y;
               mol_stress_vdw_s[0][2] -= rel_dist[0] * f_z;
               mol_stress_vdw_s[1][2] -= rel_dist[1] * f_z;
               }

               }
            }
        f_vdw_s[i][0] = f_i[0];
        f_vdw_s[i][1] = f_i[1];
        f_vdw_s[i][2] = f_i[2];
	}
/* for (i = 0; i < n_atoms; ++i)
    printf("i %d : %g %g %g\n", i, f_vdw_s[i][0], f_vdw_s[i][1], f_vdw_s[i][2]);
*/
}
/****************************************************************************/
/* Calculate short-range LJ interactions between the molecule i_mol and all the
others, using a N^2 search, and using a switching function to truncate the pair
potential and pair force smoothly. */

void lj_s_period_single(int skip, int mol, int *mol_species, int *atom_rel, 
   int *atom_mol, int *atom_type, int n_atoms, int *n_atoms_per_mol, 
   int ***exclusions, int **comb_pot, double ***comb_par, double **h, 
   double r2_on, double r2_off, double **scaled_atom_coords, double gamma, 
   double three_gamma_over_two, double two_over_gamma_cubed, 
   double *pe_vdw_s_single)
{
   int i, j, k, i_species, j_species, i_mol, j_mol, i_rel, j_rel, inter_flag,
      i_type, j_type, pot_form, compute_flag;
   double s_sep[NDIM], sep[NDIM], r2_sep, u_vdw, f_vdw, s_i[NDIM];

   /* Initialisation. */
   *pe_vdw_s_single = 0.0;

   i_species = mol_species[mol];
   /* Loop over atoms in the molecule i_mol. */
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {

      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k)
        s_i[k] = scaled_atom_coords[i][k];
      i_mol = atom_mol[i];
      i_rel = atom_rel[i];
      i_type = atom_type[i];

      /* Loop over atoms. */ 
      for (j = 0; j < n_atoms; ++j) {

         /* Get attributes of atom j. */
         j_mol = atom_mol[j];
         j_rel = atom_rel[j];
         j_type = atom_type[j];

         /* Determine whether interaction should be computed. */
         inter_flag = i_mol != j_mol;
         if (inter_flag || exclusions[i_species][i_rel][j_rel]) {

               /* Calculate scaled pair separation. */
               for (k = 0; k < NDIM; ++k) {
                 s_sep[k] = scaled_atom_coords[j][k] - s_i[k];
                 s_sep[k] -= NINT(s_sep[k]);
                 }
               sep[0] = h[0][0] * s_sep[0] + h[0][1] * s_sep[1]
                        + h[0][2] * s_sep[2];
               sep[1] = h[1][1] * s_sep[1] + h[1][2] * s_sep[2];
               sep[2] = h[2][2] * s_sep[2];
               r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

               /* Get the pair interaction form. */
               pot_form = comb_pot[i_type][j_type];
               compute_flag = 0;

/****************************************************************************/
/********************** Begin of pair interaction menu. *********************/
/****************************************************************************/
               /* Compute pair interaction. */
               compute_pot(i_type, j_type, pot_form, r2_off, r2_on, r2_sep,
                  comb_par, gamma, three_gamma_over_two, two_over_gamma_cubed,
                  &u_vdw, &f_vdw, &compute_flag);

               /* Add contribution to potential energy. */
               if (compute_flag)
                 *pe_vdw_s_single += u_vdw;

               }
            }
	}
}
/****************************************************************************/
