/* Neighbor list routines for molecular modeling package. */

#include "build.h"
#include "ff.h"
#include "proto.h"

void neigh_check(int i_species, int *n_atoms_per_mol, double **h, 
  double r_off, double r_nl, int skip, double **scaled_atom_coords,
  double **s_old, double **atom_move, int *flag_nl)
{
int i, k, abs;
double sep[NDIM], s_sep[NDIM], r2, r, skin;

skin = (r_nl - r_off) / 2.0;
for (i = 0; i < n_atoms_per_mol[i_species]; ++i)
  {
  abs = i + skip;
  for (k = 0; k < NDIM; ++k) {
    s_sep[k] = s_old[i][k] - scaled_atom_coords[abs][k];
    s_sep[k] -= NINT(s_sep[k]);
    }
  sep[0] = h[0][0] * s_sep[0] + h[0][1] * s_sep[1] + h[0][2] * s_sep[2];
  sep[1] = h[1][1] * s_sep[1] + h[1][2] * s_sep[2];
  sep[2] = h[2][2] * s_sep[2];
  atom_move[abs][0] += sep[0]; 
  atom_move[abs][1] += sep[1]; 
  atom_move[abs][2] += sep[2]; 
  r2 = SQR(atom_move[abs][0]) + SQR(atom_move[abs][1]) + SQR(atom_move[abs][2]);
  r = sqrt(r2);
  if (r >= skin) 
     *flag_nl = 1;
  }
}
/****************************************************************************/
/* Check whether neighbor lists need to be updated. */
void neighbor_check(int *mol_species, int *atom_rel, int *atom_mol, 
   int n_atoms, int *n_atoms_per_mol, int ***exclusions,
   double **h, double r2_on, double r2_off, double **scaled_atom_coords, 
   int period_switch, int *first, int *last, phantom_cell *phantoms,
   int noffsets, int *nc_p, int kc, int *phantom_skip, int *nc,
   atom_cell *atom_cells, int *offsets, double **atom_coords, 
   double **atom_move,
   nl_entry **nl_head, nl_entry **nl_tail, double r2_nl)
 
{
   int i, j;
   double r2_move, dr, test2_nl;
   int n_updates;
   double skin;
   skin = 0.5;
   
   /* Loop over atoms. */
   for (i = 0; i < n_atoms; ++i) {

       for (j = 0; j < NDIM; ++j) {
         dr += atom_move[i][j] ;
        }
        r2_move = SQR(dr);
/* printf(" In neigh. chek r2_move %g\n", r2_move); */

      /* If any atom has moved more than half the skin thickness, update the
         neighbor lists. */

      if (r2_move >= skin) {
         if (period_switch)
            neighbor_lists_period( mol_species, atom_rel, atom_mol,  n_atoms,
               n_atoms_per_mol, exclusions, h, r2_on,  r2_off,
               scaled_atom_coords, period_switch, first, last, phantoms,
               noffsets, nc_p,  kc, phantom_skip, nc, atom_cells, offsets, 
               atom_coords, atom_move, nl_head, nl_tail, r2_nl);
         ++n_updates;
         break;
      }
   }
}

/*****************************************************************************/
/* Construct neighbor lists, with periodic_boundary conditions. */
void neighbor_lists_period(int *mol_species, int *atom_rel, int *atom_mol, 
   int n_atoms, int *n_atoms_per_mol, int ***exclusions, double **h, 
   double r2_on, double r2_off, double **scaled_atom_coords, int period_switch,
   int *first, int *last, phantom_cell *phantoms, int noffsets, int *nc_p, 
   int kc, int *phantom_skip, int *nc, atom_cell *atom_cells, int *offsets, 
   double **atom_coords, double **atom_move, nl_entry **nl_head, 
   nl_entry **nl_tail, double r2_nl)
                            
{
   int i, j, k, j_real, ic, jc, i_off, i_species, i_mol, j_mol, i_rel, j_rel, 
      inter_flag;
   double sep[NDIM], r2_sep, s_i[NDIM];
   nl_entry *p;
   
   /* Purge neighbor lists. */
   for (i = 0; i < n_atoms; ++i)
      nl_tail[i] = NULL;

   /* Update cell lists. */
   update_cells(period_switch, n_atoms, h, scaled_atom_coords, atom_coords, 
     atom_cells, first, last, phantoms, kc, phantom_skip, nc, nc_p);

   /* Update positions of phantom atoms. */
   update_phantoms(n_atoms, atom_cells, phantom_skip, phantoms, atom_coords);

   /* Loop over central atoms. */
   for (i = 0; i < n_atoms; ++i) {

      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k)
         s_i[k] = atom_coords[i][k];

      i_mol = atom_mol[i];
      i_rel = atom_rel[i];
      i_species = mol_species[i_mol];

      /* Get cell index. */
      ic = atom_cells[i].cell;

      for (i_off = 0; i_off < noffsets; ++i_off) {

         /* Compute index of neighboring cell and get index of the
            first atom in that cell. */
         jc = ic + offsets[i_off];
         j = first[jc];

         /* Loop over atoms in neighboring cell. */
         while (j != -1) {

            /* Determine whether entry should be added to neighbor list. */
            j_real = atom_cells[j].real;
            j_mol = atom_mol[j_real];
            j_rel = atom_rel[j_real];
            inter_flag = i_mol != j_mol;
            if (inter_flag || exclusions[i_species][i_rel][j_rel]) {

               /* Calculate pair separation. */

               for (k = 0; k < NDIM; ++k)
                 sep[k] = atom_coords[j][k] - s_i[k];

                r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

               /* If the pair separation is less than r_nl, add entries
                  to neighbor lists. */

               if (r2_sep < r2_nl) {
                  if (nl_tail[i] == NULL){
                     p = nl_head[i];
                  } else {
                     p = nl_tail[i] -> next;
                     if (p == NULL) {
                           p = gmalloc(sizeof(nl_entry)); 
                        p -> next = NULL;
                       nl_tail[i] -> next = p ;
                     }
                  }
                  p -> atom = j_real;
                  nl_tail[i] = p;
               }
            }
            j = atom_cells[j].next;
         }
      }
   }
   /* Zero movement accumulators. */
/*   zero_move(n_atoms, atom_move);  */
}

/*****************************************************************************/
/* Construct neighbor lists, with periodic_boundary conditions. */
void neighbor_lists_period_single(int mol, int *mol_species, int *atom_rel, 
   int *atom_mol, int *mol_first_atm, 
   int n_atoms, int *n_atoms_per_mol, int ***exclusions, double **h, 
   double r2_on, double r2_off, double **scaled_atom_coords, int period_switch,
   int *first, int *last, phantom_cell *phantoms, int noffsets, int *nc_p, 
   int kc, int *phantom_skip, int *nc, atom_cell *atom_cells, int *offsets, 
   double **atom_coords, double **atom_move, nl_entry **nl_head, 
   nl_entry **nl_tail, double r2_nl)
                            
{
   int i, j, k, j_real, ic, jc, i_off, i_species, i_mol, j_mol, i_rel, j_rel, 
      inter_flag;
   double sep[NDIM], r2_sep, s_i[NDIM];
   nl_entry *p;
   
   /* Purge neighbor lists. */
   for (i = 0; i < n_atoms; ++i)
      nl_tail[i] = NULL;

  /* Update cell lists. */
   update_cells_single( mol, period_switch,  n_atoms, h, scaled_atom_coords, 
     atom_coords, atom_cells, first, last, phantoms, kc, phantom_skip, nc, 
     nc_p, n_atoms_per_mol, mol_species, mol_first_atm);

   update_phantoms_single( mol, n_atoms, atom_cells, phantom_skip, phantoms,
     atom_coords, mol_first_atm, mol_species, n_atoms_per_mol);

   /* Loop over central atoms. */
   for (i = 0; i < n_atoms; ++i) {

      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k)
         s_i[k] = atom_coords[i][k];

      i_mol = atom_mol[i];
      i_rel = atom_rel[i];
      i_species = mol_species[i_mol];

      /* Get cell index. */
      ic = atom_cells[i].cell;

      for (i_off = 0; i_off < noffsets; ++i_off) {

         /* Compute index of neighboring cell and get index of the
            first atom in that cell. */
         jc = ic + offsets[i_off];
         j = first[jc];

         /* Loop over atoms in neighboring cell. */
         while (j != -1) {

            /* Determine whether entry should be added to neighbor list. */
            j_real = atom_cells[j].real;
            j_mol = atom_mol[j_real];
            j_rel = atom_rel[j_real];
            inter_flag = i_mol != j_mol;
            if (inter_flag || exclusions[i_species][i_rel][j_rel]) {

               /* Calculate pair separation. */

               for (k = 0; k < NDIM; ++k)
                 sep[k] = atom_coords[j][k] - s_i[k];

                r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

               /* If the pair separation is less than r_nl, add entries
                  to neighbor lists. */

               if (r2_sep < r2_nl) {
                  if (nl_tail[i] == NULL){
                     p = nl_head[i];
                  } else {
                     p = nl_tail[i] -> next;
                     if (p == NULL) {
                           p = gmalloc(sizeof(nl_entry)); 
                        p -> next = NULL;
                       nl_tail[i] -> next = p ;
                     }
                  }
                  p -> atom = j_real;
                  nl_tail[i] = p;
               }
            }
            j = atom_cells[j].next;
         }
      }
   }
   /* Zero movement accumulators. */
/*   zero_move(n_atoms, atom_move);  */
}

/*****************************************************************************/
/* Zero movement accumulators. */
void zero_move(int n_atoms, double **atom_move)
{
   int i, j;

   /* Zero movement accumulators. */
   for (i = 0; i < n_atoms; ++i) 
      for (j = 0; j < NDIM; ++j)
         atom_move[i][j] = 0.0;
   
}
