/* Input/output routines for builder module. */

#include "build.h"
#include "ff.h"

/*************************************************************************/
/* Read simulator module input parameter file. */
/*
input :
    name of the file containing the parameters used in the builder (par_file)
output :
    Force field file for masses (mass_file)
    Force field file for potential interactions (pot_file)
    Switch for periodic system (period_switch - 1 means periodic)
    Number of molecular species (n_species)
    Array of number of molecules of each specie in the unit cell 
                                                       (n_mols_per_unit_cell)
    Array of files of the molecular structures (struct_file)
    Switch for the combination rule between atomic site (vdw_rad_switch)
    Overlap criterion between atomi sites (overlap_criterion)
    Switch to chose the initial state being built (init_switch)
    Number of smectic layers present in the initial state (n_layers)
    Layer spacing (layer_spacing)
    Number density for the molecules (rho)
    Number of unit cells (n_unit_cells)
    Aspect ratio for the box (aspect_xy, aspect_xz)
    Array of the molecular principal axe chosen to be along the z cartesian
                                                          axe (principal)
    Seed for the random generator (seed)
    Name of the output header file (header_file)
    Name of the output configuration file (config_file)
*/

void read_simu_params(char *par_file, char **header_file, char **config_file,
   int *vel_switch, char **header_file_save, char **config_file_save, 
   char **thermo_file, char **trajec_file, char **mass_file, 
   char **pot_file, char **stretch_file, char **bend_file, int *vdw_rad_switch, 
   int *bend_switch, int *n_species_chk,
   int **excl_switch, double *r_on, double *r_off, int *neigh_switch, 
   int *phant_switch, int *kc, double *r_nl, int *n_cycles, int *n_reshape, 
   int *reshape_axis, int *opt_switch, double *dr_max, double *ratio_dang_dr, 
   double *dh_max, int *temp_switch, double *temp_time,
   int *press_switch, int *scale_con, double *press_time,
   int ***scale, double *temp, double *press, 
   int *coul_switch, double *accuracy, double *delta_grid, int *B_spline_order,
   int *n_graph, int *n_opt, int *n_thermo, int *n_block, int *n_trajec, 
   long *seed, double *delta, 
   int *pore_switch, double *r0, double *z0, 
   int *graph_pore_switch, int *graph_switch, char **color_file, 
   int *species_choice, double *radius_sph, double *radius_cyl, int *resph, 
   int *antialias_switch, int *x_cut_switch, int *surf_tension_switch, int *n_slab)
{
   int i_species, i, j;
   FILE *f_param;
   char line[MAX_LINE];

   /* Open parameter file. */
   if ((f_param = fopen(par_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_simu_params");

   /* Read in simu parameters. */
   fgets(line, MAX_LINE, f_param);
   *header_file = allocate_1d_array(F_MAX, sizeof(char));
   *config_file = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s %s %d\n", *header_file, *config_file, vel_switch);
   fgets(line, MAX_LINE, f_param);
   *header_file_save = allocate_1d_array(F_MAX, sizeof(char));
   *config_file_save = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s %s\n", *header_file_save, *config_file_save);
   fgets(line, MAX_LINE, f_param);
   *thermo_file = allocate_1d_array(F_MAX, sizeof(char));
   *trajec_file = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s %s\n", *thermo_file, *trajec_file);
   fgets(line, MAX_LINE, f_param);
   *mass_file = allocate_1d_array(F_MAX, sizeof(char));
   *pot_file = allocate_1d_array(F_MAX, sizeof(char));
   *stretch_file = allocate_1d_array(F_MAX, sizeof(char));
   *bend_file = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s %s %s %s\n", *mass_file, *pot_file, *stretch_file, *bend_file);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d\n", vdw_rad_switch, bend_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", n_species_chk);
   fgets(line, MAX_LINE, f_param);
   *excl_switch = allocate_1d_array(*n_species_chk, sizeof(int));
   for (i = 0; i < *n_species_chk; ++i)
     fscanf(f_param, "%d ", &(*excl_switch)[i]);
   fscanf(f_param, "\n");
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf\n", r_on, r_off);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d\n", neigh_switch, phant_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %lf\n", kc, r_nl);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d %d\n", n_cycles, n_reshape, reshape_axis);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", opt_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf %lf\n", dr_max, ratio_dang_dr, dh_max);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %lf\n", temp_switch, temp_time);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d %lf\n", press_switch, scale_con, press_time);
   fgets(line, MAX_LINE, f_param);
   *scale = allocate_2d_array(3, 3, sizeof(int));
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         fscanf(f_param, "%d", &(*scale)[i][j]);
   fscanf(f_param, "\n");
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf\n", temp, press);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %lf %lf %d\n", coul_switch, accuracy, delta_grid, 
          B_spline_order); 
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d %d %d %d\n", n_graph, n_opt, n_thermo, n_block, 
          n_trajec);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%ld\n", seed);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf \n", delta);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %lf %lf\n", pore_switch, r0, z0);

/* Read parameters used in the graphics. */
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d\n", graph_switch, graph_pore_switch);
   fgets(line, MAX_LINE, f_param);
   *color_file = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s\n", *color_file);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", species_choice);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf\n", radius_sph, radius_cyl);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d\n", resph, antialias_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", x_cut_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d\n", surf_tension_switch, n_slab);

   /* Close parameter file. */
   fclose(f_param);
}

/*************************************************************************/

/* Read periodicity switch, number of molecular species, molecular structures,
   numbers of molecules of each species, and list of molecular species to
   header of output file. */
/*
input :
       Name of the input header file
output :
       Switch for periodic system (period_switch)
       Number of species (n_species)
       Array of number of atom in template molecule of each specie 
                                                           (n_atoms_per_mol)
       Array of template atom - atomic labels (temp_atm_lab)
       Array of template atom - atomic types (temp_atm_type)
       Array of template atom - atom branching number (temp_atm_nbr)
       Array of template atom - label of branched atoms (temp_atm_br)
       Array of template atom - order of the bonds branched to an atom
                                                            (temp_atm_ord)
       Array of template atom - realtive position of atoms in the template
                                molecule                    (temp_atm_pos)
       Array of template atom - atomic site charges (temp_atm_chg) 
       Arrays of template atom - LJ atomic site parameters (temp_atm_sigma,
                                                            temp_atm_eps)
       Array of number of bonds in a template molecule (n_bonds_per_mol)
       Arrays of template molecule - label of the 2 atoms sharing a bond
							(temp_bonds_1,
                                                         temp_bonds_2) 
       Array of template molecule - order of a bond (temp_bonds_ord)
       Total number of molecules (n_mols)
       Specie label of the molecules (mol_species)

*/

void read_header_direct(char *header_file, int *period_switch, int *n_species,
  int **n_atoms_per_mol, int ***temp_atm_lab, char ****temp_atm_type, 
  int ***temp_atm_nbr, int ****temp_atm_br, double ****temp_atm_ord, 
  double ****temp_atm_pos, double ***temp_atm_chg, 
  int **n_bonds_per_mol, int ***temp_bonds_1, 
  int ***temp_bonds_2, double ***temp_bonds_ord, int **n_mols_per_species, 
  int *n_mols, int *n_atoms, int **mol_species)
{
   int i, j, i_species;
   FILE *f_input;

   /* Open header file. */
   if ((f_input = fopen(header_file, "r")) == NULL)
      error_exit("Unable to open header file in read_header_direct");

   /* Read in periodicity switch. */
   fscanf(f_input,"%d\n",&(*period_switch));

   /* Read in number of molecular species. */
   fscanf(f_input,"%d\n",&(*n_species));

   /* Allocate memory for arrays containing molecular structures */
   *n_atoms_per_mol = allocate_1d_array(*n_species, sizeof(int));
   *n_mols_per_species = allocate_1d_array(*n_species, sizeof(int));
   *n_bonds_per_mol = allocate_1d_array(*n_species, sizeof(int));
   *temp_atm_lab = allocate_1d_array(*n_species, sizeof(int*));
   *temp_atm_type = allocate_1d_array(*n_species, sizeof(char**));
   *temp_atm_pos = allocate_1d_array(*n_species, sizeof(double**));
   *temp_atm_chg = allocate_1d_array(*n_species, sizeof(double*));
   *temp_atm_nbr = allocate_1d_array(*n_species, sizeof(int*));
   *temp_atm_br = allocate_1d_array(*n_species, sizeof(int**));
   *temp_atm_ord = allocate_1d_array(*n_species, sizeof(double**));
   *temp_bonds_1 = allocate_1d_array(*n_species, sizeof(int*));
   *temp_bonds_2 = allocate_1d_array(*n_species, sizeof(int*));
   *temp_bonds_ord = allocate_1d_array(*n_species, sizeof(double*));
/*   *temp_atm_sigma = allocate_1d_array(*n_species, sizeof(double*));
   *temp_atm_eps = allocate_1d_array(*n_species, sizeof(double*));
*/

   *n_mols = 0;
   *n_atoms = 0;
   /* Read in molecular structures and number of molecules for all species.*/
   for (i_species = 0; i_species < *n_species; ++i_species) {

     /* Read in number of atoms per molecule. */
     fscanf(f_input,"%d\n", &(*n_atoms_per_mol)[i_species]);

     /* Allocate memory for template atoms arrays. */
     (*temp_atm_lab)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(int));
     (*temp_atm_type)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], F_MAX, sizeof(char));
     (*temp_atm_chg)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(double));
     (*temp_atm_nbr)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(int));
     (*temp_atm_pos)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], NDIM, sizeof(double));
     (*temp_atm_br)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], 6, sizeof(int));
     (*temp_atm_ord)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], 6, sizeof(double));
 /*    (*temp_atm_sigma)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(double));
     (*temp_atm_eps)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(double));
*/
     /* Read in template atom attributes. */
     for (i = 0; i < (*n_atoms_per_mol)[i_species]; ++i)
      {
      fscanf(f_input,"%d %s %d ", &(*temp_atm_lab)[i_species][i],
         (*temp_atm_type)[i_species][i], &(*temp_atm_nbr)[i_species][i]);
         for (j = 0; j < (*temp_atm_nbr)[i_species][i]; ++j)
          fscanf(f_input,"%d %lf ",&(*temp_atm_br)[i_species][i][j],
            &(*temp_atm_ord)[i_species][i][j]);
         fscanf(f_input,"\n");
         fscanf(f_input,"%lf %lf %lf %lf\n", &(*temp_atm_pos)[i_species][i][0], 
          &(*temp_atm_pos)[i_species][i][1], &(*temp_atm_pos)[i_species][i][2], 
          &(*temp_atm_chg)[i_species][i]);
 /*     fscanf(f_input,"%lf %lf\n", &(*temp_atm_sigma)[i_species][i],
           &(*temp_atm_eps)[i_species][i]); 
*/
      }

     /* Read in number of bonds per molecule. */
     fscanf(f_input,"%d\n", &(*n_bonds_per_mol)[i_species]);

    if ( (*n_bonds_per_mol)[i_species] > 0) {
     (*temp_bonds_1)[i_species] = 
        allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(int));
     (*temp_bonds_2)[i_species] = 
        allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(int));
     (*temp_bonds_ord)[i_species] = 
        allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(double));

    }
     /* Read in template bond attributes. */
     for (i = 0; i < (*n_bonds_per_mol)[i_species]; ++i)
      fscanf(f_input, "%d %d %lf\n", &(*temp_bonds_1)[i_species][i], 
          &(*temp_bonds_2)[i_species][i], &(*temp_bonds_ord)[i_species][i]);

     /* Read in number of molecules. */
     fscanf(f_input, "%d\n", &(*n_mols_per_species)[i_species]);

     /* Increment molecule and atom counters. */
     (*n_mols) += (*n_mols_per_species)[i_species];
     (*n_atoms) += (*n_mols_per_species)[i_species] * 
                               (*n_atoms_per_mol)[i_species]; 
   }

   /* Allocate memory for molecular species array. */
   *mol_species = allocate_1d_array(*n_mols, sizeof(int));

   /* Read in list of molecular species. */
   for (i = 0; i < *n_mols; ++i)
      fscanf(f_input, "%d ",&(*mol_species)[i]);
   fscanf(f_input, "\n");

   /* Close the input file. */
   fclose(f_input);
}

/*************************************************************************/
void write_header_direct(char *header_file, int period_switch, int n_species,
  int *n_atoms_per_mol, int **temp_atm_lab, char ***temp_atm_type,
  int **temp_atm_nbr, int ***temp_atm_br, double ***temp_atm_ord,
  double ***temp_atm_pos, double **temp_atm_chg, 
  int *n_bonds_per_mol, int **temp_bonds_1,
  int **temp_bonds_2, double **temp_bonds_ord, int *n_mols_per_species,
  int n_mols, int *mol_species)
{
   int i, j, i_species;
   FILE *f_output;

   /* Open header file. */
   if ((f_output = fopen(header_file, "w")) == NULL)
      error_exit("Unable to open header file in write_config_direct");

   /* Write out periodicity switch. */
   fprintf(f_output,"%d\n",period_switch);

   /* Write out number of molecular species. */
   fprintf(f_output,"%d\n",n_species);

   /* Write out molecular structures and number of molecules for all species.*/
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Write out number of atoms per molecule. */
      fprintf(f_output,"%d\n", n_atoms_per_mol[i_species]);

      /* Write out template atom attributes. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i)
      {
      fprintf(f_output,"%d %s %d ", temp_atm_lab[i_species][i],
         temp_atm_type[i_species][i], temp_atm_nbr[i_species][i]);
         for (j = 0; j < temp_atm_nbr[i_species][i]; ++j)
          fprintf(f_output,"%d %g ",temp_atm_br[i_species][i][j],
            temp_atm_ord[i_species][i][j]);
         fprintf(f_output,"\n");
         fprintf(f_output,"%g %g %g %g\n", temp_atm_pos[i_species][i][0],
           temp_atm_pos[i_species][i][1], temp_atm_pos[i_species][i][2],
           temp_atm_chg[i_species][i]);
  /*    fprintf(f_output,"%g %g\n", temp_atm_sigma[i_species][i],
           temp_atm_eps[i_species][i]); */
      }
      /* Write out number of bonds per molecule. */
      fprintf(f_output,"%d\n", n_bonds_per_mol[i_species]);

      /* Write out template bond attributes. */
      for (i = 0; i < n_bonds_per_mol[i_species]; ++i)
      fprintf(f_output, "%d %d %g\n", temp_bonds_1[i_species][i],
           temp_bonds_2[i_species][i], temp_bonds_ord[i_species][i]);

      /* Write out number of molecules. */
      fprintf(f_output, "%d\n", n_mols_per_species[i_species]);
   }

   /* Write out list of molecular species. */
   for (i = 0; i < n_mols; ++i)
   fprintf(f_output, "%d ", mol_species[i]);
   fprintf(f_output, "\n");

   /* Close header file. */
   fclose(f_output);
}

/*************************************************************************/
/* Read atomic configuration to config file. */
/*
input :
      Name of the configuration input file (config_file)
      Switch about periodicity (period_switch)
      Total number of atoms (n_atoms)
output :
      Array of box matrix (h)
      Array of cartesian coordinates of atoms (atom_coords)
      Array of velocities of atoms (atom_vels)
*/

void read_config_direct(char *config_file, int period_switch, double ***h,
   int n_atoms, double ***atom_coords, double ***atom_vels)
{
   FILE *f_input;
   int i, j;

   /* Open config file. */
   if ((f_input = fopen(config_file, "r")) == NULL)
      error_exit("Unable to open configuration file in read_config_direct");

   /* Write out box matrix if params.period_switch == 1. */
   if (period_switch)
      {
      /* Allocate memory for the box matrix. */
      *h = allocate_2d_array(3, 3, sizeof(double));
      /* Read in box matrix elements. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fscanf(f_input, "%lf ", &(*h)[i][j]);
      fscanf(f_input, "\n");
      }

   /* Allocate memory for arrays of atomic properties. */
   if (period_switch)
      *atom_coords = allocate_2d_array(8 * n_atoms, 3, sizeof(double));
   else
      *atom_coords = allocate_2d_array(n_atoms, 3, sizeof(double));
   *atom_vels = allocate_2d_array(n_atoms, 3, sizeof(double));

   /* Read in atomic positions and velocities. */
   for (i = 0; i < n_atoms; ++i)
     fscanf(f_input, "%lf %lf %lf %lf %lf %lf\n",&(*atom_coords)[i][0],
      &(*atom_coords)[i][1], &(*atom_coords)[i][2], &(*atom_vels)[i][0], 
      &(*atom_vels)[i][1], &(*atom_vels)[i][2]);

   /* Close config file. */
   fclose(f_input);
}

/*************************************************************************/
/* Write atomic configuration to config file. */
/*
input :
      Switch about periodicity (period_switch)
      Array of box matrix (h)
      Total number of atoms (n_atoms)
      Array of cartesian coordinates of atoms (atom_coords)
      Array of velocities of atoms (atom_vels)
output :
      Name of the configuration output file (config_file)
*/

void write_config_direct(char *config_file, int period_switch, double **h,
   int n_atoms, double **atom_coords, double **atom_vels)
{
   FILE *f_output;
   int i, j;
   char par_file[F_MAX];

   /* Open config file. */
   if ((f_output = fopen(config_file, "w")) == NULL)
      error_exit("Unable to open configuration file in write_config_direct");

   /* Write out box matrix if params.period_switch == 1. */
   if (period_switch) {
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fprintf(f_output, "%g ", h[i][j]);
      fprintf(f_output, "\n");
    }

   /* Write out atomic positions and velocities. */
   for (i = 0; i < n_atoms; ++i)
     {
     fprintf(f_output, "%g %g %g %g %g %g\n", atom_coords[i][0],
      atom_coords[i][1], atom_coords[i][2], atom_vels[i][0], atom_vels[i][1],
      atom_vels[i][2]);

     }
   /* Close config file. */
   fclose(f_output);
}

/****************************************************************************/
/* Read atomic positions from trajec file. */
void read_positions_direct(FILE *f_input, double **h, int n_atoms, 
    double **atom_coords)
{
   int i, j, i_atom;

   /* Read in box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fread(&h[i][j], sizeof(double), 1, f_input);

   /* Read in atomic positions. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
         fread(atom_coords[i_atom], sizeof(double), 3, f_input);
}

/****************************************************************************/
/* Write atomic positions to trajec file. */
void write_positions_direct(FILE *f_output, double **h, int n_atoms, 
    double **atom_coords)
{
   int i, j, i_atom;

   /* Write out box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
fprintf(f_output, "%g ", h[i][j]);
      fprintf(f_output, "\n");
//            fwrite(&h[i][j], sizeof(double), 1, f_output);

   /* Write out atomic positions. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
      fprintf(f_output, "%g %g %g \n", atom_coords[i][0],
      atom_coords[i][1], atom_coords[i][2]);

//         fwrite(atom_coords[i_atom], sizeof(double), 3, f_output);
}
/*****************************************************************************/
/* Read atomic velocities from trajec file. */
void read_vels_direct(FILE *f_input, int n_atoms, double **atom_vels)
{
   int i, j, i_atom;

   /* Read in atomic velocities. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
         fread(atom_vels[i_atom], sizeof(double), 3, f_input);
}

/****************************************************************************/
/* Write atomic velocities to trajec file. */
void write_vels_direct(FILE *f_output, int n_atoms, double **atom_vels)
{
   int i_atom;

   /* Write out atomic velocities. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
      fprintf(f_output, "%g %g %g \n", atom_vels[i_atom][0],
      atom_vels[i_atom][1], atom_vels[i_atom][2]);

}
/*****************************************************************************/
void write_thermo(FILE *f_thermo, int i_cycle, int n_atoms, double **h, 
                  double volume, double rho, double at_temp, double pe_stretch, 
                  double pe_bend, double pe_vdw_s, double pe_coul_s,  
                  double pe_coul_recip, double pe, double ke, double te,  
                  double **at_stress_vdw_s, double **at_stress_coul_s,
                  double **at_stress_coul_recip,
                  double **at_stress_stretch, double **at_stress_bend,
                  double **at_stress, double **at_stress_kin,
                  double **mol_stress_vdw_s, double **mol_stress_coul_s,
                  double **mol_stress_coul_recip, double **mol_stress,
                  double **mol_stress_kin, double at_press_vdw_s,
                  double at_press_coul_s, double at_press_coul_recip,
                  double at_press_stretch, double at_press_bend,
                  double at_press, double at_press_kin,
                  double mol_press_vdw_s, double mol_press_coul_s,
                  double mol_press_coul_recip, double mol_press,
                  double mol_press_kin)

{
int i, j;

  /* Write instantaneous quantities to thermodynamics file. */
  
  /* Write MD step. */
  fprintf(f_thermo,"%d ", i_cycle);

  /* Write box tensor. */
  for (i = 0; i < 3; ++i)
    fprintf(f_thermo, "%g ", h[i][i]);

  /* Write volume and atomic number density to output file. */
  fprintf(f_thermo, "%g %g ", volume, rho);

  /* Write atomic pressur to output file. */
  fprintf(f_thermo, "%g ", at_temp);

  /* Write the different contributions to potential energy. */
  fprintf(f_thermo, "%g %g %g %g %g %g ", pe_stretch, pe_bend, pe_vdw_s, 
          pe_coul_s, pe_coul_recip, pe);

  /* Write kinetics energy and total energy. */
  fprintf(f_thermo, "%g %g ", ke, te);

  /* Write potential, kinetics total energy per atom. */
  fprintf(f_thermo, "%g %g %g ", pe / n_atoms, ke / n_atoms, te / n_atoms);

  /* Write the various contributions to the atomic stress tensor to
     the output file. */
  for (i = 0; i < 3; ++i) 
    for (j = i; j < 3; ++j)
      fprintf(f_thermo, "%g %g %g %g %g %g %g ", at_stress_vdw_s[i][j], 
              at_stress_coul_s[i][j], at_stress_coul_recip[i][j], 
              at_stress_stretch[i][j], at_stress_bend[i][j], at_stress_kin[i][j],
              at_stress[i][j]);

  /* Write the various contributions to the molecular stress tensor to
     the output file. */
  for (i = 0; i < 3; ++i) 
    for (j = i; j < 3; ++j)
      fprintf(f_thermo, "%g %g %g %g %g ", mol_stress_vdw_s[i][j], 
              mol_stress_coul_s[i][j], mol_stress_coul_recip[i][j], 
              mol_stress_kin[i][j], mol_stress[i][j]);

  /* Write atomic and molecular contributions to the hydrostatic pressure. */
  fprintf(f_thermo, "%g %g %g %g %g %g %g ", at_press_vdw_s, at_press_coul_s,
          at_press_coul_recip, at_press_stretch, at_press_bend, at_press_kin,
          at_press);
  fprintf(f_thermo, "%g %g %g %g %g\n", mol_press_vdw_s, mol_press_coul_s, 
          mol_press_coul_recip, mol_press_kin, mol_press);
}
