/* Input/output routines for template module. */

#include "build.h"
#include "ff.h"

/****************************************************************************/
/* Read molecular structure from structure file. */
/*
input : 
    array of files containing molecular structure of each specie (struct_file)

output :
    array of number of atoms per molecule for each specie (n_atoms_per_mol)
    array of template atom labels (temp_atm_lab)
    array of template atom type (temp_atm_type)
    array of template atom position (temp_atm_pos)
    array of template atom charge (temp_atm_chg)
    array of template atom numbre of branching (temp_atm_nbr)
    array of template atom label of branched atoms (temp_atm_br)
    array of template atom label order (temp_atm_ord)
    array of the number of bonds per molecule for each specie (n_bonds_per_mol)
    arrays of template bond - the labels of the 2 atoms sharing a bond
                                                  (temp_bonds_1, temp_bonds_2) 
    array of template bond - the order of a template bond (temp_bonds_ord)
*/

void read_str(char *struct_file, int *n_atoms_per_mol, int **temp_atm_lab,
   char ***temp_atm_type, double ***temp_atm_pos, double **temp_atm_chg,
   int **temp_atm_nbr, int ***temp_atm_br, double ***temp_atm_ord,
   int *n_bonds_per_mol, int **temp_bonds_1, int **temp_bonds_2,
   double **temp_bonds_ord)
{
   FILE *f_input;
   char line[MAX_LINE];
   int i, atom_1, atom_2, order;

   /* Open structure file. */
   if ((f_input = fopen(struct_file, "r")) == NULL)
      error_exit("Unable to open structure file in read_str");

   /* Skip comments in header of structure file. */
   do {
      if (fgets(line, MAX_LINE, f_input) == NULL)
         error_exit("Molecular structure data missing in read_str");
   } while (line[0] == '#');

   /* Read in number of atoms per molecule. */
   sscanf(line, "%d", n_atoms_per_mol);

   /* Allocate memory for array of template atom attributes. */
   *temp_atm_lab = allocate_1d_array(*n_atoms_per_mol, sizeof(int));
   *temp_atm_type = allocate_2d_array(*n_atoms_per_mol, TYPE_MAX, sizeof(char));
   *temp_atm_pos = allocate_2d_array(*n_atoms_per_mol, NDIM, sizeof(double));
   *temp_atm_chg = allocate_1d_array(*n_atoms_per_mol, sizeof(double));
   *temp_atm_nbr = allocate_1d_array(*n_atoms_per_mol, sizeof(int));
   *temp_atm_br = allocate_2d_array(*n_atoms_per_mol, 6, sizeof(int));
   *temp_atm_ord = allocate_2d_array(*n_atoms_per_mol, 6, sizeof(double));

   for (i = 0; i < *n_atoms_per_mol; ++i)
      {
      fscanf(f_input, "%d %s %lf %lf %lf %lf",
         &(*temp_atm_lab)[i], (*temp_atm_type)[i], &(*temp_atm_pos)[i][0],
         &(*temp_atm_pos)[i][1], &(*temp_atm_pos)[i][2], &(*temp_atm_chg)[i]);
      }
   /* Read in number of bonds per molecule. */
   fscanf(f_input, "%d", n_bonds_per_mol);

   /* Allocate memory for array of template bond attributes. */
   if (*n_bonds_per_mol > 0) {
   *temp_bonds_1 = allocate_1d_array(*n_bonds_per_mol, sizeof(int));
   *temp_bonds_2 = allocate_1d_array(*n_bonds_per_mol, sizeof(int));
   *temp_bonds_ord = allocate_1d_array(*n_bonds_per_mol, sizeof(double));

}
   /* Read in template bond attributes. */
   for (i = 0; i < *n_bonds_per_mol; ++i)
      {
      fscanf(f_input, "%d %d %lf",
         &(*temp_bonds_1)[i], &(*temp_bonds_2)[i], &(*temp_bonds_ord)[i]);
      }

   /* Close structure file. */
   fclose(f_input);

   /* Add connectivity information to template atom array. */
   for (i = 0; i < *n_atoms_per_mol; ++i)
      (*temp_atm_nbr)[i] = 0;
   for (i = 0; i < *n_bonds_per_mol; ++i) {
      atom_1 = (*temp_bonds_1)[i];
      atom_2 = (*temp_bonds_2)[i];
      order = (*temp_bonds_ord)[i];
      (*temp_atm_br)[atom_1][(*temp_atm_nbr)[atom_1]] = atom_2;
      (*temp_atm_br)[atom_2][(*temp_atm_nbr)[atom_2]] = atom_1;
      (*temp_atm_ord)[atom_1][(*temp_atm_nbr)[atom_1]] = order;
      (*temp_atm_ord)[atom_2][(*temp_atm_nbr)[atom_2]] = order;
      ++(*temp_atm_nbr)[atom_1];
      ++(*temp_atm_nbr)[atom_2];
   }
}

/****************************************************************************/
/* Read mass parameters from specified data file. */
/*input :  
      force field file name (mass_file)

output : 
      the FF mass structure (mass)
      the number of entries in FF mass structure (n_mass)
*/

void read_mass_params(char *mass_file, mass_entry **mass, int *n_mass)
{
   FILE *f_input;
   char line[MAX_LINE];
   mass_entry tmp_mass;

   /* Open parameter file. */
   if ((f_input = fopen(mass_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_mass_params");

   /* Read in mass entries. */
   while (fgets(line, MAX_LINE, f_input) != NULL) {
      sscanf(line, "%s %lf %lf",
         tmp_mass.type, &tmp_mass.true_mass, &tmp_mass.eff_mass);
      if (((*mass) = realloc((*mass), 
        ((*n_mass) + 1) * sizeof(mass_entry))) == NULL)
         error_exit("Out of memory in read_mass_params");
      (*mass)[(*n_mass)] = tmp_mass;
      ++(*n_mass);
   }
 
   /* Close input file. */
   fclose(f_input);
}

/****************************************************************************/
/* Read van der Waals (LJ) parameters from specified data file. */
/*
input :  
         the LJ file name (lj_file)

output : 
         the FF lj structure (lj)
         the number of entries in FF lj structure (n_lj)
*/

void read_lj_params(char *lj_file, lj_entry **lj, int *n_lj)
{
   FILE *f_input;
   char line[MAX_LINE];
   lj_entry tmp_lj;

   /* Open parameter file. */
   if ((f_input = fopen(lj_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_lj_params");

   /* Read in lj entries. */
   while (fgets(line, MAX_LINE, f_input) != NULL) {
      sscanf(line, "%s %s %lf %lf",
         tmp_lj.type_1, tmp_lj.type_2, &tmp_lj.sigma, &tmp_lj.epsilon);
      if ((*lj = realloc(*lj,
         (*n_lj + 1) * sizeof(lj_entry))) == NULL)
         error_exit("Out of memory in read_lj_params");
      (*lj)[*n_lj] = tmp_lj;
      ++(*n_lj);
   }

   /* Close input file. */
   fclose(f_input);
}

/****************************************************************************/
/* Read van der Waals (exp-6) parameters from specified data file. */
/*
input :  
         the exp-6 file name (e6_file)
output : 
         the FF exp-6 structure (e6)
         the number of entries in FF exp-6 structure (n_e6)
*/

void read_e6_params(char *e6_file, e6_entry **e6, int *n_e6)
{
   FILE *f_input;
   char line[MAX_LINE];
   e6_entry tmp_e6;

   /* Open parameter file. */
   if ((f_input = fopen(e6_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_e6_params");

   /* Read in e6 entries. */
   while (fgets(line, MAX_LINE, f_input) != NULL) {
      sscanf(line, "%s %s %lf %lf %lf",
         tmp_e6.type_1, tmp_e6.type_2, &tmp_e6.A, &tmp_e6.B, &tmp_e6.C);
      if ((*e6 = realloc(*e6,
         (*n_e6 + 1) * sizeof(e6_entry))) == NULL)
         error_exit("Out of memory in read_e6_params");
      (*e6)[*n_e6] = tmp_e6;
      ++(*n_e6);
   }

   /* Close input file. */
   fclose(f_input);
}

/* Read Spring constant and equilibrium bond distance for stretching potential */
/*
input :
         the bend file name (stretch_file)
output :
         the FF bend structure (strecth)
         the number of entries in FF strecth structure (n_strecth)
*/
void read_stretch_params(char *stretch_file, stretch_entry **stretch, int *n_stretch)
{
   FILE *f_input;
   char line[MAX_LINE];
   stretch_entry tmp_stretch;

    /* Open parameter file. */
   if ((f_input = fopen(stretch_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_stretch_params");

    /* Read in bend entries. */
         while (fgets(line, MAX_LINE, f_input) != NULL) {
            sscanf(line, "%s %s %lf %lf",
               tmp_stretch.type_1, tmp_stretch.type_2, 
               &tmp_stretch.spring, &tmp_stretch.r_equil);
            if ((*stretch = realloc(*stretch,
               (*n_stretch + 1) * sizeof(stretch_entry))) == NULL)
               error_exit("Out of memory in read_stretch_params");
            (*stretch)[*n_stretch] = tmp_stretch;
            ++(*n_stretch);
         }
/* Close input file. */
   fclose(f_input);
}

/* Read Spring constant and equilibrium angle for bending potential */
/*
input :
         the bend file name (bend_file)
output :
         the FF bend structure (bend)
         the number of entries in FF bend structure (n_bend)
*/
void read_bend_params(char *bend_file, bend_entry **bend, int *n_bend)
{
   FILE *f_input;
   char line[MAX_LINE];
   bend_entry tmp_bend;

    /* Open parameter file. */
   if ((f_input = fopen(bend_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_bend_params");

    /* Read in bend entries. */
         while (fgets(line, MAX_LINE, f_input) != NULL) {
            sscanf(line, "%s %s %s %lf %lf",
               tmp_bend.type_1, tmp_bend.type_2, tmp_bend.type_3,
               &tmp_bend.spring_bend, &tmp_bend.theta_equil);
            if ((*bend = realloc(*bend,
               (*n_bend + 1) * sizeof(bend_entry))) == NULL)
               error_exit("Out of memory in read_bend_params");
            (*bend)[*n_bend] = tmp_bend;
            ++(*n_bend);
         }
/* Close input file. */
   fclose(f_input);
}
