/* Shared electrostatics initialization routines for molecular 
   simulation programs. */

#include "build.h"
#include "ff.h"
#include "proto.h"


/* Calculate Ewald self-energy correction. */
void ewald_self(int n_atoms, double *atom_charges, double alpha, 
                double *pe_coul_self)
{
     int i;

     *pe_coul_self = 0.0;
     for (i = 0; i < n_atoms; ++i)
         *pe_coul_self += SQR(atom_charges[i]);
      *pe_coul_self *= alpha / sqrt(PI);
}

/* Create lookup tables. */
void create_lookup_tables(double accuracy, double *one_over_erf_tab_incr, 
                          double **erfc_sqrt_tab, double **erf_sqrt_tab)
{
   int i_tab;
   double x, erf_tab_incr, erf_tab_max;
 
   /* Allocate memory for lookup table arrays. */
   *erfc_sqrt_tab = allocate_1d_array(N_ERF_TAB + 1, sizeof(double));
   *erf_sqrt_tab = allocate_1d_array(N_ERF_TAB + 1, sizeof(double));

   /* Set up erfc and erf tables. */
   erf_tab_max = - log(accuracy);
   erf_tab_incr = erf_tab_max / N_ERF_TAB;
   *one_over_erf_tab_incr = 1.0 / erf_tab_incr;
   for (i_tab = 0; i_tab <= N_ERF_TAB; ++i_tab) {
      x = i_tab * erf_tab_incr;
      (*erfc_sqrt_tab)[i_tab] = erffc(sqrt(x));
      (*erf_sqrt_tab)[i_tab] = erff1(sqrt(x));
   }
}

/* Test whether a given integer is a product of powers of 2, 3, and 5
   (used to optimize procedures that utilize FFTs). */
int product_of_powers_of_2_3_5(int i, int *pow_2, int *pow_3, int *pow_5)
{
   int j, k;

   /* Initialize j. */
   j = i;

   /* Set power counters to zero. */
   pow_2 = pow_3 = pow_5 = 0;

   /* Test for factors of 2. */
   while (j != 1) {
      k = j / 2;
      if (j == 2 * k) {
         ++pow_2;
         j = k;
      }
      else
         break;
   }

   /* Test for factors of 3. */
   while (j != 1) {
      k = j / 3;
      if (j == 3 * k) {
         ++pow_3;
         j = k;
      }
      else
         break;
   }

   /* Test for factors of 5. */
   while (j != 1) {
      k = j / 5;
      if (j == 5 * k) {
         ++pow_5;
         j = k;
      }
      else
         break;
   }

   /* Return 1 if i is a product of powers of 2, 3, and 5, and 0 if not. */
   return (j == 1);
}

void init_array_pme(int n_atoms, int B_spline_order, 
                    double **gridded_atom_coords, double **M_x, double **M_y, 
                    double **M_z, double **dM_du_x, double **dM_du_y,
                    double **dM_du_z)
{
  int i, j;
  
  for (i = 0; i < n_atoms; ++i){
      gridded_atom_coords[i][0] = 0.0; 
      gridded_atom_coords[i][1] = 0.0; 
      gridded_atom_coords[i][2] = 0.0; 
      for (j = 0; j < B_spline_order; ++j){
          M_x[i][j] = 0.0;
          M_y[i][j] = 0.0;
          M_z[i][j] = 0.0;
          dM_du_x[i][j] = 0.0;
          dM_du_y[i][j] = 0.0;
          dM_du_z[i][j] = 0.0;
      }
  }
}

void init_array_ewald(int n_atoms, int m1_max, int m2_max, int m3_max,
                      double **cos_G_1_dot_r, double **sin_G_1_dot_r,
                      double **cos_G_2_dot_r, double **sin_G_2_dot_r,
                      double **cos_G_3_dot_r, double **sin_G_3_dot_r,
                      double *q_cos_G_12_dot_r, double *q_sin_G_12_dot_r,
                      double *q_cos_G_123_dot_r, double *q_sin_G_123_dot_r)
{
  int i, m1, m2, m3;

  for (i = 0; i < n_atoms; ++i){
     for (m1 = 0; m1 < m1_max; ++m1)
         cos_G_1_dot_r[i][m1] = sin_G_1_dot_r[i][m1] = 0.0;
     for (m2 = 0; m2 < m2_max; ++m2)
         cos_G_2_dot_r[i][m2] = sin_G_2_dot_r[i][m2] = 0.0;
     for (m3 = 0; m3 < m3_max; ++m3)
         cos_G_3_dot_r[i][m2] = sin_G_3_dot_r[i][m2] = 0.0;

     q_cos_G_12_dot_r[i] = q_sin_G_12_dot_r[i] = q_cos_G_123_dot_r[i] =
     q_sin_G_123_dot_r[i] = 0.0;
  }

}
               
