/* Bonded force routines for molecular modeling package. */

#include "build.h"
#include "ff.h"
#include "proto.h"
#include "stress_prof.h"

/* Calculate bond stretching interactions. */
void stretch_harmonic(int period_switch, int eval_flag, 
             double **f_stretch, int *mol_first_atm, int *n_bonds_per_mol,
             double **at_stress_stretch, double *pe_stretch, 
             int n_atoms, int n_mols, int *mol_species,
             int **temp_bonds_1, int **temp_bonds_2, double **h, 
             double **scaled_atom_coords, bond_prop **bond_props,
             int ift_switch, int n_slab, double *p_tangent_stretch,
             double *p_normal_stretch)
{
   int i_species, i, j, i_bond, i_mol, rel_1, rel_2, abs_1, abs_2, skip,
      v_rel, v_abs, v_skip, k;
   double x_bond, y_bond, z_bond, r2_bond, r_bond, fx_bond, fy_bond, fz_bond,
      elongation, dum1, dum2, p_norm, s_bond[NDIM], f_bond[NDIM], volume,
      bond[NDIM], spring, r_equil;
   int slab_index;
   double px = 0, tx = 0;

   volume = h[0][0]*h[1][1]*h[2][2];

   /* Zero forces. */
   for (i = 0; i < n_atoms; ++i)
      for (k = 0; k < NDIM; ++k)
         f_stretch[i][k] = 0.0;

   /* If eval_flag == 1, zero bond stretching contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_stretch = 0.0;
      for (i = 0; i < NDIM; ++i)
          for (k = 0; k < NDIM; ++k)
             at_stress_stretch[i][k] = 0.0;
   }

   if (ift_switch >0)
   for ( i = 0; i < n_slab; ++i){
        p_normal_stretch[i] = p_tangent_stretch[i] = 0.0;
       }

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom and bond vector label offsets. */
      skip = mol_first_atm[i_mol];

      /* Loop over bonds. */
      for (i_bond = 0; i_bond < n_bonds_per_mol[i_species]; ++i_bond) {

         /* Get relative labels of atoms that share bond. */
         rel_1 = temp_bonds_1[i_species][i_bond];
         rel_2 = temp_bonds_2[i_species][i_bond];


         /* Calculate absolute labels of atoms that share bond. */
         abs_1 = skip + rel_1;
         abs_2 = skip + rel_2;

         /* Get spring constant and equilibrium bond length. */
         spring = bond_props[i_species][i_bond].spring;
         r_equil = bond_props[i_species][i_bond].r_equil;


         /* Calculate bond vector and bond length. */
         for (k = 0; k < NDIM; ++k)
         {
            s_bond[k] = scaled_atom_coords[abs_2][k]
                      - scaled_atom_coords[abs_1][k];
            s_bond[k] -= NINT(s_bond[k]);
         }
         bond[0] = h[0][0] * s_bond[0] + h[0][1] * s_bond[1]
                  + h[0][2] * s_bond[2];
         bond[1] = h[1][1] * s_bond[1] + h[1][2] * s_bond[2];
         bond[2] = h[2][2] * s_bond[2];
         r2_bond = SQR(bond[0]) + SQR(bond[1]) + SQR(bond[2]);
         r_bond = sqrt(r2_bond);

         /* Calculate forces. */
         elongation = r_bond - r_equil;
         dum1 = spring * elongation;
         dum2 = dum1 / r_bond;
         for (k = 0; k < NDIM; ++k)
         f_bond[k] = dum2 * bond[k];
         for (k = 0; k < NDIM; ++k){
             f_stretch[abs_1][k] += f_bond[k];
             f_stretch[abs_2][k] -= f_bond[k];
         }
/* printf(" bond atom 1 %d 2 %d force x %g y %g z %g\n", abs_1,abs_2,f_stretch[abs_1][0],
        f_stretch[abs_1][1], f_stretch[abs_1][2]); */

         /* Add contributions to potential energy and atomic pressure
            tensor. */
         if (eval_flag) {
            *pe_stretch += dum1 * elongation;
            at_stress_stretch[0][0] -= bond[0] * f_bond[0];
            at_stress_stretch[1][1] -= bond[1] * f_bond[1];
            at_stress_stretch[2][2] -= bond[2] * f_bond[2];
            at_stress_stretch[0][1] -= bond[0] * f_bond[1];
            at_stress_stretch[0][2] -= bond[0] * f_bond[2];
            at_stress_stretch[1][2] -= bond[1] * f_bond[2];
         }
            /* Addition to compute normal and tangential stress profile */
             if (ift_switch > 0){
              px = bond[2] * f_bond[2]*0.5;
              tx = (bond[0] * f_bond[0] + bond[1] * f_bond[1])*0.5*0.5;
              slab_index = (int) ((scaled_atom_coords[abs_1][2] + 0.5) * n_slab);
              p_normal_stretch[slab_index] -= px;
              p_tangent_stretch[slab_index] -= tx;

              slab_index = (int) ((scaled_atom_coords[abs_2][2] + 0.5) * n_slab);
              p_normal_stretch[slab_index] -= px;
              p_tangent_stretch[slab_index] -= tx;
             }
      }
   }

   /* If eval_flag == 1, normalize bond stretching contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_stretch *= 0.5;
      p_norm =  1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_stretch[i][j] *= p_norm;
   }
    if (ift_switch >0 ){
     p_norm = n_slab/volume;
      for (i = 0; i <n_slab; ++i){
          p_normal_stretch[i] *=p_norm;
          p_tangent_stretch[i] *=p_norm;
      }
       
   }
}

/* Calculate bond angle bending interactions, using a cosine harmonic form
   for the bond angle bending potential. */

void bend_cos_harmonic(int period_switch, int eval_flag,
             double **f_bend, int *mol_first_atm, int *n_angles_per_mol,
             double **at_stress_bend, double *pe_bend, 
             int n_atoms, int n_mols, int *mol_species,
             angle_prop **angle_props, double **h,
             double **scaled_atom_coords, int ift_switch, int n_slab,
             double *p_tangent_bend, double *p_normal_bend)
{
   int i_species, i, j, k, i_angle, i_mol, rel_1, rel_2, rel_3,
      skip, abs_1, abs_2, abs_3, v_skip, v_abs_1, v_abs_2;
   double bond_1[NDIM], c_11, s_bond_1[NDIM], s_bond_2[NDIM],
      bond_2[NDIM], c_22, c_12, norm_1, norm_2,
      norm_3, cos_theta, delta_cos_theta, d_pe_d_cos_theta,
      d_cos_theta_d_b1[NDIM], d_cos_theta_d_b2[NDIM], 
      f_bond_1[NDIM], f_bond_2[NDIM], p_norm, volume, spring_bend,
      cos_theta_equil;

   int slab_index;
   double px = 0, tx = 0;

    volume = h[0][0] * h[1][1] * h[2][2] ;

   /* Zero forces. */
   for (i = 0; i < n_atoms; ++i) {
       for (k = 0; k < NDIM; ++k)
       f_bend[i][k] = 0.0;
   }

   /* If vars.eval_flag == 1, zero bond angle bending contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_bend = 0.0;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_bend[i][j] = 0.0;
   }

   if (ift_switch >0)
   for ( i = 0; i < n_slab; ++i){
        p_normal_bend[i] = p_tangent_bend[i] = 0.0;
       }
 
   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom and bond vector label offsets. */
      skip = mol_first_atm[i_mol];

      /* Loop over bond angles. */
      for (i_angle = 0; i_angle < n_angles_per_mol[i_species]; ++i_angle) {

         /* Get relative labels of atoms that share angle. */
         rel_1 = angle_props[i_species][i_angle].atom_1;
         rel_2 = angle_props[i_species][i_angle].atom_2;
         rel_3 = angle_props[i_species][i_angle].atom_3;

         /* Calculate absolute labels of atoms that share angle. */
         abs_1 = skip + rel_1;
         abs_2 = skip + rel_2;
         abs_3 = skip + rel_3;

         /* Get spring constant and cosine of equilibrium bond angle. */
         spring_bend =  angle_props[i_species][i_angle].spring_bend;
         cos_theta_equil = angle_props[i_species][i_angle].cos_theta_equil;


         /* Calculate bond vector and bond length. */
         for (k = 0; k < NDIM; ++k)
         {
            s_bond_1[k] = scaled_atom_coords[abs_2][k]
                      - scaled_atom_coords[abs_1][k];
            s_bond_2[k] = scaled_atom_coords[abs_3][k]
                      - scaled_atom_coords[abs_2][k];
            s_bond_1[k] -= NINT(s_bond_1[k]);
            s_bond_2[k] -= NINT(s_bond_2[k]);
         }
         bond_1[0] = h[0][0] * s_bond_1[0] + h[0][1] * s_bond_1[1]
                  + h[0][2] * s_bond_1[2];
         bond_1[1] = h[1][1] * s_bond_1[1] + h[1][2] * s_bond_1[2];
         bond_1[2] = h[2][2] * s_bond_1[2];
         bond_2[0] = h[0][0] * s_bond_2[0] + h[0][1] * s_bond_2[1]
                  + h[0][2] * s_bond_2[2];
         bond_2[1] = h[1][1] * s_bond_2[1] + h[1][2] * s_bond_2[2];
         bond_2[2] = h[2][2] * s_bond_2[2];
         c_11 = SQR(bond_1[0]) + SQR(bond_1[1]) + SQR(bond_1[2]);
         c_22 = SQR(bond_2[0]) + SQR(bond_2[1]) + SQR(bond_2[2]);

         /* Calculate forces. */
         c_12 = bond_1[0] * bond_2[0] + bond_1[1] * bond_2[1] 
                + bond_1[2] * bond_2[2];
         norm_1 = 1.0 / sqrt(c_11 * c_22);
         cos_theta = - norm_1 * c_12;
         delta_cos_theta = cos_theta - cos_theta_equil;
         d_pe_d_cos_theta = spring_bend * delta_cos_theta;
         norm_2 = c_12 / c_11;
         norm_3 = c_12 / c_22;
         d_cos_theta_d_b1[0] = norm_1 * (norm_2 * bond_1[0] - bond_2[0]);
         d_cos_theta_d_b1[1] = norm_1 * (norm_2 * bond_1[1] - bond_2[1]);
         d_cos_theta_d_b1[2] = norm_1 * (norm_2 * bond_1[2] - bond_2[2]);
         d_cos_theta_d_b2[0] = norm_1 * (norm_3 * bond_2[0] - bond_1[0]);
         d_cos_theta_d_b2[1] = norm_1 * (norm_3 * bond_2[1] - bond_1[1]);
         d_cos_theta_d_b2[2] = norm_1 * (norm_3 * bond_2[2] - bond_1[2]);
         for (k = 0; k < NDIM; ++k) {
             f_bond_1[k] = d_pe_d_cos_theta * d_cos_theta_d_b1[k];
             f_bond_2[k] = d_pe_d_cos_theta * d_cos_theta_d_b2[k];
         }
         for ( k = 0; k < NDIM; ++k) {
            f_bend[abs_1][k] += f_bond_1[k];
            f_bend[abs_2][k] += f_bond_2[k] - f_bond_1[k];
            f_bend[abs_3][k] -= f_bond_2[k];
             
         }

   /*    printf(" d_pe_d_cos_theta %lf \n", d_pe_d_cos_theta); */
         /* Add contributions to potential energy and atomic pressure tensor. */
         if (eval_flag) {
            *pe_bend += spring_bend * SQR(delta_cos_theta);
            at_stress_bend[0][0]
               -= bond_1[0] * f_bond_1[0] + bond_2[0] * f_bond_2[0];
            at_stress_bend[1][1]
               -= bond_1[1] * f_bond_1[1] + bond_2[1] * f_bond_2[1];
            at_stress_bend[2][2]
               -= bond_1[2] * f_bond_1[2] + bond_2[2] * f_bond_2[2];
            at_stress_bend[0][1]
               -= bond_1[0] * f_bond_1[1] + bond_2[0] * f_bond_2[1];
            at_stress_bend[0][2]
               -= bond_1[0] * f_bond_1[2] + bond_2[0] * f_bond_2[2];
            at_stress_bend[1][2]
               -= bond_1[1] * f_bond_1[2] + bond_2[1] * f_bond_2[2];
         }
            /* Addition to compute normal and tangential stress profile */
             if (ift_switch > 0){
              px = bond_1[2] * f_bond_1[2] + bond_2[2] * f_bond_2[2];
              tx = (bond_1[0] * f_bond_1[0] + bond_2[0] * f_bond_2[0]
                     + bond_1[1] * f_bond_1[1] + bond_2[1] * f_bond_2[1])*0.5;
              slab_index = (int) ((scaled_atom_coords[abs_1][2] + 0.5) * n_slab);
              p_normal_bend[slab_index] -= px;
              p_tangent_bend[slab_index] -= tx;

              slab_index = (int) ((scaled_atom_coords[abs_2][2] + 0.5) * n_slab);
              p_normal_bend[slab_index] -= px;
              p_tangent_bend[slab_index] -= tx;

              slab_index = (int) ((scaled_atom_coords[abs_3][2] + 0.5) * n_slab);
              p_normal_bend[slab_index] -= px;
              p_tangent_bend[slab_index] -= tx;
             }
      }
   }

   /* If vars.eval_flag == 1, normalize bond angle bending contributions to
      potential energy and pressure. */
   if (eval_flag) {
      *pe_bend *= 0.5;
      p_norm = 1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            at_stress_bend[i][j] *= p_norm;
   }
    if (ift_switch>0) {
        p_norm = n_slab/volume;
        for (i = 0; i <n_slab; ++i){
             p_normal_bend[i] *= p_norm;
             p_tangent_bend[i] *= p_norm;
        } 
    }
}


