/* Real-space Ewald summation routines for molecular modeling package. */

#include "build.h"

/* Calculate real-space contributions to electrostatic energy, virial,
   and forces, using an all-pairs neighbor search. */
void ewald_real_all_pairs(int n_atoms,
   double **scaled_atom_coords, double *atom_charges,
   double *a_1, double *a_2, double *a_3,
   double alpha, double r2_cutoff,
   double erf_tab_incr, double *erfc_sqrt_tab,
   double *pe_coul_real, double **vir_coul_real,
   double **atom_forces_coul_real)
{
   int i, j;
   double alpha2, two_alpha_over_sqrt_pi, one_over_erf_tab_incr,
      sx_i, sy_i, sz_i, fx_i_coul, fy_i_coul, fz_i_coul, charge_i,
      sx_sep, sy_sep, sz_sep, x_sep, y_sep, z_sep, r2_sep, r_sep,
      alpha2_r2, erfc_alpha_r, exp_minus_alpha2_r2, erfc_alpha_r_over_r,
      charge_prod, u_coul, f_coul, fx_coul, fy_coul, fz_coul;
   int i_tab;
   double x_tab, delt_tab, y1, y2;

   /* Compute various constants. */
   alpha2 = SQR(alpha);
   two_alpha_over_sqrt_pi = 2.0 * alpha / sqrt(PI);
   one_over_erf_tab_incr = 1.0 / erf_tab_incr;

   /* Zero accumulators. */
   *pe_coul_real = 0.0;
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         vir_coul_real[i][j] = 0.0;
   for (i = 0; i < n_atoms; ++i) {
      atom_forces_coul_real[i][0] = 0.0;
      atom_forces_coul_real[i][1] = 0.0;
      atom_forces_coul_real[i][2] = 0.0;
   }

   /* Loop over first member of pair. */
   for (i = 0; i < n_atoms - 1; ++i) {

      /* Get attributes of atom i. */
      sx_i = scaled_atom_coords[i][0];
      sy_i = scaled_atom_coords[i][1];
      sz_i = scaled_atom_coords[i][2];
      fx_i_coul = atom_forces_coul_real[i][0];
      fy_i_coul = atom_forces_coul_real[i][1];
      fz_i_coul = atom_forces_coul_real[i][2];
      charge_i = atom_charges[i];

      /* Loop over second member of pair. */
      for (j = i + 1; j < n_atoms; ++j) {

         /* Calculate scaled pair separation. */
         sx_sep = scaled_atom_coords[j][0] - sx_i;
         sy_sep = scaled_atom_coords[j][1] - sy_i;
         sz_sep = scaled_atom_coords[j][2] - sz_i;

         /* Apply nearest image convention. */
         sx_sep -= NINT(sx_sep);
         sy_sep -= NINT(sy_sep);
         sz_sep -= NINT(sz_sep);

         /* Calculate real pair separation. */
         x_sep = sx_sep * a_1[0] + sy_sep * a_2[0] + sz_sep * a_3[0];
         y_sep = sy_sep * a_2[1] + sz_sep * a_3[1];
         z_sep = sz_sep * a_3[2];
         r2_sep = SQR(x_sep) + SQR(y_sep) + SQR(z_sep);

         /* If the pair separation is less than r2_cutoff, compute
            pair interaction. */
         if (r2_sep < r2_cutoff) {

            /* Calculate pair interaction. The real-space part of the
               Ewald summation is truncated at r_cutoff. */
            r_sep = sqrt(r2_sep);
            alpha2_r2 = alpha2 * r2_sep;
            x_tab = alpha2_r2 * one_over_erf_tab_incr;
            i_tab = (int) x_tab;
            if (i_tab < N_ERF_TAB) {
               delt_tab = x_tab - i_tab;
               y1 = erfc_sqrt_tab[i_tab];
               y2 = erfc_sqrt_tab[i_tab + 1];
               erfc_alpha_r = y1 + delt_tab * (y2 - y1);
            }
            else
               erfc_alpha_r = erffc(alpha * r_sep);
            exp_minus_alpha2_r2 = exp(- alpha2_r2);
            erfc_alpha_r_over_r = erfc_alpha_r / r_sep;
            charge_prod = charge_i * atom_charges[j];
            u_coul = charge_prod * erfc_alpha_r_over_r;
            f_coul = charge_prod * (erfc_alpha_r_over_r
               + two_alpha_over_sqrt_pi * exp_minus_alpha2_r2) / r2_sep;
            fx_coul = x_sep * f_coul;
            fy_coul = y_sep * f_coul;
            fz_coul = z_sep * f_coul;
            fx_i_coul -= fx_coul;
            fy_i_coul -= fy_coul;
            fz_i_coul -= fz_coul;
            atom_forces_coul_real[j][0] += fx_coul;
            atom_forces_coul_real[j][1] += fy_coul;
            atom_forces_coul_real[j][2] += fz_coul;

            /* Add contributions to potential energy and virial. */
            *pe_coul_real += u_coul;
            vir_coul_real[0][0] -= x_sep * fx_coul;
            vir_coul_real[1][1] -= y_sep * fy_coul;
            vir_coul_real[2][2] -= z_sep * fz_coul;
            vir_coul_real[0][1] -= x_sep * fy_coul;
            vir_coul_real[0][2] -= x_sep * fz_coul;
            vir_coul_real[1][2] -= y_sep * fz_coul;
         }
      }
      atom_forces_coul_real[i][0] = fx_i_coul;
      atom_forces_coul_real[i][1] = fy_i_coul;
      atom_forces_coul_real[i][2] = fz_i_coul;
   }
}
