/* Link-cell routines for molecular modeling package. */

#include "build.h"
#include "ff.h"
#include "proto.h"

/* Set up data structures for cell-search algorithm. */
void set_up_cells_nophantom(double **h, int *nc, int *nc_p, int *nc_p_total, 
                  atom_cell *atom_cells, int kc, int period_switch, 
                  double r_nl, double r_off, int neigh_switch, 
                  int **first, int **last, 
                  int *offsets_x, int *offsets_y, int *offsets_z, int n_atoms,
                  double **scaled_atom_coords, double **atom_coords)
{
   int nc_p_old[NDIM], nc_p_total_old, cell_flag_1, cell_flag_2;
   int k, n;
   int i, ic_old;

   /* Save previous number of cells. */
   for (k = 0; k < NDIM; ++k )
        nc_p_old[k] = nc_p[k];
   nc_p_total_old = *nc_p_total;

   /* Calculate number of "real" cells. */
   if (neigh_switch == 2) {
      for (k = 0; k < NDIM; ++k)
         nc[k] = (int) (kc * h[k][k] / r_nl);
   }
   else {
      for (k = 0; k < NDIM; ++k)
         nc[k] = (int) (kc * h[k][k] / r_off);
   }

   /* Don't take into account 2 times the same atoms (i.e real + image):
      the number of cells must be at least 2 kc + 1. */
   for ( k = 0; k < NDIM; ++k) {
      if (nc[k] < 2 * kc + 1)
          error_exit("nc < 2 * kc + 1 in set_up_cells");
   }

   /* Calculate number of cells (real cells plus "phantom" or
      "padding" cells) in each direction. */
    for (k = 0; k < NDIM; ++k) {
      nc_p[k] = nc[k] ;
    }

   /* Calculate total number of cells (real cells plus "phantom" or
      "padding" cells). */
   *nc_p_total = nc_p[0] * nc_p[1] * nc_p[2];

   /* Set cell data structure update flags. */
   cell_flag_1 = *nc_p_total != nc_p_total_old;
   cell_flag_2 = nc_p[0] != nc_p_old[0] || nc_p[1] != nc_p_old[1]
                 || nc_p[2] != nc_p_old[2];

   /* Set up array of cell offsets. */

   if (cell_flag_2)
      set_up_offsets_nophantom(kc, nc_p, offsets_x, offsets_y, offsets_z);



   /* Allocate memory for labels of the first and last atoms in cell. */
   if (cell_flag_1) {
      if ((*first = realloc(*first, *nc_p_total * sizeof(int))) == NULL)
         error_exit("Out of memory in set_up_cells");
      if ((*last = realloc(*last, *nc_p_total * sizeof(int))) == NULL)
         error_exit("Out of memory in set_up_cells");
   }

   /* Initialize cell lists. */
   if (cell_flag_2)
      initialize_cells_nophantom( n_atoms, period_switch, h,
                      scaled_atom_coords, atom_coords,
                      *first, *last, *nc_p_total, nc_p,
                      atom_cells,  kc, nc);

/* for (i = 0; i < n_atoms; ++i){
      ic_old = atom_cells[i].cell;
printf("inatom i %d cell_index %d prev %d next %d cell %d real %d\n", i, ic_old, 
   atom_cells[i].prev, atom_cells[i].next, atom_cells[i].cell, atom_cells[i].real);
} 
*/
}

/* Set up array of cell offsets. */

void set_up_offsets_nophantom(int kc, int *nc_p, int *offsets_x, int *offsets_y, 
                              int *offsets_z)
{
   int icnt, ix, iy, iz, ic;
       

   /* Set up array of cell offsets used in computing forces. */
   icnt = 0;

   
   for (ix = -kc; ix <= kc; ++ix) {
       for (iy = -kc; iy <= kc; ++iy) {
          for (iz = -kc; iz <= kc; ++iz) {

               offsets_x[icnt] = ix;
               offsets_y[icnt] = iy;
               offsets_z[icnt] = iz;
               icnt = icnt +1; 
               }
        }
    }
}

/* Initialize cell lists. */
void initialize_cells_nophantom(int n_atoms, int period_switch, double **h, 
                      double **scaled_atom_coords, double **atom_coords, 
                      int *first, int *last, int nc_p_total, int *nc_p,
                      atom_cell *atom_cells, int kc, int *nc)
{
   int i, ic, i_phant, i_p, ic_p;

/*
for(i=0; i< n_atoms; ++i)
printf("ibperiod %d coordx %g y %g z %g\n", i, scaled_atom_coords[i][0], 
scaled_atom_coords[i][1], scaled_atom_coords[i][2]) ;
*/

   /* If period_switch == 1, apply periodic boundary conditions. */
   if (period_switch)
       periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, atom_coords);

/*
for(i=0; i< n_atoms; ++i)
printf("iafperiod %d coordx %g y %g z %g\n", i, scaled_atom_coords[i][0], 
scaled_atom_coords[i][1], scaled_atom_coords[i][2]) ;
*/

   /* Purge cell lists. */
   for (ic = 0; ic < nc_p_total; ++ic) {
      first[ic] = -1;
      last[ic] = -1;
   }
   for (i = 0; i < n_atoms; ++i) {
      atom_cells[i].prev = -1;
      atom_cells[i].next = -1;
      atom_cells[i].cell = -1;
      atom_cells[i].real = i;
   }
   if (period_switch) {
      for (i = n_atoms; i < 8 * n_atoms; ++i) {
         atom_cells[i].prev = -1;
         atom_cells[i].next = -1;
         atom_cells[i].cell = -1;
         atom_cells[i].real = -1;
      }
   }

   /* Assign atoms to cells. */
   for (i = 0; i < n_atoms; ++i) {

      /* Calculate cell index for atom i. */
      if (period_switch)
         ic = cell_index_period_nophantom(scaled_atom_coords[i], nc, nc_p, kc);

      /* Add entry to real cell list. */
      add_entry(i, ic, atom_cells, first, last);

   }
/*
     for (i = 0; i < n_atoms; ++i){
           ic = atom_cells[i].cell;
printf("atom i %d, cell_index %d coordx %g y %g z %g\n", i, ic ,
scaled_atom_coords[i][0], scaled_atom_coords[i][1], scaled_atom_coords[i][2]);
     }
*/
}

/* Update cell lists for all atoms. */
void update_cells_nophantom(int period_switch, int n_atoms, double **h, 
                  double **scaled_atom_coords, double **atom_coords,
                  atom_cell *atom_cells, int *first, int *last, 
                  int kc, int *nc, int *nc_p)
{
   int i, ic_old, ic, i_phant, i_p, ic_old_p, ic_p;

/* for (i = 0; i < n_atoms; ++i){
      ic_old = atom_cells[i].cell;
printf("atom i %d cell_index %d prev %d next %d cell %d real %d\n", i, ic_old, 
   atom_cells[i].prev, atom_cells[i].next, atom_cells[i].cell, atom_cells[i].real);
} 
exit(1);
*/
   /* If params.period_switch == 1, apply periodic boundary conditions. */
   if (period_switch)
    periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, atom_coords);


   /* Update cell lists as necessary. */
   for (i = 0; i < n_atoms; ++i) {

      /* Save previous cell index. */
      ic_old = atom_cells[i].cell;

      /* Calculate cell index for atom i. */
      if (period_switch)
         ic = cell_index_period_nophantom(scaled_atom_coords[i], nc, nc_p, kc);

/* printf("upatom i %d ic_old %d ic %d cellsx %d y %d z %d kc %d\n", i, ic_old,ic, 
n_p[0], n_p[1], n_p[2], kc);
 scaled_atom_coords[i][0], scaled_atom_coords[i][1], scaled_atom_coords[i][2]);*/ 
      /* if ic != ic_old, update cell lists. */
      if (ic != ic_old) {

         /* Update real cells. */
         remove_entry(i, ic_old, atom_cells, first, last);
         add_entry(i,  ic, atom_cells, first, last);

      }
   }
/*
 for (i = 0; i < n_atoms; ++i){
ic = atom_cells[i].cell;
printf("atom i %d, cell_index %d coordx %g y %g z %g\n", i, ic ,
scaled_atom_coords[i][0], scaled_atom_coords[i][1], scaled_atom_coords[i][2]);
}
*/
}


/* Calculate cell index for atom i, with periodic boundary conditions. */

/* Update cell lists for a single molecule. */
void update_cells_single_nophantom(int i_mol, int period_switch, int n_atoms, double **h,
                  double **scaled_atom_coords, double **atom_coords,
                  atom_cell *atom_cells, int *first, int *last,
                  int kc, int *nc,
                  int *nc_p, int *n_atoms_per_mol, int *mol_species,
                  int *mol_first_atm)
{
   int i_species, i, ic_old, ic, skip, i_phant, i_p, ic_old_p, ic_p;

   /* If period_switch == 1, apply periodic boundary conditions. */
   if (period_switch)
      periodic_boundary_conditions_single(i_mol, n_atoms_per_mol,
            mol_first_atm, mol_species, h, atom_coords, scaled_atom_coords);

   /* Get label of molecular species and label of first atom in molecule. */
   i_species = mol_species[i_mol];
   skip = mol_first_atm[i_mol];

   /* Update cell lists as necessary. */
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {

      /* Save previous cell index. */
      ic_old = atom_cells[i].cell;

      /* Calculate cell index for atom i. */
      if (period_switch)
         ic = cell_index_period_nophantom(scaled_atom_coords[i], nc, nc_p, kc);

      /* if ic != ic_old, update cell lists. */
      if (ic != ic_old) {

         /* Update real cells. */
         remove_entry(i, ic_old, atom_cells, first, last);
         add_entry(i,  ic, atom_cells, first, last);
      }
   }
}

int cell_index_period_nophantom(double *scaled_coords, int *nc, int *nc_p, int kc)
{
   int ix, iy, iz, ic;

   ix = (int) ((scaled_coords[0] + 0.5) * nc[0]);
   iy = (int) ((scaled_coords[1] + 0.5) * nc[1]);
   iz = (int) ((scaled_coords[2] + 0.5) * nc[2]);

   if ( scaled_coords[0] == 0.5) ix -=1;
   if ( scaled_coords[1] == 0.5) iy -=1;
   if ( scaled_coords[2] == 0.5) iz -=1;

   /* Avoid to go out of range when x=0.5 (and/or y=0.5, and/or z=0.5).
      Here we chose to fold it back to the first REAL cell index in the row. */
/*  if (ix > nc[0] - 1) ix = kc;
   if (iy > nc[1] - 1) iy = kc;
   if (iz > nc[2] - 1) iz = kc;
*/
   ic = ix + nc_p[0] * (iy + nc_p[1] * iz);

   return ic;
}

