/* Thermodynamics routines for molecular modeling package */

#include "build.h"
#include "ff.h"
#include "proto.h"


/* Calculate thermodynamic quantities for MD simulation. */
void thermodynamics(int ke_flag, int n_mols, double **mol_vels,
                    double **atom_vels, double *atom_mass, double *mol_mass,
                    int *mol_first_atm, int *n_atoms_per_mol, 
                    int n_free_atom, int n_free_mol, double boltzmann,
                    double temperature, double **h, 
                    double *pe_vdw_s, double *pe_stretch, int n_atoms, 
                    int *mol_species, double *at_temp, double *ke, double *pe,
                    double *te, double **at_stress_vdw_s,
                    double **at_stress_stretch, double **mol_stress_kin,
                    double **at_stress_kin, 
                    double **mol_stress, double **at_stress,
                    double **mol_stress_vdw, double **mol_stress_vdw_s, 
                    double *pe_bend, double **at_stress_bend,
                    int bend_switch)
{
double volume, density, mass_density , mol_ke, mol_temp, att_temp;
int i, j;

     *pe = 0.0;
     *te = 0.0;

volume = h[0][0]*h[1][1]*h[2][2];

   /* Calculate atomic and molecular kinetic energies and temperatures.
      If ke_flag == 1, these are calculated from the atomic and molecular
      velocities. Otherwise, their canonical values are used. */
    if (ke_flag) {
      center_of_mass_velocities(n_mols, mol_vels,
                 atom_vels, atom_mass, mol_mass,
                 mol_first_atm, n_atoms_per_mol, mol_species);
      kinetic_energy_and_temperature(n_mols, n_atoms, mol_species,
         mol_mass, atom_vels, mol_vels, boltzmann, n_free_atom, 
                       n_free_mol, atom_mass, at_temp, ke);
   }
   else {
      *ke = 0.5 * n_free_atom * boltzmann * temperature;
      *at_temp = temperature;
      mol_ke = 0.5 * n_free_mol * boltzmann * temperature;
      mol_temp = temperature;
   }

   /* Calculate total and intermolecular potential energies. */
/*   printf( " I am in thermodynamics pe_vdw_s %g pe_stretch %g, ke %g pe_bend %g\n", 
            *pe_vdw_s, *pe_stretch, *ke/n_atoms, *pe_bend); */


   if(bend_switch) 
   *pe = *pe_vdw_s + *pe_stretch + *pe_bend; 
   else
   *pe = *pe_vdw_s + *pe_stretch ; 

   /* Calculate total energy. */
   *te = *pe + *ke;

   /* Calculate total atomic and molecular stress tensors. */
   stress_tensors(ke_flag, at_stress_kin, mol_stress_kin,
                  atom_mass, atom_vels, mol_vels,
                  n_atoms, n_mols, mol_species, h,
                  mol_stress, at_stress, at_stress_vdw_s,
                  mol_stress_vdw, n_free_atom, n_free_mol,
                  boltzmann, temperature, at_stress_stretch, 
                  mol_stress_vdw_s, mol_mass, at_stress_bend);


   /* Calculate molecular density and mass density. */
   density = n_mols / volume;
}

/* Calculate center of mass velocities of molecules. */
void center_of_mass_velocities(int n_mols, double **mol_vels,
                 double **atom_vels, double *atom_mass, double *mol_mass,
                 int *mol_first_atm, int *n_atoms_per_mol, int *mol_species)
{
   int i_species, i_mol, i_atom, skip, abs, k;

   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Zero center of mass velocities. */
      for ( k = 0; k < NDIM; ++k)
      mol_vels[i_mol][k] = 0.0;

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Loop over atoms within molecule i_mol. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

         /* Calculate absolute label of atom. */
         abs = skip + i_atom;

         /* Add mass-weighted atomic velocities to center of mass velocities. */
        for (k = 0; k < NDIM;  ++k)
            mol_vels[i_mol][k] += atom_mass[abs] * atom_vels[abs][k];
      }

      /* Normalize center of mass velocities. */
      for (k = 0; k < NDIM;  ++k)
          mol_vels[i_mol][k] /= mol_mass[i_species];
   }
}

/* Calculate atomic and molecular kinetic energies and temperatures. */
  /* Number of Atomic degrees of freedom (n_free_atom = 3*n_atoms) */
  /* Number of molecular degrees of freedom (n_free_mol = 3*n_mols) */

void kinetic_energy_and_temperature(int n_mols, int n_atoms, int *mol_species,
            double *mol_mass, double **atom_vels, double **mol_vels,
            double boltzmann, int n_free_atom, int n_free_mol, double *atom_mass, 
            double *at_temp, double *ke)
{
   int i_species, i, i_mol;
   double mol_ke, mol_temp;

   *at_temp = 0.0;

   /* Calculate atomic kinetic energy and temperature. */
   *ke = 0.0;
   for (i = 0; i < n_atoms; ++i)
      *ke += atom_mass[i] * (SQR(atom_vels[i][0])
         + SQR(atom_vels[i][1]) + SQR(atom_vels[i][2]));
   *ke *= 0.5;
   *at_temp = 2.0 * *ke / (boltzmann * n_free_atom);

   /* Calculate molecular translational kinetic energy and molecular
      temperature. */
   if (n_mols == 1) {
      mol_ke = 0.0;
      mol_temp = 0.0;
   }
   else {
      mol_ke = 0.0;
      for (i_mol = 0; i_mol < n_mols; ++i_mol) {
         i_species = mol_species[i_mol];
         mol_ke += mol_mass[i_species]
            * (SQR(mol_vels[i_mol][0]) + SQR(mol_vels[i_mol][1]) 
                   + SQR(mol_vels[i_mol][2]));
      }
      mol_ke *= 0.5;
      mol_temp = 2.0 * mol_ke
         / (boltzmann * n_free_mol);
   }

}


/* Calculate total atomic and molecular stress tensors. */
void stress_tensors(int ke_flag, double **at_stress_kin, double **mol_stress_kin,
                    double *atom_mass, double **atom_vels, double **mol_vels,
                    int n_atoms, int n_mols, int *mol_species, double **h,
                    double **mol_stress, double **at_stress, double **at_stress_vdw_s,
                    double **mol_stress_vdw, int n_free_atom, int n_free_mol, 
                    double boltzmann, double temperature, 
                    double **at_stress_stretch, double **mol_stress_vdw_s,
                    double *mol_mass, double **at_stress_bend)
{
   int i_species, i, j, i_mol;
   double mass, p_norm, press_kin, volume;
   double at_press_kin, mol_press_kin;
   
   volume = h[0][0]*h[1][1]*h[2][2];

   /* Calculate kinetic contributions to atomic and molecular stress tensors.
      If ke_flag == 1, these are calculated from the atomic and molecular
      velocities. Otherwise, their canonical values are used. */
   if (ke_flag) {
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            at_stress_kin[i][j] = 0.0;
            mol_stress_kin[i][j] = 0.0;
         }
      for (i = 0; i < n_atoms; ++i) {
         mass = atom_mass[i];
         at_stress_kin[0][0] += mass * SQR(atom_vels[i][0]);
         at_stress_kin[1][1] += mass * SQR(atom_vels[i][1]);
         at_stress_kin[2][2] += mass * SQR(atom_vels[i][2]);
         at_stress_kin[0][1] += mass * atom_vels[i][0]*atom_vels[i][1];
         at_stress_kin[0][2] += mass * atom_vels[i][0]*atom_vels[i][2];
         at_stress_kin[1][2] += mass * atom_vels[i][1]*atom_vels[i][2];
      }
      for (i_mol = 0; i_mol < n_mols; ++i_mol) {
         i_species = mol_species[i_mol];
         mass = mol_mass[i_species];
         mol_stress_kin[0][0] += mass * SQR(mol_vels[i_mol][0]);
         mol_stress_kin[1][1] += mass * SQR(mol_vels[i_mol][1]);
         mol_stress_kin[2][2] += mass * SQR(mol_vels[i_mol][2]);
         mol_stress_kin[0][1] += mass * mol_vels[i_mol][0]*mol_vels[i_mol][1];
         mol_stress_kin[0][2] += mass * mol_vels[i_mol][0]*mol_vels[i_mol][2];
         mol_stress_kin[1][2] += mass * mol_vels[i_mol][1]*mol_vels[i_mol][2];
      }
      p_norm = 1.0 / volume;
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            at_stress_kin[i][j] *= p_norm;
            mol_stress_kin[i][j] *= p_norm;
         }
   }
   else {
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            at_stress_kin[i][j] = 0.0;
            mol_stress_kin[i][j] = 0.0;
         }
      press_kin = n_free_atom * boltzmann
         * temperature / (3.0 * volume);
      for (i = 0; i < 3; ++i)
         at_stress_kin[i][i] = press_kin;
      press_kin = n_free_mol * boltzmann
         * temperature / (3.0 * volume);
      for (i = 0; i < 3; ++i)
         mol_stress_kin[i][i] = press_kin;
   }

   /* Calculate total atomic stress tensor. */
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         at_stress[i][j] = at_stress_kin[i][j] + at_stress_vdw_s[i][j] 
            + at_stress_stretch[i][j] + at_stress_bend[i][j];

   for (i = 1; i < 3; ++i)
      for (j = 0; j < i; ++j)
         at_stress[i][j] = at_stress_kin[j][i]
            + at_stress_vdw_s[j][i] 
            + at_stress_stretch[j][i]+ at_stress_bend[j][i];

   /* Calculate isotropic atomic pressure. */
/*   at_press_kin = at_press_vdw_s = at_press_str = at_press = 0.0;

   for (i = 0; i < 3; ++i) {
      at_press_kin += at_stress_kin[i][i] / 3.0;
      at_press_vdw_s += at_stress_vdw_s[i][i] / 3.0;
      at_press += at_stress[i][i] / 3.0;
   }

*/
   /* Calculate total molecular stress tensor. */
   for (i = 0; i < 3; ++i)
      for (j = 0; j < 3; ++j) {
         mol_stress_vdw[i][j] = mol_stress_vdw_s[i][j];
      }
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         mol_stress[i][j] = mol_stress_kin[i][j] + mol_stress_vdw[i][j];
   for (i = 1; i < 3; ++i)
      for (j = 0; j < i; ++j)
         mol_stress[i][j] = mol_stress_kin[j][i] + mol_stress_vdw[i][j];

   /* Calculate isotropic molecular pressure. */
/*   mol_press_kin = mol_press_vdw_s = mol_press_vdw = mol_press = 0.0;
   for (i = 0; i < 3; ++i) {
      mol_press_kin += mol_stress_kin[i][i] / 3.0;
      mol_press_vdw_s += mol_stress_vdw_s[i][i] / 3.0;
      mol_press_vdw += mol_stress_vdw[i][i] / 3.0;
      mol_press += mol_stress[i][i] / 3.0;
   }
*/
}
