/* Initialization routines. */

#include "build.h"

/* Carry out an MC cycle. */
void mc_cycle(int n_mols, int n_atoms, int vdw_switch, int period_switch, 
      long *i_ran, int *mol_first_atm, int *n_atoms_per_mol, int *mol_species,
      int *atom_rel, int *atom_mol, int *atom_type, double ***temp_atm_pos, 
      double **h, double **h_inv, double aspect_xy, double aspect_xz, 
      double **mol_coords, double **scaled_mol_coords, double **atom_coords, 
      double **scaled_atom_coords, double **rel_atom_coords, int ***exclusions,
      int **comb_pot, double ***comb_par, double gamma, 
      double three_gamma_over_two, double two_over_gamma_cubed, double dr_max, 
      double *ds_mol_max, double dang_mol_max, double dh_max, int reshape_axis, 
      double mol_max_length, double r2_on, 
      double r2_off, double r_off, double r_nl, double r2_nl, double **s_old, 
      double **old, double **rel_old, double **mol_old, int press_switch, 
      double reduce_boltz, double press, int neigh_switch, 
      int phant_switch, int **first, int **last, 
      phantom_cell **phantoms, int noffsets, int *nc, int *nc_p, 
      int *nc_p_total, int kc, int *phantom_skip, atom_cell *atom_cells, 
      int *offsets, nl_entry **nl_head, nl_entry **nl_tail, double **atom_move,
      int n_reshape, double *pe_vdw_s, int *n_mol_moves, 
      int *n_mol_accept, int *n_reshape_moves, int *n_reshape_accept, 
      double **ds_opt,
      double **at_stress_vdw_s, double **mol_stress_vdw_s, double **vdw_s,
      int *offsets_x, int *offsets_y, int *offsets_z)

{
   int n_moves, i_move, i_type;

   /* n_moves is the total number of MC moves per cycle. */
   n_moves = n_mols + n_reshape;

   /* Loop over trial MC moves. One cycle consists of n_mols attempted
      reorientation/translation moves plus n_reshape constant-volume
      changes in unit cell shape, on average. */
   *n_mol_moves = *n_mol_accept = *n_reshape_moves = *n_reshape_accept = 0;
   for (i_move = 0; i_move < n_moves; ++i_move) {
      i_type = (int) (ran3(i_ran) * n_moves);

      if (i_type < n_mols) {

         ++(*n_mol_moves);
         (*n_mol_accept) += mol_move(period_switch, vdw_switch, i_type, 
            n_mols, n_atoms, i_ran, mol_first_atm, n_atoms_per_mol, 
            mol_species, atom_rel, atom_mol, atom_type, temp_atm_pos, h, 
            h_inv, mol_coords, scaled_mol_coords, atom_coords, 
            scaled_atom_coords, rel_atom_coords, exclusions, 
            comb_pot, comb_par, gamma, three_gamma_over_two, 
            two_over_gamma_cubed, ds_mol_max, 
            dang_mol_max, r2_on, r_off, r2_off, r_nl, r2_nl, s_old, old, 
            rel_old, reduce_boltz, neigh_switch, phant_switch, *first, *last, 
            *phantoms, noffsets, nc,
            nc_p, kc, phantom_skip, atom_cells, offsets, nl_head, nl_tail, 
            atom_move, pe_vdw_s, ds_opt, offsets_x, offsets_y, offsets_z);
     }
     else {
         ++(*n_reshape_moves);
         if (press_switch)
         (*n_reshape_accept) += reshape_move_press(period_switch, vdw_switch, 
            reshape_axis, 
            mol_max_length, n_mols, n_atoms, i_ran, mol_first_atm, 
            n_atoms_per_mol, mol_species, atom_rel, atom_mol, atom_type,
            h, h_inv, aspect_xy, aspect_xz, mol_coords, scaled_mol_coords, 
            atom_coords, scaled_atom_coords, rel_atom_coords, exclusions, 
            comb_pot, comb_par, gamma, three_gamma_over_two, 
            two_over_gamma_cubed, 
            dh_max, dr_max, ds_mol_max, r2_on, r2_off, r_off, r2_nl, r_nl, 
            mol_old, reduce_boltz, press, neigh_switch, phant_switch, first, 
            last, phantoms, noffsets, nc, nc_p, nc_p_total, kc, phantom_skip, 
            atom_cells, offsets, nl_head, nl_tail, atom_move, pe_vdw_s, 
            at_stress_vdw_s, mol_stress_vdw_s, vdw_s, offsets_x, offsets_y, 
            offsets_z);
	 else
         (*n_reshape_accept) += reshape_move(period_switch, vdw_switch, 
            reshape_axis, mol_max_length, n_mols, n_atoms, i_ran, 
            mol_first_atm, n_atoms_per_mol, mol_species, atom_rel, atom_mol, 
            atom_type, h, h_inv, mol_coords, scaled_mol_coords, atom_coords, 
            scaled_atom_coords, rel_atom_coords, exclusions, comb_pot, 
            comb_par, gamma, three_gamma_over_two, two_over_gamma_cubed, 
            dh_max, dr_max, ds_mol_max, r2_on, r2_off, r_off, r2_nl, r_nl, 
            mol_old, 
            reduce_boltz, neigh_switch, phant_switch, first, last, phantoms, 
            noffsets, nc, nc_p, nc_p_total, kc, phantom_skip, atom_cells, 
            offsets, nl_head, nl_tail, atom_move, pe_vdw_s, at_stress_vdw_s, 
            mol_stress_vdw_s, vdw_s, offsets_x, offsets_y, offsets_z);
    }
  }
}

/***************************************************************************/
/* Make a combined - position, orientation - MC move of a randomly chosen 
   molecule.

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int mol_move(int period_switch, int vdw_switch, int i_mol, int n_mols, 
      int n_atoms, long *i_ran, int *mol_first_atm, int *n_atoms_per_mol, 
      int *mol_species, int *atom_rel, int *atom_mol, int *atom_type, 
      double ***temp_atm_pos, double **h, double **h_inv, double **mol_coords, 
      double **scaled_mol_coords, double **atom_coords, 
      double **scaled_atom_coords, double **rel_atom_coords, 
      int ***exclusions, int **comb_pot, double ***comb_par, double gamma, 
      double three_gamma_over_two, 
      double two_over_gamma_cubed, double *ds_mol_max, double dang_mol_max, 
      double r2_on, double r_off, double r2_off, double r_nl, double r2_nl, 
      double **s_old, double **old, double **rel_old, double reduce_boltz, 
      int neigh_switch, 
      int phant_switch, int *first, int *last, phantom_cell *phantoms, 
      int noffsets, int *nc, int *nc_p, int kc, int *phantom_skip, 
      atom_cell *atom_cells, int *offsets, nl_entry **nl_head, 
      nl_entry **nl_tail, double **atom_move, 
      double *pe_vdw_s, double **ds_opt, int *offsets_x, int *offsets_y, 
      int *offsets_z)
{
   int j_mol, skip, abs, i_species, i, j, move_accept, flag_nl;
   double q[4], random[4], ds[NDIM], s1, s2, mag2, mag, factor, x_rot, y_rot, 
      z_rot, mol_old[NDIM], s_mol_old[NDIM], a11, a12, a13, a21, a22, a23, 
      a31, a32, a33, delta_pe, pe_vdw_s_old, pe_vdw_s_new, old_move[200][3]; 

double pe_vdw_s_single_old, pe_vdw_s_single_new, delta_pe_p, diff, 
   pe_vdw_s_old_chk;
    
      /* Set acceptance counter to zero. */
      move_accept = 0;

      /* Get label of molecular species and label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];

      /* Compute contribution to the potential energy from molecule i_mol
         before the move. If neigh_switch = 0, we are using a N^2 search 
         otherwise if neigh_switch = 1, we are using a cell list (with or
         without phantom cells and atoms). */  
      if (neigh_switch == 0) {
          lj_s_period_single(skip, i_mol, mol_species, atom_rel, atom_mol, 
            atom_type, n_atoms, n_atoms_per_mol, exclusions, comb_pot, 
            comb_par, h, r2_on, r2_off, scaled_atom_coords, gamma, 
            three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_s_old);
      }
      else if (neigh_switch == 1) {
      if (phant_switch == 1)
          wca_period_single_phantom(skip, i_mol, mol_species, atom_rel,
            atom_mol, atom_type, n_atoms, n_atoms_per_mol, exclusions, 
            comb_pot, comb_par, r2_on, r2_off, h, 
            scaled_atom_coords, gamma, three_gamma_over_two, 
            two_over_gamma_cubed, &pe_vdw_s_old, period_switch, first, last, 
            phantoms, noffsets, nc_p, kc, phantom_skip, nc, atom_cells, 
            offsets, atom_coords, mol_first_atm);
/*      else
          lj_s_period_single_phantom(skip, i_mol, mol_species, atom_rel, 
            atom_mol, atom_type, n_atoms, n_atoms_per_mol, exclusions, 
            comb_pot, comb_par, h, r2_on,
            r2_off, scaled_atom_coords, gamma, three_gamma_over_two,
            two_over_gamma_cubed, &pe_vdw_s_old,
            period_switch, first, last, phantoms, noffsets, nc_p, kc, 
            phantom_skip, nc, atom_cells, offsets, atom_coords, mol_first_atm);
*/
      else {
      if (phant_switch == 0)
      lj_s_period_single_nophantom(skip, i_mol, mol_species, atom_rel, 
         atom_mol, atom_type, n_atoms, n_atoms_per_mol, exclusions, 
         comb_pot, comb_par, h, r2_on, r2_off, scaled_atom_coords, gamma, 
         three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_s_old,
         period_switch, first, last, noffsets, nc_p, kc, 
         nc, atom_cells, offsets_x, offsets_y, offsets_z, atom_coords, 
         mol_first_atm);
         }
      }
      else {  

       /* Update neighbor lists flag. */
/*     flag_nl = 0;

       lj_s_nl_period_phantom_single(skip, i_mol, mol_species, atom_rel, 
           atom_mol, n_atoms, n_atoms_per_mol, exclusions, comb_sigma2,
           comb_four_eps, h, r2_on, r2_off, scaled_atom_coords, gamma, 
           three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_s_old, 
           period_switch, first, last, phantoms, noffsets, nc_p, kc, 
           phantom_skip, nc, atom_cells, offsets, atom_coords, nl_head, 
           nl_tail, r2_nl, rel_atom_coords, mol_first_atm);
*/
      }

      /* Store positions of atoms in mol_i before the move. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) 
         for (j = 0; j < NDIM; ++j)
           {
           old[i][j] = atom_coords[skip + i][j];
           s_old[i][j] = scaled_atom_coords[skip + i][j];
           rel_old[i][j] = rel_atom_coords[skip + i][j];
           if (neigh_switch == 2) 
              old_move[i][j] = atom_move[skip + i][j];
           }

      /* Store centers of mass of the molecules. */
      for (j = 0; j < NDIM; ++j) {
        mol_old[j] = mol_coords[i_mol][j];
        s_mol_old[j] = scaled_mol_coords[i_mol][j];
        } 

      /* Generate new scaled position of i_mol. */
      for (j = 0; j < NDIM; ++j)
        {
        ds[j] = (2.0 * ran3(i_ran) - 1.0) * ds_mol_max[j];
        scaled_mol_coords[i_mol][j] += ds[j]; 
        scaled_mol_coords[i_mol][j] -= NINT(scaled_mol_coords[i_mol][j]);
        }
      /* Compute new position of i_mol. */
      mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0]
                           + h[0][1] * scaled_mol_coords[i_mol][1]
                           + h[0][2] * scaled_mol_coords[i_mol][2];
      mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1]
                           + h[1][2] * scaled_mol_coords[i_mol][2];
      mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

      /* Generate a finite but small angular rotation.
         The quaternion corresponding to the unity matrix
         is qI = (1.0, 0.0, 0.0, 0.0). We generate a quaternion
         close to the unity matrix. */ 
      do {
      mag2 = 0.0;
        for (i = 1; i < 4; ++i){ 
           q[i] = (2.0 * ran3(i_ran) - 1.0) * dang_mol_max;
           mag2 += SQR(q[i]);
        }
      } while (mag2 > 1.0);
      q[0] = sqrt(1 - mag2); 

      /* Compute the elements of the rotation matrix. */
      a11 = SQR(q[0]) + SQR(q[1]) - SQR(q[2]) - SQR(q[3]);
      a12 = 2.0 * q[1] * q[2] + 2.0 * q[0] * q[3];
      a13 = 2.0 * q[1] * q[3] - 2.0 * q[0] * q[2];
      a21 = 2.0 * q[1] * q[2] - 2.0 * q[0] * q[3];
      a22 = SQR(q[0]) - SQR(q[1]) + SQR(q[2]) - SQR(q[3]);
      a23 = 2.0 * q[2] * q[3] + 2.0 * q[0] * q[1];
      a31 = 2.0 * q[1] * q[3] + 2.0 * q[0] * q[2];
      a32 = 2.0 * q[2] * q[3] - 2.0 * q[0] * q[1];
      a33 = SQR(q[0]) - SQR(q[1]) - SQR(q[2]) + SQR(q[3]);

      /* Calculate new relative atomic positions. The rotations are performed
         in the unit-cell-based reference systems. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize relative atomic position. */
            x_rot = rel_atom_coords[abs][0];
            y_rot = rel_atom_coords[abs][1];
            z_rot = rel_atom_coords[abs][2];

            /* Rotate and store relative atomic positions. */
            rel_atom_coords[abs][0] = a11 * x_rot + a12 * y_rot + a13 * z_rot;
            rel_atom_coords[abs][1] = a21 * x_rot + a22 * y_rot + a23 * z_rot;
            rel_atom_coords[abs][2] = a31 * x_rot + a32 * y_rot + a33 * z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] 
                                  + rel_atom_coords[abs][0];
            atom_coords[abs][1] = mol_coords[i_mol][1] 
                                  + rel_atom_coords[abs][1];
            atom_coords[abs][2] = mol_coords[i_mol][2] 
                                  + rel_atom_coords[abs][2];
         }

         /* Calculate scaled atomic coordinates and apply periodic
            boundary condition for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
            mol_species, h_inv, atom_coords, scaled_atom_coords);
         periodic_boundary_conditions_single(i_mol, n_atoms_per_mol,
            mol_first_atm, mol_species, h, atom_coords, scaled_atom_coords);

         /* Compute contribution to the potential from the new position
            of i_mol. */
         if (neigh_switch == 0) { /*N^2 search */
          lj_s_period_single(skip, i_mol, mol_species, atom_rel, atom_mol, 
            atom_type, n_atoms, n_atoms_per_mol, exclusions, comb_pot, 
            comb_par, h, r2_on, r2_off, scaled_atom_coords, gamma, 
            three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_s_new);
         }
         else if (neigh_switch == 1) {
           if (phant_switch == 1) {
             update_cells_single(i_mol, period_switch,  n_atoms, h,
               scaled_atom_coords, atom_coords, atom_cells, first, last,
               phantoms,  kc, phantom_skip, nc, nc_p, n_atoms_per_mol, 
               mol_species, mol_first_atm);

             update_phantoms_single(i_mol, n_atoms, atom_cells, phantom_skip, 
               phantoms, atom_coords, mol_first_atm, mol_species, 
               n_atoms_per_mol);

               wca_period_single_phantom(skip, i_mol, mol_species, atom_rel,
                 atom_mol, atom_type, n_atoms, n_atoms_per_mol, exclusions, 
                 comb_pot, comb_par, r2_on, r2_off, h, scaled_atom_coords, 
                 gamma, three_gamma_over_two, two_over_gamma_cubed, 
                 &pe_vdw_s_new, period_switch, first, last, phantoms, noffsets,
                 nc_p, kc, phantom_skip, nc, atom_cells, offsets, atom_coords, 
                 mol_first_atm);
 /*            else
             lj_s_period_single_phantom(skip, i_mol, mol_species, atom_rel, 
               atom_mol, n_atoms, n_atoms_per_mol, exclusions, comb_sigma2, 
               comb_four_eps, h, r2_on, r2_off, scaled_atom_coords, gamma, 
               three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_s_new,
               period_switch, first, last, phantoms, noffsets, nc_p, kc, 
               phantom_skip, nc, atom_cells, offsets, atom_coords, 
               mol_first_atm);
*/
            }
            else{
               if (phant_switch == 0) {
                 update_cells_single_nophantom(i_mol, period_switch, n_atoms, 
                   h, scaled_atom_coords, atom_coords, atom_cells, first, last,
                   kc, nc, nc_p, n_atoms_per_mol, mol_species, mol_first_atm);

                 lj_s_period_single_nophantom(skip, i_mol, mol_species, 
                    atom_rel, atom_mol, atom_type, n_atoms,
                    n_atoms_per_mol, exclusions, comb_pot, comb_par, 
                    h, r2_on, r2_off, scaled_atom_coords, gamma, 
                    three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_s_new,
                    period_switch, first, last, noffsets, nc_p, kc,
                    nc, atom_cells, offsets_x, offsets_y, offsets_z, 
                    atom_coords, mol_first_atm);

               }
            }
         }
      else {  

        /* Check if the atoms in the molecule have moved more than the half 
           skin distance since the last neighbor lists update. */
        neigh_check(i_species, n_atoms_per_mol, h, r_off, r_nl, skip, 
           scaled_atom_coords, s_old, atom_move, &flag_nl);

        /* Update neighbor lists if necessary. */
        if (flag_nl)
            neighbor_lists_period( mol_species, atom_rel, atom_mol, n_atoms,
               n_atoms_per_mol, exclusions, h, r2_on,  r2_off,
               scaled_atom_coords, period_switch, first, last, phantoms,
               noffsets, nc_p,  kc, phantom_skip, nc, atom_cells, offsets,
               atom_coords, atom_move, nl_head, nl_tail, r2_nl);

/*        lj_s_nl_period_phantom_single(skip, i_mol, mol_species, atom_rel, 
           atom_mol, n_atoms, n_atoms_per_mol, exclusions, comb_sigma2,
           comb_four_eps, h, r2_on, r2_off, scaled_atom_coords, gamma, 
           three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_s_new, 
           period_switch, first, last, phantoms, noffsets, nc_p, kc, 
           phantom_skip, nc, atom_cells, offsets, atom_coords, nl_head, 
           nl_tail, r2_nl, rel_atom_coords, mol_first_atm);
*/
        }

         /* Compute change in potential energy and accept the move 
            using Metropolis criterion. */
         delta_pe = pe_vdw_s_new - pe_vdw_s_old;

         if ((delta_pe <= 0.0) || (exp(-delta_pe * reduce_boltz) > ran3(i_ran)))
            {
	    *pe_vdw_s += delta_pe;
            move_accept = 1;
            /* Add contribution to displacement for optimization. */
            for (j = 0; j < NDIM; ++j)
              ds_opt[i_mol][j] += ds[j];
            if ((neigh_switch == 2) && (flag_nl))
              zero_move(n_atoms, atom_move);
            }
         else
            {
            for (i = 0; i < n_atoms_per_mol[i_species]; ++i) 
              for (j = 0; j < NDIM; ++j)
              {
              scaled_atom_coords[skip + i][j] = s_old[i][j];
              atom_coords[skip + i][j] = old[i][j];
              rel_atom_coords[skip + i][j] = rel_old[i][j];
              if ((neigh_switch == 2) && (!flag_nl))
                atom_move[skip + i][j] = old_move[i][j];
              }
            for (j = 0; j < NDIM; ++j)
              {
              mol_coords[i_mol][j] = mol_old[j];
              scaled_mol_coords[i_mol][j] = s_mol_old[j];
              } 
            /* Update the cell list if we are using a cell or neighbor list. */
            if (neigh_switch == 1) {
              if ( phant_switch == 1){
              update_cells_single(i_mol, period_switch, n_atoms, h, 
                scaled_atom_coords, atom_coords, atom_cells, first, last, 
                phantoms, kc, phantom_skip, nc, nc_p, n_atoms_per_mol, 
                mol_species, mol_first_atm);

              update_phantoms_single(i_mol, n_atoms, atom_cells, phantom_skip, 
                phantoms, atom_coords, mol_first_atm, mol_species, 
                n_atoms_per_mol);
             }else{
            if ( phant_switch == 0)
            update_cells_single_nophantom(i_mol, period_switch, n_atoms, h,
                scaled_atom_coords, atom_coords, atom_cells, first, last,
                kc, nc, nc_p, n_atoms_per_mol,
                mol_species, mol_first_atm);
            }
              }
             else if ((neigh_switch == 2) && (flag_nl)) {
             neighbor_lists_period( mol_species, atom_rel, atom_mol,  
                   n_atoms, n_atoms_per_mol, exclusions, h, r2_on, r2_off, 
                   scaled_atom_coords, period_switch, first, last, phantoms,
                   noffsets, nc_p,  kc, phantom_skip, nc, atom_cells, offsets, 
                   atom_coords, atom_move, nl_head, nl_tail, r2_nl);

 /*       neighbor_lists_period_single(i_mol, mol_species, atom_rel, atom_mol, 
               mol_first_atm, n_atoms,
               n_atoms_per_mol, exclusions, h, r2_on,  r2_off,
               scaled_atom_coords, period_switch, first, last, phantoms,
               noffsets, nc_p,  kc, phantom_skip, nc, atom_cells, offsets,
               atom_coords, atom_move, nl_head, nl_tail, r2_nl);
*/
                zero_move(n_atoms, atom_move);
                }
            }
   /* Return move_accept. */
   return(move_accept);
}

/***************************************************************************/
int reshape_move(int period_switch, int vdw_switch, int reshape_axis, 
      double mol_max_length, int n_mols, int n_atoms, 
      long *i_ran, int *mol_first_atm, int *n_atoms_per_mol, int *mol_species,
      int *atom_rel, int *atom_mol, int *atom_type, double **h, double **h_inv, 
      double **mol_coords, double **scaled_mol_coords, double **atom_coords,
      double **scaled_atom_coords, double **rel_atom_coords, int ***exclusions,
      int **comb_pot, double ***comb_par, double gamma, 
      double three_gamma_over_two, double two_over_gamma_cubed, double dh_max, 
      double dr_max, double *ds_mol_max, double r2_on, double r2_off, 
      double r_off, double r2_nl, double r_nl, double **mol_old, 
      double reduce_boltz, 
      int neigh_switch, int phant_switch, int **first, int **last, 
      phantom_cell **phantoms, int noffsets, int *nc, int *nc_p, 
      int *nc_p_total, int kc, int *phantom_skip, atom_cell *atom_cells, 
      int *offsets, nl_entry **nl_head, nl_entry **nl_tail, double **atom_move,
      double *pe_vdw, double **at_stress_vdw_s, double **mol_stress_vdw_s, 
      double **vdw_s, int *offsets_x, int *offsets_y, int *offsets_z)

{
   int i, j, k, i0, i1, i2, i_axis, i_mol, i_atom, abs, i_species, skip, 
      reshape_accept;
   double h_old[NDIM][NDIM], h_inv_old[NDIM][NDIM], dh[NDIM], sum, 
      stress_old[NDIM][NDIM], pe_vdw_old, pe_vdw_new, delta_pe;

   /* Set acceptance counter to zero. */
   reshape_accept = 0;

   /* Get total potential energy before the reshaping attempt. */
   pe_vdw_old = *pe_vdw;

   /* Get box coordinates. */
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j){ 
        h_old[i][j] = h[i][j];
        h_inv_old[i][j] = h_inv[i][j];
      }

   /* Get stress tensor. */
   for (i = 0; i < 3; ++i)
      for (j = 0; j < 3; ++j)
        stress_old[i][j] = mol_stress_vdw_s[i][j];
      
   /* Determine unique axis for reshaping move. */
   i0 = reshape_axis;
   if (reshape_axis == 3) {
      i0 = (int) (3.0 * ran3(i_ran));
      if (i0 == 3) i0 = 0;
   }
   i1 = (i0 + 1) % 3;
   i2 = (i0 + 2) % 3;

   /* Generate random change in box dimensions keeping a constant
   volume. */
   dh[0] = (2.0 * ran3(i_ran) - 1.0) * dh_max;
   h[i0][i0] += dh[0];
   sum = (h[i1][i1] + h[i2][i2]) * 0.5;
   dh[1] = dh[2] = - sum + sqrt(SQR(sum) - dh[0] * h[i1][i1] * h[i2][i2] 
                   / h[i0][i0]);
   h[i1][i1] += dh[1];
   h[i2][i2] += dh[2];

   /* Calculate quantities that depend on box dimensions and exit if
      system is too small. */
   box_dimensions(h, h_inv, period_switch, r_off, mol_max_length);

   /* Scale molecular center of mass coordinates. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Store centers of mass of the molecules before reshaping. */
      for (k = 0; k < NDIM; ++k)
         mol_old[i_mol][k] = mol_coords[i_mol][k];
     
      /* Compute new center of mass after box reshaping. */ 
      mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0]
         + h[0][1] * scaled_mol_coords[i_mol][1]
         + h[0][2] * scaled_mol_coords[i_mol][2];
      mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1]
         + h[1][2] * scaled_mol_coords[i_mol][2];
      mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Shift atomic positions. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

        /* Get absolute label of the atom. */
        abs = skip + i_atom;

        /* Compute new atomic positions. Remember that the relative atom
        coordinates are unfolded. */
        atom_coords[abs][0] = mol_coords[i_mol][0] + rel_atom_coords[abs][0];
        atom_coords[abs][1] = mol_coords[i_mol][1] + rel_atom_coords[abs][1];
        atom_coords[abs][2] = mol_coords[i_mol][2] + rel_atom_coords[abs][2];
      }
   }

   /* Compute scaled atom positions and apply periodic boundary conditions. */
   if (period_switch) {
     scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);
     periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, atom_coords);
   }

   /* Compute new total potential energy. */
   if (neigh_switch == 0) {
       lj_s_period(mol_species, atom_rel, atom_mol, atom_type, n_atoms,
          n_atoms_per_mol, exclusions, comb_pot, comb_par,
          h, r2_on, r2_off, scaled_atom_coords, scaled_mol_coords,
          rel_atom_coords, gamma, three_gamma_over_two, 
          two_over_gamma_cubed, &pe_vdw_new, vdw_s, mol_stress_vdw_s);

   }
   else if (neigh_switch == 1) {
   /* Set up cell list with phantom cells. */
     if (phant_switch == 1) {
       set_up_cells_phantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
         period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
         n_atoms, scaled_atom_coords, atom_coords, phantoms, phantom_skip);

       update_cells(period_switch, n_atoms, h, scaled_atom_coords, 
         atom_coords, atom_cells, *first, *last, *phantoms, kc, 
         phantom_skip, nc, nc_p);

       update_phantoms(n_atoms, atom_cells, phantom_skip, *phantoms, 
         atom_coords);

/*   if (vdw_switch == 2)   */
        wca_period_phantom(mol_species, atom_rel, atom_mol, atom_type,
          n_atoms, n_atoms_per_mol, exclusions,
          comb_pot, comb_par, r2_on, r2_off, h, scaled_atom_coords, gamma, 
          three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_new, 
          period_switch, *first, *last, *phantoms, noffsets, nc_p, kc,
          phantom_skip, nc, atom_cells, offsets, atom_coords,
          rel_atom_coords, at_stress_vdw_s, mol_stress_vdw_s,
          vdw_s);
/*   else
   lj_s_period_phantom(mol_species, atom_rel, atom_mol, n_atoms, 
     n_atoms_per_mol,
     exclusions, comb_sigma2, comb_four_eps, h, r2_on, r2_off,
     scaled_atom_coords, gamma, three_gamma_over_two, two_over_gamma_cubed,
     &pe_vdw_new,
     period_switch, *first, *last, *phantoms, noffsets, nc_p, kc, phantom_skip,
     nc, atom_cells, offsets, atom_coords, 
     rel_atom_coords, at_stress_vdw_s, mol_stress_vdw_s, vdw_s);
*/
     }
    else{
    if (phant_switch == 0) {
       set_up_cells_nophantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
         period_switch, r_nl, r_off, neigh_switch, first, last, offsets_x,
         offsets_y, offsets_z,
         n_atoms, scaled_atom_coords, atom_coords);

       update_cells_nophantom( period_switch,  n_atoms, h, scaled_atom_coords, 
         atom_coords, atom_cells, *first, *last, kc, nc, nc_p);

       lj_s_period_nophantom(mol_species, atom_rel, atom_mol, atom_type, 
         n_atoms, n_atoms_per_mol, exclusions, comb_pot, comb_par, h, r2_on, 
         r2_off, scaled_atom_coords, gamma, three_gamma_over_two, 
         two_over_gamma_cubed, &pe_vdw_new, period_switch, *first, *last, 
         noffsets, nc_p, kc, nc, atom_cells, offsets_x, offsets_y, offsets_z, 
         atom_coords, rel_atom_coords, at_stress_vdw_s, mol_stress_vdw_s, 
         vdw_s);

     }
  }
   }
   else {
       set_up_cells_phantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
         period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
         n_atoms, scaled_atom_coords, atom_coords, phantoms, phantom_skip);

       update_cells(period_switch, n_atoms, h, scaled_atom_coords,
         atom_coords, atom_cells, *first, *last, *phantoms, kc,
         phantom_skip, nc, nc_p);

       update_phantoms(n_atoms, atom_cells, phantom_skip, *phantoms, 
         atom_coords);

       neighbor_lists_period(mol_species, atom_rel, atom_mol,  n_atoms,
          n_atoms_per_mol, exclusions, h, r2_on,  r2_off, scaled_atom_coords,
          period_switch, *first, *last, *phantoms, noffsets, nc_p, kc,
          phantom_skip, nc, atom_cells, offsets, atom_coords, atom_move,
          nl_head, nl_tail, r2_nl);

 /*   lj_s_nl_period_phantom( mol_species, atom_rel, atom_mol, n_atoms,
       n_atoms_per_mol, exclusions, comb_sigma2, comb_four_eps, h, r2_on,
       r2_off, scaled_atom_coords, gamma, three_gamma_over_two,
       two_over_gamma_cubed, &pe_vdw_new, period_switch, *first, *last, 
       *phantoms,
       noffsets, nc_p, kc, phantom_skip, nc, atom_cells, offsets, atom_coords,
       nl_head, nl_tail, r2_nl, rel_atom_coords, at_stress_vdw_s,
       mol_stress_vdw_s, vdw_s, atom_move);
*/
   }

   /* Compute change in potential energy and accept the reshaping move 
      using Metropolis criterion. */
   delta_pe = pe_vdw_new - pe_vdw_old;

   if ((delta_pe <= 0.0) || (exp(-delta_pe * reduce_boltz) > ran3(i_ran))) 
      {
      /* Assign new total potential energy. */
      *pe_vdw = pe_vdw_new;
      reshape_accept = 1;
      /* Recompute maximum scaled displacement. */
      for (i = 0; i < NDIM; ++i)
        ds_mol_max[i] = dr_max / h[i][i];
      /* Initialize atom move counter if necessary. */
      if (neigh_switch == 2)
         zero_move(n_atoms, atom_move);
      }
      else {
      /* Restore the previous state. */

      /* Get box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            h[i][j] = h_old[i][j];
            h_inv[i][j] = h_inv_old[i][j];
         }
   
      /* Get stress tensor. */
      for (i = 0; i < 3; ++i)
         for (j = 0; j < 3; ++j)
            mol_stress_vdw_s[i][j] = stress_old[i][j];

      for (i_mol = 0; i_mol < n_mols; ++i_mol) {
        
        /* Get old molecular coordinates. */
        for (k = 0; k < NDIM; ++k)
          mol_coords[i_mol][k] = mol_old[i_mol][k];

        /* Get label of molecular species. */
        i_species = mol_species[i_mol];

        /* Get atom label offset. */
        skip = mol_first_atm[i_mol];

        /* Compute old atomic positions. */
        for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {
          abs = skip + i_atom;
          atom_coords[abs][0] = mol_coords[i_mol][0] + rel_atom_coords[abs][0];
 	  atom_coords[abs][1] = mol_coords[i_mol][1] + rel_atom_coords[abs][1];
          atom_coords[abs][2] = mol_coords[i_mol][2] + rel_atom_coords[abs][2];
          }
        }

        /* Compute scaled atom coordinates and apply periodic boundary 
           conditions. */
        if (period_switch){
          scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);
          periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, 
            atom_coords);
         }
        if (neigh_switch != 0)
           if (phant_switch == 1) {

             /* Update cell list. */
             set_up_cells_phantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
               period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
               n_atoms, scaled_atom_coords, atom_coords, phantoms, 
               phantom_skip);

             update_cells(period_switch, n_atoms, h, scaled_atom_coords, 
               atom_coords, atom_cells, *first, *last, *phantoms, kc, 
               phantom_skip, nc, nc_p);

             update_phantoms(n_atoms, atom_cells, phantom_skip, *phantoms, 
               atom_coords);
           } 
        else{
       if (phant_switch == 0) {

             /* Update cell list. */
             set_up_cells_nophantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
               period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
               n_atoms, scaled_atom_coords, atom_coords );

             update_cells_nophantom(period_switch, n_atoms, h, 
               scaled_atom_coords, atom_coords, atom_cells, *first, *last, kc,
               nc, nc_p);

        } 
     }
     if (neigh_switch == 2)
       neighbor_lists_period(mol_species, atom_rel, atom_mol,  n_atoms,
          n_atoms_per_mol, exclusions, h, r2_on,  r2_off, scaled_atom_coords,
          period_switch, *first, *last, *phantoms, noffsets, nc_p, kc,
          phantom_skip, nc, atom_cells, offsets, atom_coords, atom_move,
          nl_head, nl_tail, r2_nl);
       }

   /* Return reshape_accept. */
   return(reshape_accept);
}
/***************************************************************************/
int reshape_move_press(int period_switch, int vdw_switch, int reshape_axis, 
      double mol_max_length, int n_mols, int n_atoms, long *i_ran, 
      int *mol_first_atm, int *n_atoms_per_mol, int *mol_species,
      int *atom_rel, int *atom_mol, int *atom_type, double **h, double **h_inv,
      double aspect_xy, double aspect_xz, double **mol_coords, 
      double **scaled_mol_coords, double **atom_coords,
      double **scaled_atom_coords, double **rel_atom_coords, int ***exclusions,
      int **comb_pot, double ***comb_par, double gamma, 
      double three_gamma_over_two, double two_over_gamma_cubed, double dh_max, 
      double dr_max, double *ds_mol_max, double r2_on, double r2_off, 
      double r_off, double r2_nl, double r_nl, double **mol_old, 
      double reduce_boltz, double press, int neigh_switch, int phant_switch, 
      int **first, int **last, phantom_cell **phantoms, int noffsets, 
      int *nc, int *nc_p, int *nc_p_total, int kc, int *phantom_skip, 
      atom_cell *atom_cells, int *offsets, nl_entry **nl_head, 
      nl_entry **nl_tail, double **atom_move, double *pe_vdw, 
      double **at_stress_vdw_s, double **mol_stress_vdw_s, double **vdw_s,
      int *offsets_x, int *offsets_y, int *offsets_z)

{
   int i, j, k, n_axis, i_axis, i_mol, i_atom, abs, i_species, skip, 
      reshape_accept;
   double h_old[NDIM][NDIM], h_inv_old[NDIM][NDIM], dh[NDIM], pe_vdw_old, 
      stress_old[NDIM][NDIM], pe_vdw_new, delta_pe, volume_new, volume_old;

   /* Set acceptance counter to zero. */
   reshape_accept = 0;

   /* Get total potential energy before the reshaping attempt. */
   pe_vdw_old = *pe_vdw;

   /* Get box coordinates. */
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j){ 
        h_old[i][j] = h[i][j];
        h_inv_old[i][j] = h_inv[i][j];
      }
   volume_old = h[0][0] * h[1][1] * h[2][2];

   /* Get stress tensor. */
   for (i = 0; i < 3; ++i)
      for (j = 0; j < 3; ++j)
         stress_old[i][j] = mol_stress_vdw_s[i][j];

   /* Generate random change in box dimensions. */
   dh[0] = (2.0 * ran3(i_ran) - 1.0) * dh_max;
   if ((reshape_axis == 0) || (reshape_axis == 1)) {
     dh[1] = (2.0 * ran3(i_ran) - 1.0) * dh_max;
     dh[2] = (2.0 * ran3(i_ran) - 1.0) * dh_max;
     for (i = 0; i < 3; ++i) 
        h[i][i] += dh[i];
     }
   else if (reshape_axis == 2) {
        h[0][0] += dh[0];
        h[1][1] = aspect_xy * h[0][0];
        h[2][2] = aspect_xz * h[0][0];
   }
   else {
      n_axis = (int) (3.0 * ran3(i_ran));
      if (n_axis == 3) n_axis = 0;
         h[n_axis][n_axis] += dh[0];
      }
   volume_new = h[0][0] * h[1][1] * h[2][2];

   /* Calculate quantities that depend on box dimensions and exit if
      system is too small. */
   box_dimensions(h, h_inv, period_switch, r_off, mol_max_length);

   /* Scale molecular center of mass coordinates. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Store centers of mass of the molecules before reshaping. */
      for (k = 0; k < NDIM; ++k)
         mol_old[i_mol][k] = mol_coords[i_mol][k];
     
      /* Compute new center of mass after box reshaping. */ 
      mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0]
         + h[0][1] * scaled_mol_coords[i_mol][1]
         + h[0][2] * scaled_mol_coords[i_mol][2];
      mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1]
         + h[1][2] * scaled_mol_coords[i_mol][2];
      mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Shift atomic positions. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

        /* Get absolute label of the atom. */
        abs = skip + i_atom;

        /* Compute new atomic positions. Remember that the relative atom
        coordinates are unfolded. */
        atom_coords[abs][0] = mol_coords[i_mol][0] + rel_atom_coords[abs][0];
        atom_coords[abs][1] = mol_coords[i_mol][1] + rel_atom_coords[abs][1];
        atom_coords[abs][2] = mol_coords[i_mol][2] + rel_atom_coords[abs][2];
      }
   }

   /* Compute scaled atom positions and apply periodic boundary conditions. */
   if (period_switch) {
     scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);
     periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, atom_coords);
   }

   /* Compute new total potential energy. */
   if (neigh_switch == 0) {
       lj_s_period(mol_species, atom_rel, atom_mol, atom_type, n_atoms,
          n_atoms_per_mol, exclusions, comb_pot, comb_par,
          h, r2_on, r2_off, scaled_atom_coords, scaled_mol_coords,
          rel_atom_coords, gamma, three_gamma_over_two, 
          two_over_gamma_cubed, &pe_vdw_new, vdw_s, mol_stress_vdw_s);
   }
   else if (neigh_switch == 1) {
   /* Set up cell list with phantom cells. */
     if (phant_switch == 1) {

       set_up_cells_phantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
         period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
         n_atoms, scaled_atom_coords, atom_coords, phantoms, phantom_skip);

       update_cells(period_switch, n_atoms, h, scaled_atom_coords,
         atom_coords, atom_cells, *first, *last, *phantoms, kc,
         phantom_skip, nc, nc_p);

       update_phantoms(n_atoms, atom_cells, phantom_skip, *phantoms, 
         atom_coords);

/*      if (vdw_switch == 2)  */
        wca_period_phantom(mol_species, atom_rel, atom_mol, atom_type,
          n_atoms, n_atoms_per_mol, exclusions,
          comb_pot, comb_par, r2_on, r2_off, h, scaled_atom_coords, gamma, 
          three_gamma_over_two, two_over_gamma_cubed, &pe_vdw_new, 
          period_switch, *first, *last, *phantoms, noffsets, nc_p, kc,
          phantom_skip, nc, atom_cells, offsets, atom_coords,
          rel_atom_coords, at_stress_vdw_s, mol_stress_vdw_s,
          vdw_s);
/*     else
     lj_s_period_phantom(mol_species, atom_rel, atom_mol, n_atoms, 
     n_atoms_per_mol,
     exclusions, comb_sigma2, comb_four_eps, h, r2_on, r2_off,
     scaled_atom_coords, gamma, three_gamma_over_two, two_over_gamma_cubed,
     &pe_vdw_new,
     period_switch, *first, *last, *phantoms, noffsets, nc_p, kc, phantom_skip,
     nc, atom_cells, offsets, atom_coords, 
     rel_atom_coords, at_stress_vdw_s, mol_stress_vdw_s, vdw_s);
*/
     }
    else{
    if (phant_switch == 0) {
       set_up_cells_nophantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
         period_switch, r_nl, r_off, neigh_switch, first, last, offsets_x,
         offsets_y, offsets_z,
         n_atoms, scaled_atom_coords, atom_coords);

       update_cells_nophantom( period_switch,  n_atoms, h, scaled_atom_coords, 
         atom_coords, atom_cells, *first, *last, kc, nc, nc_p);

       lj_s_period_nophantom(mol_species, atom_rel, atom_mol, atom_type, 
         n_atoms, n_atoms_per_mol, exclusions, comb_pot, comb_par, h, r2_on, 
         r2_off, scaled_atom_coords, gamma, three_gamma_over_two, 
         two_over_gamma_cubed, &pe_vdw_new, period_switch, *first, *last, 
         noffsets, nc_p, kc, nc, atom_cells, offsets_x, offsets_y, offsets_z, 
         atom_coords, rel_atom_coords, at_stress_vdw_s, mol_stress_vdw_s, 
         vdw_s);

     }
  }
   }
   else {
       set_up_cells_phantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
         period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
         n_atoms, scaled_atom_coords, atom_coords, phantoms, phantom_skip);

       update_cells(period_switch, n_atoms, h, scaled_atom_coords,
         atom_coords, atom_cells, *first, *last, *phantoms, kc,
         phantom_skip, nc, nc_p);

       update_phantoms(n_atoms, atom_cells, phantom_skip, *phantoms, 
         atom_coords);

       neighbor_lists_period(mol_species, atom_rel, atom_mol,  n_atoms,
          n_atoms_per_mol, exclusions, h, r2_on,  r2_off, scaled_atom_coords,
          period_switch, *first, *last, *phantoms, noffsets, nc_p, kc,
          phantom_skip, nc, atom_cells, offsets, atom_coords, atom_move,
          nl_head, nl_tail, r2_nl);

 /*   lj_s_nl_period_phantom( mol_species, atom_rel, atom_mol, n_atoms,
       n_atoms_per_mol, exclusions, comb_sigma2, comb_four_eps, h, r2_on,
       r2_off, scaled_atom_coords, gamma, three_gamma_over_two,
       two_over_gamma_cubed, &pe_vdw_new, period_switch, *first, *last, 
       *phantoms,
       noffsets, nc_p, kc, phantom_skip, nc, atom_cells, offsets, atom_coords,
       nl_head, nl_tail, r2_nl, rel_atom_coords, at_stress_vdw_s,
       mol_stress_vdw_s, vdw_s, atom_move);
*/
   }

   /* Compute change in potential energy and accept the box move 
      using Metropolis criterion. */
   delta_pe = pe_vdw_new - pe_vdw_old + press * (volume_new - volume_old)
              - n_mols * log(volume_new / volume_old) / reduce_boltz;

   if ((delta_pe <= 0.0) || (exp(-delta_pe * reduce_boltz) > ran3(i_ran))) 
      {
      /* Assign new total potential energy. */
      *pe_vdw = pe_vdw_new;
      reshape_accept = 1;
      /* Recompute maximum scaled displacement. */
      for (i = 0; i < NDIM; ++i)
        ds_mol_max[i] = dr_max / h[i][i];
      /* Initialize atom move counter if necessary. */
      if (neigh_switch == 2)
         zero_move(n_atoms, atom_move);
      }
      else {
      /* Restore the previous state. */

      /* Get box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j) {
            h[i][j] = h_old[i][j];
            h_inv[i][j] = h_inv_old[i][j];
         }
   
      /* Get stress tensor. */
      for (i = 0; i < 3; ++i)
         for (j = 0; j < 3; ++j)
            mol_stress_vdw_s[i][j] = stress_old[i][j];

      for (i_mol = 0; i_mol < n_mols; ++i_mol) {
        
        /* Get old molecular coordinates. */
        for (k = 0; k < NDIM; ++k)
          mol_coords[i_mol][k] = mol_old[i_mol][k];

        /* Get label of molecular species. */
        i_species = mol_species[i_mol];

        /* Get atom label offset. */
        skip = mol_first_atm[i_mol];

        /* Compute old atomic positions. */
        for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {
          abs = skip + i_atom;
          atom_coords[abs][0] = mol_coords[i_mol][0] + rel_atom_coords[abs][0];
 	  atom_coords[abs][1] = mol_coords[i_mol][1] + rel_atom_coords[abs][1];
          atom_coords[abs][2] = mol_coords[i_mol][2] + rel_atom_coords[abs][2];
          }
        }

        /* Compute scaled atom coordinates and apply periodic boundary 
           conditions. */
        if (period_switch){
          scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);
          periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, 
            atom_coords);
         }
        if (neigh_switch != 0)
           if (phant_switch == 1) {

             /* Update cell list. */
             set_up_cells_phantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
               period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
               n_atoms, scaled_atom_coords, atom_coords, phantoms, 
               phantom_skip);

             update_cells(period_switch, n_atoms, h, scaled_atom_coords, 
               atom_coords, atom_cells, *first, *last, *phantoms, kc, 
               phantom_skip, nc, nc_p);

             update_phantoms(n_atoms, atom_cells, phantom_skip, *phantoms, 
               atom_coords);
           } 
        else{
       if (phant_switch == 0) {

             /* Update cell list. */
             set_up_cells_nophantom(h, nc, nc_p, nc_p_total, atom_cells,  kc,
               period_switch, r_nl, r_off, neigh_switch, first, last, offsets,
               n_atoms, scaled_atom_coords, atom_coords );

             update_cells_nophantom(period_switch, n_atoms, h, 
               scaled_atom_coords, atom_coords, atom_cells, *first, *last, kc,
               nc, nc_p);
        } 
     }
     if (neigh_switch == 2)
       neighbor_lists_period(mol_species, atom_rel, atom_mol,  n_atoms,
          n_atoms_per_mol, exclusions, h, r2_on,  r2_off, scaled_atom_coords,
          period_switch, *first, *last, *phantoms, noffsets, nc_p, kc,
          phantom_skip, nc, atom_cells, offsets, atom_coords, atom_move,
          nl_head, nl_tail, r2_nl);
       }

   /* Return reshape_accept. */
   return(reshape_accept);
}
