/* Input/output routines for template module. */

#include "build.h"
#include "ff.h"

/****************************************************************************/
/* Read and Set up potential parameters for nonbonded interactions. */
/*
input :
     Combination rule switch between atomic sites (vdw_rad_switch)
     Number of molecular species (n_species)
     Array of number of atomic sites per template molecule (n_atoms_per_mol)
     Number of LJ parameters (n_lj)
     Array of LJ parameters (lj)
     Array of template atom - type of atomic sites (temp_atm_type)
     Array of template atom - sigma LJ parameter (temp_atm_sigma) 
     Array of template atom - epsilon LJ parameter (temp_atm_eps)
     Array of template atom - charge of atomic sites (temp_atm_chg) 
output :
     Array of sigma combination between 2 atomic sites (comb_sigma)
     Array of sigma2 combination between 2 atomic sites (comb_sigma2)
     Array of r_min2 combination between 2 atomic sites (comb_r_min2)
     Array of epsilon combination between 2 atomic sites (comb_eps)
     Array of 4*epsilon combination between 2 atomic sites (comb_four_eps)
     Array of pair interaction  combination between 2 atomic sites (comb_pot)
     Array of charge combination between 2 atomic sites (comb_chg)
*/

void read_set_up_pot(char *pot_file, int vdw_switch, int vdw_rad_switch,  
     type_entry *atm_type, int n_type, int ***comb_pot, double ****comb_par)
{
   FILE *f_input;
   char line[MAX_LINE], **token = NULL;
   char *type_1, *type_2, *type_i, *type_j;
   int i_type, j_type, id_0, id_1, identifier[2], test_1, test_2, 
      found, same, pot_form, pot_form_1, pot_form_2, n_tokens, **found_entry;
   double cube_root_of_two, max_min;

   /* Allocate memory for combination array. */
   *comb_pot = allocate_2d_array(n_type, n_type, sizeof(int));
   *comb_par = allocate_3d_array(n_type, n_type, MAX_PAR, sizeof(double));

   /* Allocate memory for temporary array. */
   found_entry = allocate_2d_array(n_type, n_type, sizeof(int)); 

   /* Look only for pair interactions. */
   /* Initialisation. */
   for (i_type = 0; i_type < n_type; ++i_type)
     for (j_type = 0; j_type < n_type; ++j_type)
       found_entry[i_type][j_type] = 0;

   /* Open potential file. */
   if ((f_input = fopen(pot_file, "r")) == NULL)
      error_exit("Unable to open potential file in combination_array2");

   /* Read and parse pot_file line. */ 
   while (fgets(line, MAX_LINE, f_input) != NULL) {
     if (strncmp(line, "#", 1) != 0) {
       n_tokens = parse_tokens(line, &token);
       pot_form = atoi(token[0]); 
       type_i = token[1];
       type_j = token[2];

       /* Check if the atoms are of the same type. */ 
       same = 0;  
       if (strcmp(type_i, type_j) == 0) 
         same = 1;
        
       /* Search for the integer identifiers corresponding 
          to the atom types. */
       if (same) {
         found = 0;
         for (i_type = 0; i_type < n_type; ++i_type) {
            type_1 = atm_type[i_type].type;
            test_1 = strcmp(type_i, type_1) == 0;
            if (test_1) {
              found = 1;
              break;
            }
         }
                  
         if (found) {
           /* Get integer identifier. */
           id_0 = atm_type[i_type].id;
           /* Assign potential form. */
           (*comb_pot)[id_0][id_0] = pot_form;
           /* Update found entry counter. */
           found_entry[id_0][id_0] = 1;          
           /* Select the form of the potential between the 2 atom types. */
           select_pot(id_0, id_0, pot_form, token, n_tokens, vdw_rad_switch,
             0, *comb_par);
         } 
       }
       else {
         found = 0;
         for (i_type = 0; i_type < n_type; ++i_type) {
            type_1 = atm_type[i_type].type;
            test_1 = strcmp(type_i, type_1) == 0;
            test_2 = strcmp(type_j, type_1) == 0;
            if (test_1 || test_2) {
              identifier[found] = atm_type[i_type].id;
              ++found;
              if (found == 2) break;
            }
         }
                  
         if (found == 2) {
           /* Assign potential form. */
           id_0 = identifier[0];
           id_1 = identifier[1];
           (*comb_pot)[id_0][id_1] = pot_form;
           (*comb_pot)[id_1][id_0] = pot_form;
           /* Update found entry counter. */
           found_entry[id_0][id_1] = 1;
           found_entry[id_1][id_0] = 1;
           /* Select the form of the potential between the 2 atom types. */
           select_pot(id_0, id_1, pot_form, token, n_tokens, vdw_rad_switch,
             0, *comb_par);
        }
       }
     }
   } /* end while */

   /* Check that an entry has been found for each atom, otherwise exit.*/
   for (i_type = 0; i_type < n_type; ++i_type)
      if (found_entry[i_type][i_type] == 0){
         printf("No entry found in pot.ff, for type %s (id = %d)\n", atm_type[i_type].type, i_type);
         exit(1);
      }

   /* Apply combination rules between atoms of unlike type. */
   /* We are looking for the off-diagonal combinations which were 
   not explicitely defined in pot_file. */

   for (i_type = 0; i_type < n_type - 1; ++i_type)
     for (j_type = i_type + 1; j_type < n_type; ++j_type) {
       /* Check if off-diagonal combination have not been found yet. */
       if (found_entry[i_type][j_type] == 0) {

         /* Check if combination rules may apply. */
         pot_form_1 = (*comb_pot)[i_type][i_type];
         pot_form_2 = (*comb_pot)[j_type][j_type];
         if (pot_form_1 == pot_form_2) {
           (*comb_pot)[i_type][j_type] = pot_form_1;
           (*comb_pot)[j_type][i_type] = pot_form_1;
           /* Update found_entry counter. */
           found_entry[i_type][j_type] = 1;
           found_entry[j_type][i_type] = 1;
           /* Select the form of the potential between the 2 atom types. */
           select_pot(i_type, j_type, pot_form_1, token, n_tokens, 
             vdw_rad_switch, 1, *comb_par);
         }
         else {
           printf("i_type = %d, pot_form = %d\n", i_type, pot_form_1); 
           printf("j_type = %d, pot_form = %d\n", j_type, pot_form_2);
           error_exit("No possible combination rule. Missing entry\n");
         }
         
       }
     }
   /* Close potential file. */
   fclose(f_input); 
}
/*****************************************************************************/

