/*****************************************************************************/
/* Center and orient template molecules. */
/*
input :
       Number of molecular species (n_species)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of principal molecular axis aligned along the frame-z axis
                                                              (principal)
       Array of template atom - mass of atomic sites (temp_atm_mass)
       Array of template atom - position of atomic sites (temp_atm_pos)
output :
       Array of template atom - position of atomic sites (temp_atm_pos)
*/

void inertia_molecules(int n_species, int *n_atoms_per_mol, 
      int *principal, double **temp_atm_mass, double ***temp_atm_pos)
{
   int i_species, i, j, k, nrot;
   double **inertia, *d, **v, x_cm, y_cm, z_cm, x_tmp, y_tmp, z_tmp, test,
      x_min, y_min, z_min, x_max, y_max, z_max, x_cg, y_cg, z_cg, mol_mass,
      x_range, y_range, z_range;

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Calculate center of mass position of template molecule. */
      x_cm = y_cm = z_cm = 0.0;
      mol_mass = 0.0;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         x_cm += temp_atm_mass[i_species][i] 
                        * temp_atm_pos[i_species][i][0];
         y_cm += temp_atm_mass[i_species][i] 
                        * temp_atm_pos[i_species][i][1];
         z_cm += temp_atm_mass[i_species][i] 
                        * temp_atm_pos[i_species][i][2];
      mol_mass +=  temp_atm_mass[i_species][i];
      }
      x_cm /= mol_mass;
      y_cm /= mol_mass;
      z_cm /= mol_mass;

      /* Translate center of mass of template molecule to origin. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         temp_atm_pos[i_species][i][0] -= x_cm;
         temp_atm_pos[i_species][i][1] -= y_cm;
         temp_atm_pos[i_species][i][2] -= z_cm;
      }

      /* Calculate inertia tensor of template molecule. */
      inertia = dmatrix(1, 3, 1, 3);
      for (j = 1; j <= 3; ++j)
         for (k = 1; k <= 3; ++k)
            inertia[j][k] = 0.0;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         inertia[1][1] += temp_atm_mass[i_species][i]
            * (SQR(temp_atm_pos[i_species][i][1])
            + SQR(temp_atm_pos[i_species][i][2]));
         inertia[2][2] += temp_atm_mass[i_species][i]
            * (SQR(temp_atm_pos[i_species][i][0])
            + SQR(temp_atm_pos[i_species][i][2]));
         inertia[3][3] += temp_atm_mass[i_species][i]
            * (SQR(temp_atm_pos[i_species][i][0])
            + SQR(temp_atm_pos[i_species][i][1]));
         inertia[1][2] -= temp_atm_mass[i_species][i]
            * temp_atm_pos[i_species][i][0] * temp_atm_pos[i_species][i][1];
         inertia[1][3] -= temp_atm_mass[i_species][i]
            * temp_atm_pos[i_species][i][0] * temp_atm_pos[i_species][i][2];
         inertia[2][3] -= temp_atm_mass[i_species][i]
            * temp_atm_pos[i_species][i][1] * temp_atm_pos[i_species][i][2];
      }
      inertia[2][1] = inertia[1][2];
      inertia[3][1] = inertia[1][3];
      inertia[3][2] = inertia[2][3];

      /* Calculate eigenvalues and eigenvectors of inertia tensor. */
      d = dvector(1, 3);
      v = dmatrix(1, 3, 1, 3);
      jacobi(inertia, 3, d, v, &nrot);
      eigsrt(d, v, 3);

      /* Make sure that eigenvectors constitute a right-handed 
         coordinate system, and make it so if not. */
      test = (v[2][1] * v[3][2] - v[3][1] * v[2][2]) * v[1][3]
         + (v[3][1] * v[1][2] - v[1][1] * v[3][2]) * v[2][3]
         + (v[1][1] * v[2][2] - v[2][1] * v[1][2]) * v[3][3];
      if (test < 0.0) {
         v[1][3] = - v[1][3];
         v[2][3] = - v[2][3];
         v[3][3] = - v[3][3];
      }

      /* Measure geometrical dimensions of molecule. */
      x_min = x_max = temp_atm_pos[i_species][0][0];
      y_min = y_max = temp_atm_pos[i_species][0][1];
      z_min = z_max = temp_atm_pos[i_species][0][2];
      for (i = 1; i < n_atoms_per_mol[i_species]; ++i) {
         x_min = MIN(x_min, temp_atm_pos[i_species][i][0]);
         y_min = MIN(y_min, temp_atm_pos[i_species][i][1]);
         z_min = MIN(z_min, temp_atm_pos[i_species][i][2]);
         x_max = MAX(x_max, temp_atm_pos[i_species][i][0]);
         y_max = MAX(y_max, temp_atm_pos[i_species][i][1]);
         z_max = MAX(z_max, temp_atm_pos[i_species][i][2]);
      }
      x_range = x_max - x_min;
      y_range = y_max - y_min;
      z_range = z_max - z_min;

      /* Shift geometrical center of template molecule to the origin. */
      x_cg = x_min + 0.5 * x_range;
      y_cg = y_min + 0.5 * y_range;
      z_cg = z_min + 0.5 * z_range;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         temp_atm_pos[i_species][i][0] -= x_cg;
         temp_atm_pos[i_species][i][1] -= y_cg;
         temp_atm_pos[i_species][i][2] -= z_cg;
      }
   }
}

