/* Input/output routines for force field module. */

#include "shared.h"
#include "create.h"

/* Read force field parameters from specified data file. */
void read_ff_params(char *ff_file)
{
   FILE *f_input;
   char line[MAX_LINE];
   mass_entry mass;
   vdw_entry vdw;
   str_entry str;
   bend_entry bend;
   tors_entry tors;
   inv_entry inv;
   int j;

   /* Open parameter file. */
   if ((f_input = fopen(ff_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_ff_params");

   /* Read in parameter sets. */
   while (fgets(line, MAX_LINE, f_input) != NULL) {

      /* Read in mass entries. */
      if (strncmp(line, "# mass", 6) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %lf %lf",
               mass.type, &mass.true_mass, &mass.eff_mass);
            if ((ff.mass = realloc(ff.mass,
               (ff.n_mass + 1) * sizeof(mass_entry))) == NULL)
               error_exit("Out of memory in read_ff_params");
            ff.mass[ff.n_mass] = mass;
            ++ff.n_mass;
         }
      }

      /* Read in vdw entries. */
      if (strncmp(line, "# vdw", 5) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %lf %lf %lf %lf %lf",
               vdw.type_1, vdw.type_2,
               &vdw.sigma, &vdw.epsilon,
               &vdw.A, &vdw.B, &vdw.C);
            if ((ff.vdw = realloc(ff.vdw,
               (ff.n_vdw + 1) * sizeof(vdw_entry))) == NULL)
               error_exit("Out of memory in read_ff_params");
            ff.vdw[ff.n_vdw] = vdw;
            ++ff.n_vdw;
         }
      }

      /* Read in stretch entries. */
      else if (strncmp(line, "# stretch", 9) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %lf %lf",
               str.type_1, str.type_2,
               &str.spring, &str.r_equil);
            if ((ff.str = realloc(ff.str,
               (ff.n_str + 1) * sizeof(str_entry))) == NULL)
               error_exit("Out of memory in read_ff_params");
            ff.str[ff.n_str] = str;
            ++ff.n_str;
         }
      }

      /* Read in bend entries. */
      else if (strncmp(line, "# bend", 6) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %s %lf %lf",
               bend.type_1, bend.type_2, bend.type_3,
               &bend.spring, &bend.theta_equil);
            if ((ff.bend = realloc(ff.bend,
               (ff.n_bend + 1) * sizeof(bend_entry))) == NULL)
               error_exit("Out of memory in read_ff_params");
            ff.bend[ff.n_bend] = bend;
            ++ff.n_bend;
         }
      }

      /* Read in torsion entries. */
      else if (strncmp(line, "# torsion", 9) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            strcpy(tors.type_1, strtok(line, " "));
            strcpy(tors.type_2, strtok(NULL, " "));
            strcpy(tors.type_3, strtok(NULL, " "));
            strcpy(tors.type_4, strtok(NULL, " "));
            tors.n_order = atoi(strtok(NULL, " "));
            for (j = 0; j <= tors.n_order; ++j)
               tors.c_a[j] = atof(strtok(NULL, " "));
            if ((ff.tors = realloc(ff.tors,
               (ff.n_tors + 1) * sizeof(tors_entry))) == NULL)
               error_exit("Out of memory in read_ff_params");
            ff.tors[ff.n_tors] = tors;
            ++ff.n_tors;
         }
      }

      /* Read in inversion entries. */
      else if (strncmp(line, "# inversion", 11) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %s %s %lf %lf",
               inv.type_1, inv.type_2, inv.type_3, inv.type_4,
               &inv.spring, &inv.omega_equil);
            if ((ff.inv = realloc(ff.inv,
               (ff.n_inv + 1) * sizeof(inv_entry))) == NULL)
               error_exit("Out of memory in read_ff_params");
            ff.inv[ff.n_inv] = inv;
            ++ff.n_inv;
         }
      }

      /* Skip comments, and exit if file has the wrong format. */
      else {
         if (line[0] != '#') {
            printf("%s\n", line);
            fflush(NULL);
            error_exit("Inappropriate entry found in read_ff_params");
         }
      }
   }
 
   /* Close input file. */
   fclose(f_input);
}

/* Read force field parameters from specified data file, eliminating multiple
   entries (only the last entry for a given interaction term is used). */
void read_consolidate_ff_params(char *ff_file)
{
   FILE *f_input;
   char line[MAX_LINE];
   int test_1, test_2, test_3, test_4, test_5,
      test_6, test_7, test_8, test_9, test_10,
      found, i, j;
   mass_entry mass;
   vdw_entry vdw;
   str_entry str;
   bend_entry bend;
   tors_entry tors;
   inv_entry inv;

   /* Open parameter file. */
   if ((f_input = fopen(ff_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_ff_params");

   /* Read in parameter sets. */
   while (fgets(line, MAX_LINE, f_input) != NULL) {

      /* Read in mass entries. */
      if (strncmp(line, "# mass", 6) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %lf %lf",
               mass.type, &mass.true_mass, &mass.eff_mass);
            found = 0;
            for (i = 0; i < ff.n_mass; ++i) {
               test_1 = strcmp(ff.mass[i].type, mass.type) == 0;
               if (test_1) {
                  found = 1;
                  ff.mass[i] = mass;
                  break;
               }
            }
            if (!found) {
               if ((ff.mass = realloc(ff.mass,
                  (ff.n_mass + 1) * sizeof(mass_entry))) == NULL)
                  error_exit("Out of memory in read_ff_params");
               ff.mass[ff.n_mass] = mass;
               ++ff.n_mass;
            }
         }
      }

      /* Read in vdw entries. */
      if (strncmp(line, "# vdw", 5) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %lf %lf %lf %lf %lf",
               vdw.type_1, vdw.type_2,
               &vdw.sigma, &vdw.epsilon,
               &vdw.A, &vdw.B, &vdw.C);
            found = 0;
            for (i = 0; i < ff.n_vdw; ++i) {
               test_1 = strcmp(ff.vdw[i].type_1, vdw.type_1) == 0;
               test_2 = strcmp(ff.vdw[i].type_2, vdw.type_2) == 0;
               test_3 = strcmp(ff.vdw[i].type_1, vdw.type_2) == 0;
               test_4 = strcmp(ff.vdw[i].type_2, vdw.type_1) == 0;
               if (test_1 && test_2 || test_3 && test_4) {
                  found = 1;
                  ff.vdw[i] = vdw;
                  break;
               }
            }
            if (!found) {
               if ((ff.vdw = realloc(ff.vdw,
                  (ff.n_vdw + 1) * sizeof(vdw_entry))) == NULL)
                  error_exit("Out of memory in read_ff_params");
               ff.vdw[ff.n_vdw] = vdw;
               ++ff.n_vdw;
            }
         }
      }

      /* Read in stretch entries. */
      else if (strncmp(line, "# stretch", 9) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %lf %lf",
               str.type_1, str.type_2,
               &str.spring, &str.r_equil);
            found = 0;
            for (i = 0; i < ff.n_str; ++i) {
               test_1 = strcmp(ff.str[i].type_1, str.type_1) == 0;
               test_2 = strcmp(ff.str[i].type_2, str.type_2) == 0;
               test_3 = strcmp(ff.str[i].type_1, str.type_2) == 0;
               test_4 = strcmp(ff.str[i].type_2, str.type_1) == 0;
               if (test_1 && test_2 || test_3 && test_4) {
                  found = 1;
                  ff.str[i] = str;
                  break;
               }
            }
            if (!found) {
               if ((ff.str = realloc(ff.str,
                  (ff.n_str + 1) * sizeof(str_entry))) == NULL)
                  error_exit("Out of memory in read_ff_params");
               ff.str[ff.n_str] = str;
               ++ff.n_str;
            }
         }
      }

      /* Read in bend entries. */
      else if (strncmp(line, "# bend", 6) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %s %lf %lf",
               bend.type_1, bend.type_2, bend.type_3,
               &bend.spring, &bend.theta_equil);
            found = 0;
            for (i = 0; i < ff.n_bend; ++i) {
               test_1 = strcmp(ff.bend[i].type_2, bend.type_2) == 0;
               test_2 = strcmp(ff.bend[i].type_1, bend.type_1) == 0;
               test_3 = strcmp(ff.bend[i].type_3, bend.type_3) == 0;
               test_4 = strcmp(ff.bend[i].type_1, bend.type_3) == 0;
               test_5 = strcmp(ff.bend[i].type_3, bend.type_1) == 0;
               if (test_1 && (test_2 && test_3 || test_4 && test_5)) {
                  found = 1;
                  ff.bend[i] = bend;
                  break;
               }
            }
            if (!found) {
               if ((ff.bend = realloc(ff.bend,
                  (ff.n_bend + 1) * sizeof(bend_entry))) == NULL)
                  error_exit("Out of memory in read_ff_params");
               ff.bend[ff.n_bend] = bend;
               ++ff.n_bend;
            }
         }
      }

      /* Read in torsion entries. */
      else if (strncmp(line, "# torsion", 9) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            strcpy(tors.type_1, strtok(line, " "));
            strcpy(tors.type_2, strtok(NULL, " "));
            strcpy(tors.type_3, strtok(NULL, " "));
            strcpy(tors.type_4, strtok(NULL, " "));
            tors.n_order = atoi(strtok(NULL, " "));
            for (j = 0; j <= tors.n_order; ++j)
               tors.c_a[j] = atof(strtok(NULL, " "));
            found = 0;
            for (i = 0; i < ff.n_tors; ++i) {
               test_1 = strcmp(ff.tors[i].type_1, tors.type_1) == 0;
               test_2 = strcmp(ff.tors[i].type_2, tors.type_2) == 0;
               test_3 = strcmp(ff.tors[i].type_3, tors.type_3) == 0;
               test_4 = strcmp(ff.tors[i].type_4, tors.type_4) == 0;
               test_5 = strcmp(ff.tors[i].type_1, tors.type_4) == 0;
               test_6 = strcmp(ff.tors[i].type_2, tors.type_3) == 0;
               test_7 = strcmp(ff.tors[i].type_3, tors.type_2) == 0;
               test_8 = strcmp(ff.tors[i].type_4, tors.type_1) == 0;
               if (test_1 && test_2 && test_3 && test_4
                  || test_5 && test_6 && test_7 && test_8) {
                  found = 1;
                  ff.tors[i] = tors;
                  break;
               }
            }
            if (!found) {
               if ((ff.tors = realloc(ff.tors,
                  (ff.n_tors + 1) * sizeof(tors_entry))) == NULL)
                  error_exit("Out of memory in read_ff_params");
               ff.tors[ff.n_tors] = tors;
               ++ff.n_tors;
            }
         }
      }

      /* Read in inversion entries. */
      else if (strncmp(line, "# inversion", 11) == 0) {
         while (strncmp(fgets(line, MAX_LINE, f_input), "# end", 5) != 0) {
            sscanf(line, "%s %s %s %s %lf %lf",
               inv.type_1, inv.type_2, inv.type_3, inv.type_4,
               &inv.spring, &inv.omega_equil);
            found = 0;
            for (i = 0; i < ff.n_inv; ++i) {
               test_1 = strcmp(ff.inv[i].type_1, inv.type_1) == 0;
               test_2 = strcmp(ff.inv[i].type_2, inv.type_2) == 0;
               test_3 = strcmp(ff.inv[i].type_2, inv.type_3) == 0;
               test_4 = strcmp(ff.inv[i].type_2, inv.type_4) == 0;
               test_5 = strcmp(ff.inv[i].type_3, inv.type_2) == 0;
               test_6 = strcmp(ff.inv[i].type_3, inv.type_3) == 0;
               test_7 = strcmp(ff.inv[i].type_3, inv.type_4) == 0;
               test_8 = strcmp(ff.inv[i].type_4, inv.type_2) == 0;
               test_9 = strcmp(ff.inv[i].type_4, inv.type_3) == 0;
               test_10 = strcmp(ff.inv[i].type_4, inv.type_4) == 0;
               if (test_1 && (test_2 && (test_6 && test_10 || test_7 && test_9)
                  || test_3 && (test_7 && test_8 || test_5 && test_10)
                  || test_4 && (test_5 && test_9 || test_6 && test_8))) {
                  found = 1;
                  ff.inv[i] = inv;
                  break;
               }
            }
            if (!found) {
               if ((ff.inv = realloc(ff.inv,
                  (ff.n_inv + 1) * sizeof(inv_entry))) == NULL)
                  error_exit("Out of memory in read_ff_params");
               ff.inv[ff.n_inv] = inv;
               ++ff.n_inv;
            }
         }
      }

      /* Skip comments, and exit if file has the wrong format. */
      else {
         if (line[0] != '#') {
            printf("%s\n", line);
            fflush(NULL);
            error_exit("Inappropriate entry found in read_ff_params");
         }
      }
   }
 
   /* Close input file. */
   fclose(f_input);
}

/* Write force field parameters to specified data file. */
void write_ff_params(char *ff_file)
{
   FILE *f_output;
   time_t now;
   int i, j;

   /* Open parameter file. */
   if ((f_output = fopen(ff_file, "w")) == NULL)
      error_exit("Unable to open parameter file in write_ff_params");

   /* Write date and time to header of parameter file. */
   now = time(NULL);
   fprintf(f_output, "#\n# File created on %s#\n", asctime(localtime(&now)));

   /* Write out mass entries. */
   fprintf(f_output, "#\n# mass\n");
   for (i = 0; i < ff.n_mass; ++i)
      fprintf(f_output, "%s %g %g\n",
         ff.mass[i].type, ff.mass[i].true_mass, ff.mass[i].eff_mass);
   fprintf(f_output, "# end\n");

   /* Write out vdw entries. */
   fprintf(f_output, "#\n# vdw\n");
   for (i = 0; i < ff.n_vdw; ++i)
      fprintf(f_output, "%s %s %g %g %g %g %g\n",
         ff.vdw[i].type_1, ff.vdw[i].type_2,
         ff.vdw[i].sigma, ff.vdw[i].epsilon,
         ff.vdw[i].A, ff.vdw[i].B, ff.vdw[i].C);
   fprintf(f_output, "# end\n");

   /* Write out stretch entries. */
   fprintf(f_output, "#\n# stretch\n");
   for (i = 0; i < ff.n_str; ++i)
      fprintf(f_output, "%s %s %g %g\n",
         ff.str[i].type_1, ff.str[i].type_2,
         ff.str[i].spring, ff.str[i].r_equil);
   fprintf(f_output, "# end\n");

   /* Write out bend entries. */
   fprintf(f_output, "#\n# bend\n");
   for (i = 0; i < ff.n_bend; ++i)
      fprintf(f_output, "%s %s %s %g %g\n",
         ff.bend[i].type_1, ff.bend[i].type_2, ff.bend[i].type_3,
         ff.bend[i].spring, ff.bend[i].theta_equil);
   fprintf(f_output, "# end\n");

   /* Write out torsion entries. */
   fprintf(f_output, "#\n# torsion\n");
   for (i = 0; i < ff.n_tors; ++i) {
      fprintf(f_output, "%s %s %s %s",
         ff.tors[i].type_1, ff.tors[i].type_2,
         ff.tors[i].type_3, ff.tors[i].type_4);
      fprintf(f_output, " %d", ff.tors[i].n_order);
      for (j = 0; j <= ff.tors[i].n_order; ++j)
         fprintf(f_output, " %g", ff.tors[i].c_a[j]);
      fprintf(f_output, "\n");
   }
   fprintf(f_output, "# end\n");

   /* Write out inversion entries. */
   fprintf(f_output, "#\n# inversion\n");
   for (i = 0; i < ff.n_inv; ++i)
      fprintf(f_output, "%s %s %s %s %g %g\n",
         ff.inv[i].type_1, ff.inv[i].type_2,
         ff.inv[i].type_3, ff.inv[i].type_4,
         ff.inv[i].spring, ff.inv[i].omega_equil);
   fprintf(f_output, "# end\n");
 
   /* Close output file. */
   fclose(f_output);
}

/* Read force field creation module input parameter file. */
void read_crt_params(char *par_file, parameters *par_set)
{
   FILE *f_param;
   char line[MAX_LINE];
   int i;

   /* Open parameter file. */
   if ((f_param = fopen(par_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_crt_params");

   /* Read in force field creation module parameters. */
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%s\n", (*par_set).ff_file);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", &(*par_set).n_part_ff_files);
   if (((*par_set).part_ff_file = calloc((*par_set).n_part_ff_files,
      F_MAX * sizeof(char))) == NULL)
      error_exit("Out of memory in read_crt_params");
   fgets(line, MAX_LINE, f_param);
   for (i = 0; i < (*par_set).n_part_ff_files; ++i)
      fscanf(f_param, "%s\n", (*par_set).part_ff_file[i]);

   /* Close parameter file. */
   fclose(f_param);
}

/* Write force field creation module input parameter file. */
void write_crt_params(char *par_file, parameters par_set)
{
   FILE *f_param;
   int i;

   /* Open parameter file. */
   if ((f_param = fopen(par_file, "w")) == NULL)
      error_exit("Unable to open parameter file in write_crt_params");

   /* Write out force field creation module parameters. */
   fprintf(f_param, "# ff_file\n");
   fprintf(f_param, "%s\n", par_set.ff_file);
   fprintf(f_param, "# n_part_ff_files\n");
   fprintf(f_param, "%d\n", par_set.n_part_ff_files);
   fprintf(f_param, "# part_ff_file[i], i = 0, n_part_ff_files - 1\n");
   for (i = 0; i < par_set.n_part_ff_files; ++i)
      fprintf(f_param, "%s\n", par_set.part_ff_file[i]);

   /* Close parameter file. */
   fclose(f_param);
}
