/* Mean squared displacement routines. */

#include "shared.h"
#include "process.h"

/* Calculate "unfolded" molecular center of mass positions and write them
   to output file. */
void create_com_pos_file(double **mol_coords_corr, double **scaled_mol_coords_corr,
                         double **h, )
{
   FILE *f_output, *f_trajec;
   int i_data, i_mol;
   double s_dx, s_dy, s_dz;

   /* Open input and output files. */
   if ((f_trajec = fopen(posit_file, "r")) == NULL)
      error_exit("Unable to open positions file in create_com_pos_file");
   if ((f_output = fopen("pro.com_pos_tmp", "w")) == NULL)
      error_exit("Unable to open com_pos file in create_com_pos_file");
   
   /* Position trajectory file pointer at beginning of first data set. */
   fseek(f_posit, pos_1, 0);

   /* If period_switch == 1, allocate memory for temporary array. */
   if (period_switch) {
      if ((mol_coords_corr = calloc(n_mols, sizeof(vector_3d))) == NULL)
         error_exit("Out of memory in create_com_pos_file");
      if ((scaled_mol_coords_corr = calloc(n_mols, sizeof(vector_3d))) == NULL)
         error_exit("Out of memory in create_com_pos_file");
   }

   /* Read in first data set. */
   read_positions_direct(files.f_posit);

   /* If params.period_switch == 1, calculate quantities that depend
      on box dimensions, calculate scaled atomic coordinates, and apply
      periodic boundary conditions. */
   if (params.period_switch) {
      box_dimensions();
      scaled_atomic_coords();
      periodic_boundary_conditions();
   }

   /* Calculate center of mass positions. */
   center_of_mass_positions();

   /* If params.period_switch == 1, initialize corrected center of mass
      positions. */
   if (params.period_switch)
      for (i_mol = 0; i_mol < consts.n_mols; ++i_mol) {
         scaled_mol_coords_corr[i_mol].x = scaled_mol_coords[i_mol].x;
         scaled_mol_coords_corr[i_mol].y = scaled_mol_coords[i_mol].y;
         scaled_mol_coords_corr[i_mol].z = scaled_mol_coords[i_mol].z;
         mol_coords_corr[i_mol].x = mol_coords[i_mol].x;
         mol_coords_corr[i_mol].y = mol_coords[i_mol].y;
         mol_coords_corr[i_mol].z = mol_coords[i_mol].z;
      }

   /* If params.period_switch == 1, write out corrected center of mass
      positions. Otherwise, write out "raw" center of mass positions. */
   if (params.period_switch)
      fwrite(mol_coords_corr, sizeof(vector_3d), consts.n_mols, f_output);
   else
      fwrite(mol_coords, sizeof(vector_3d), consts.n_mols, f_output);

   /* Loop over data sets. */
   for (i_data = 1; i_data < consts.n_snapshots; ++i_data) {

      /* Read in atomic positions. */
      read_positions_direct(files.f_posit);

      /* If params.period_switch == 1, calculate quantities that depend
         on box dimensions, calculate scaled atomic coordinates, and apply
         periodic boundary conditions. */
      if (params.period_switch) {
         box_dimensions();
         scaled_atomic_coords();
         periodic_boundary_conditions();
      }

      /* Calculate center of mass positions. */
      center_of_mass_positions();

      /* If params.period_switch == 1, calculate corrected scaled
         center of mass positions. */
      if (params.period_switch)
         for (i_mol = 0; i_mol < consts.n_mols; ++i_mol) {
            s_dx = scaled_mol_coords[i_mol].x - scaled_mol_coords_corr[i_mol].x;
            s_dy = scaled_mol_coords[i_mol].y - scaled_mol_coords_corr[i_mol].y;
            s_dz = scaled_mol_coords[i_mol].z - scaled_mol_coords_corr[i_mol].z;
            s_dx -= NINT(s_dx);
            s_dy -= NINT(s_dy);
            s_dz -= NINT(s_dz);
            scaled_mol_coords_corr[i_mol].x += s_dx;
            scaled_mol_coords_corr[i_mol].y += s_dy;
            scaled_mol_coords_corr[i_mol].z += s_dz;
            mol_coords_corr[i_mol].x =
               vars.h[0][0] * scaled_mol_coords_corr[i_mol].x +
               vars.h[0][1] * scaled_mol_coords_corr[i_mol].y +
               vars.h[0][2] * scaled_mol_coords_corr[i_mol].z;
            mol_coords_corr[i_mol].y =
               vars.h[1][1] * scaled_mol_coords_corr[i_mol].y +
               vars.h[1][2] * scaled_mol_coords_corr[i_mol].z;
            mol_coords_corr[i_mol].z =
               vars.h[2][2] * scaled_mol_coords_corr[i_mol].z;
         }

      /* If params.period_switch == 1, write out corrected center of mass
         positions. Otherwise, write out "raw" center of mass positions. */
      if (params.period_switch)
         fwrite(mol_coords_corr, sizeof(vector_3d), consts.n_mols, f_output);
      else
         fwrite(mol_coords, sizeof(vector_3d), consts.n_mols, f_output);
   }

   /* Close input and output files. */
   fclose(files.f_posit);
   fclose(f_output);

   /* If params.period_switch == 1, free up memory. */
   if (params.period_switch) {
      free(mol_coords_corr);
      free(scaled_mol_coords_corr);
   }
}

/* Calculate mean squared center of mass displacements and write them to
   output file. */
void mean_squared_displacement(vector_3d a_1, vector_3d a_2, vector_3d a_3)
{
   FILE *f_input, *f_output;
   int pos_1, pos_2, record_size, i0, j0, j0max,
      i_mol, i_msd, offset, *norm, n_fit, i_fit;
   vector_3d *mol_coords_corr, *mol_coords_corr_0;
   double dx, dy, dz, dx_p, dy_p, dz_p, dum1, dum2, interval,
      *x, *y, *sig, a, b, chi2, q, siga, sigb, y_fit,
      diffus_x, sig_diffus_x,
      diffus_y, sig_diffus_y,
      diffus_z, sig_diffus_z,
      diffus_xy, sig_diffus_xy,
      diffus_xyz, sig_diffus_xyz,
      *x_displ2, *sig_x_displ2,
      *y_displ2, *sig_y_displ2,
      *z_displ2, *sig_z_displ2,
      *xy_displ2, *sig_xy_displ2,
      *xyz_displ2, *sig_xyz_displ2;

   /* Allocate memory for and zero center of mass position and mean
      squared center of mass displacement arrays. */
   if ((mol_coords_corr = calloc(consts.n_mols, sizeof(vector_3d))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((mol_coords_corr_0 = calloc(consts.n_mols, sizeof(vector_3d))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((x_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((sig_x_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((y_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((sig_y_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((z_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((sig_z_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((xy_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((sig_xy_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((xyz_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((sig_xyz_displ2 = calloc(params.n_msd + 1, sizeof(double))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");
   if ((norm = calloc(params.n_msd + 1, sizeof(int))) == NULL)
      error_exit("Out of memory in mean_squared_com_displacement");

   /* Determine size (in bytes) of records. */
   if ((f_input = fopen("pro.com_pos_tmp", "r")) == NULL)
      error_exit("Unable to open com_pos file in mean_squared_com_displacement");
   pos_1 = ftell(f_input);
   fread(mol_coords_corr, sizeof(vector_3d), consts.n_mols, f_input);
   pos_2 = ftell(f_input);
   record_size = pos_2 - pos_1;
   fclose(f_input);

   /* Exit if params.n_msd > consts.n_snapshots - 1. */
   if (params.n_msd > consts.n_snapshots - 1)
      error_exit("n_msd > n_snapshots - 1 in mean_squared_com_displacement.");

   /* Reopen input file. */
   if ((f_input = fopen("pro.com_pos_tmp", "r")) == NULL)
      error_exit("Unable to open com_pos file in mean_squared_com_displacement");

   /* Add contributions to mean squared center of mass displacement. */
   for (i0 = 0; i0 < consts.n_snapshots; ++i0) {

      /* Calculate file position offset for initial data set, and read data
         set starting at that point in the data file. */
      offset = i0 * record_size;
      fseek(f_input, offset, 0);
      fread(mol_coords_corr_0, sizeof(vector_3d), consts.n_mols, f_input);

      /* Loop over time offsets. */
      j0max = MIN(consts.n_snapshots - 1, i0 + params.n_msd);
      for (j0 = i0; j0 <= j0max; ++j0) {

         /* Calculate file position offset for final data set, and read data
            set starting at that point in the data file. */
         offset = j0 * record_size;
         fseek(f_input, offset, 0);
         fread(mol_coords_corr, sizeof(vector_3d), consts.n_mols, f_input);

         /* Add contributions to mean squared center of mass displacement. */
         i_msd = j0 - i0;
         for (i_mol = 0; i_mol < consts.n_mols; ++i_mol) {
            dx = mol_coords_corr[i_mol].x - mol_coords_corr_0[i_mol].x;
            dy = mol_coords_corr[i_mol].y - mol_coords_corr_0[i_mol].y;
            dz = mol_coords_corr[i_mol].z - mol_coords_corr_0[i_mol].z;
            dx_p = dx * a_1.x + dy * a_1.y + dz * a_1.z;
            dy_p = dx * a_2.x + dy * a_2.y + dz * a_2.z;
            dz_p = dx * a_3.x + dy * a_3.y + dz * a_3.z;
            x_displ2[i_msd] += SQR(dx_p);
            y_displ2[i_msd] += SQR(dy_p);
            z_displ2[i_msd] += SQR(dz_p);
         }
         ++norm[i_msd];
      }
   }

   /* Close input file. */
   fclose(f_input);

   /* Normalize mean squared center of mass displacement, calculate
      uncertainties, and write results to output file. Some remarks are
      in order regarding the calculation of uncertainties. The uncertainty
      in <[x(t)-x(0)]^2> is assumed to be equal to the standard deviation
      in [x(t)-x(0)]^2 (which is equal to sqrt(2) * <[x(t)-x(0)]^2> for a
      Gaussian distribution) divided by the square root of the number of
      independent measurements of <[x(t)-x(0)]^2>. Following Allen and
      Tildesley, the number of independent measurements is taken to be
      the total sampling interval divided by twice the correlation
      time for [x(t)-x(0)]^2 , which is estimated to be equal to t (valid
      for large t). We have further assumed that each molecule makes an
      independent contribution to the mean squared displacement. */
   interval = consts.intern_to_ps * consts.delta[0] * params.n_posit;
   f_output = fopen("pro.msd", "w");
   for (i_msd = 0; i_msd <= params.n_msd; ++i_msd) {
      dum1 = 1.0 / (consts.n_mols * norm[i_msd]);
      x_displ2[i_msd] *= dum1;
      y_displ2[i_msd] *= dum1;
      z_displ2[i_msd] *= dum1;
      xy_displ2[i_msd] = x_displ2[i_msd] + y_displ2[i_msd];
      xyz_displ2[i_msd] = x_displ2[i_msd] + y_displ2[i_msd] + z_displ2[i_msd];
      dum2 = sqrt((4.0 * i_msd) / (consts.n_mols * (consts.n_snapshots - i_msd)));
      sig_x_displ2[i_msd] = dum2 * x_displ2[i_msd];
      sig_y_displ2[i_msd] = dum2 * y_displ2[i_msd];
      sig_z_displ2[i_msd] = dum2 * z_displ2[i_msd];
      sig_xy_displ2[i_msd] = sqrt(SQR(sig_x_displ2[i_msd])
         + SQR(sig_y_displ2[i_msd]));
      sig_xyz_displ2[i_msd] = sqrt(SQR(sig_x_displ2[i_msd])
         + SQR(sig_y_displ2[i_msd]) + SQR(sig_z_displ2[i_msd]));
      fprintf(f_output, "%g %g %g %g %g %g %g %g %g %g %g\n",
         i_msd * interval,
         x_displ2[i_msd], sig_x_displ2[i_msd],
         y_displ2[i_msd], sig_y_displ2[i_msd],
         z_displ2[i_msd], sig_z_displ2[i_msd],
         xy_displ2[i_msd], sig_xy_displ2[i_msd],
         xyz_displ2[i_msd], sig_xyz_displ2[i_msd]);
   }
   fclose(f_output);

   /* Calculate diffusion constants from linear fits to the mean square
      displacements. */
   x = dvector(1, params.n_msd + 1);
   y = dvector(1, params.n_msd + 1);
   sig = dvector(1, params.n_msd + 1);
   n_fit = params.n_msd - params.first_msd + 1;
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + params.first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = x_displ2[i_msd];
      sig[i_fit] = sig_x_displ2[i_msd];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_x = b / 2.0;
   sig_diffus_x = sigb / 2.0;
   f_output = fopen("pro.msd_fit_x", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + params.first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = y_displ2[i_msd];
      sig[i_fit] = sig_y_displ2[i_msd];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_y = b / 2.0;
   sig_diffus_y = sigb / 2.0;
   f_output = fopen("pro.msd_fit_y", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + params.first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = z_displ2[i_msd];
      sig[i_fit] = sig_z_displ2[i_msd];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_z = b / 2.0;
   sig_diffus_z = sigb / 2.0;
   f_output = fopen("pro.msd_fit_z", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + params.first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = xy_displ2[i_msd];
      sig[i_fit] = sig_xy_displ2[i_msd];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_xy = b / 4.0;
   sig_diffus_xy = sigb / 4.0;
   f_output = fopen("pro.msd_fit_xy", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + params.first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = xyz_displ2[i_msd];
      sig[i_fit] = sig_xyz_displ2[i_msd];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_xyz = b / 6.0;
   sig_diffus_xyz = sigb / 6.0;
   f_output = fopen("pro.msd_fit_xyz", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);

   /* Write diffusion constants to output file. */
   dum1 = 1.0e-4;
   dum2 = consts.intern_to_ps * consts.delta[0];
   f_output = fopen("pro.diffusion", "w");
   fprintf(f_output, "diffus_x = %g +/- %g angstrom^2 / ps\n",
      diffus_x, sig_diffus_x);
   fprintf(f_output, "         = %g +/- %g cm^2 / s\n",
      dum1 * diffus_x, dum1 * sig_diffus_x);
   fprintf(f_output, "         = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_x, dum2 * sig_diffus_x);
   fprintf(f_output, "diffus_y = %g +/- %g angstrom^2 / ps\n",
      diffus_y, sig_diffus_y);
   fprintf(f_output, "         = %g +/- %g cm^2 / s\n",
      dum1 * diffus_y, dum1 * sig_diffus_y);
   fprintf(f_output, "         = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_y, dum2 * sig_diffus_y);
   fprintf(f_output, "diffus_z = %g +/- %g angstrom^2 / ps\n",
      diffus_z, sig_diffus_z);
   fprintf(f_output, "         = %g +/- %g cm^2 / s\n",
      dum1 * diffus_z, dum1 * sig_diffus_z);
   fprintf(f_output, "         = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_z, dum2 * sig_diffus_z);
   fprintf(f_output, "diffus_xy = %g +/- %g angstrom^2 / ps\n",
      diffus_xy, sig_diffus_xy);
   fprintf(f_output, "          = %g +/- %g cm^2 / s\n",
      dum1 * diffus_xy, dum1 * sig_diffus_xy);
   fprintf(f_output, "          = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_xy, dum2 * sig_diffus_xy);
   fprintf(f_output, "diffus_xyz = %g +/- %g angstrom^2 / ps\n",
      diffus_xyz, sig_diffus_xyz);
   fprintf(f_output, "           = %g +/- %g cm^2 / s\n",
      dum1 * diffus_xyz, dum1 * sig_diffus_xyz);
   fprintf(f_output, "           = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_xyz, dum2 * sig_diffus_xyz);
   fclose(f_output);

   /* Free up memory. */
   free(mol_coords_corr);
   free(mol_coords_corr_0);
   free(x_displ2);
   free(sig_x_displ2);
   free(y_displ2);
   free(sig_y_displ2);
   free(z_displ2);
   free(sig_z_displ2);
   free(xy_displ2);
   free(sig_xy_displ2);
   free(xyz_displ2);
   free(sig_xyz_displ2);
   free(norm);
   free_dvector(x, 1, n_fit);
   free_dvector(y, 1, n_fit);
   free_dvector(sig, 1, n_fit);
}
