/* Periodic boundary condition nonbonded force routines for molecular
   modeling package, with LJ form for vdW interactions. */

#include "build.h"

/* Calculate short-range LJ interactions using a cell search, and using a
   switching function to truncate the pair potential and pair force smoothly. */

void lj_s_period_p(int *mol_species, int *atom_rel, int *atom_mol, int n_atoms, 
   int *n_atoms_per_mol, int ***exclusions, double ****comb_sigma2, 
   double ****comb_four_eps, double **h, double r2_on, double r2_off, 
   double **scaled_atom_coords, double gamma, double three_gamma_over_two,
   double two_over_gamma_cubed, double *pe_vdw_s, 
   int period_switch, int *first, int *last, phantom_cell *phantoms,
   int noffsets, int *nc, int *nc_p, int kc, int *phantom_skip, 
   atom_cell *atom_cells, int *offsets, double **atom_coords)
{
   int i, j, k, i_species, j_species, i_mol, j_mol, i_rel, j_rel, inter_flag;
   double s_sep[NDIM], sep[NDIM], r2_sep, one_over_r2_sep,
      rho_2, rho_6, rho_12, four_epsilon, u_vdw, 
      sw1, sw2, sw3, sw4, s_i[NDIM], l_range[NDIM], shift[NDIM];
   int in_plane, icxy, icx, icy, icz, jcx, jcy, jcz, jc, i_off;
   int ic, j_real;
double junk;

   /* Initialisation. */
   *pe_vdw_s = 0.0;
  
   /* Update real and phantom cells. */ 
   update_cells(period_switch,  n_atoms, h, scaled_atom_coords, atom_coords, 
     atom_cells, first, last, phantoms,  kc, phantom_skip, nc, nc_p); 

   update_phantoms(n_atoms, atom_cells, phantom_skip, phantoms, atom_coords);

   /* Loop over atoms. */
   for (i = 0; i < n_atoms ; ++i) {

      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k) 
        s_i[k] = atom_coords[i][k];
      
      i_mol = atom_mol[i];
      i_rel = atom_rel[i];
      i_species = mol_species[i_mol];

      /* Get cell index. */
      ic = atom_cells[i].cell;
     
      for (i_off = 0; i_off < noffsets; ++i_off) {

         /* Compute index of neighboring cell and get index of the
         first atom in that cell. */
         jc = ic + offsets[i_off];
         j = first[jc];

         /* Loop over atoms in neighboring cell. */
         while (j != -1) {

         /* Get attributes of atom j. */
         j_real = atom_cells[j].real;
         j_mol = atom_mol[j_real];
         j_rel = atom_rel[j_real];

         /* Determine whether interaction should be computed. */
         inter_flag = i_mol != j_mol;
         if (inter_flag || exclusions[i_species][i_rel][j_rel]) {

               /* Calculate scaled pair separation. */
               for (k = 0; k < NDIM; ++k)
                 sep[k] = atom_coords[j][k] - s_i[k];

               r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

               /* If the pair separation is less than r_off, calculate
                  vdW interaction. */

               if (r2_sep < r2_off) {

/*printf("  %d", j_real);*/

                 /* Calculate pair interaction. The vdW pair potential
                     is switched off smoothly between r_on and r_off. */
                  j_species = mol_species[j_mol];
                  one_over_r2_sep = 1.0 / r2_sep;
                  rho_2 = comb_sigma2[i_species][i_rel][j_species][j_rel] 
                          * one_over_r2_sep;
                  rho_6 = CUBE(rho_2);
                  rho_12 = SQR(rho_6);
                  u_vdw = rho_12 - rho_6;
                  four_epsilon = 
                            comb_four_eps[i_species][i_rel][j_species][j_rel];
                  u_vdw *= four_epsilon;
                  if (r2_sep > r2_on) {
                     sw1 = r2_off - r2_sep;
                     sw2 = two_over_gamma_cubed * sw1;
                     sw3 = sw1 * sw2 * (three_gamma_over_two - sw1);
                     sw4 = 6.0 * sw2 * (gamma - sw1);
                     u_vdw *= sw3;
                  }
                  /* Add contributions to potential energy. */
                     *pe_vdw_s += u_vdw;
                  }
               }
            j = atom_cells[j].next;
            }
	}
/*
printf("u = %g\n",junk);
printf("-----------------------------------------\n");
fflush(NULL);
*/
   }
}
/****************************************************************************/
/* Calculate short-range LJ interactions between the molecule i_mol and all the
others, using a cell list, and using a switching function to truncate the pair
potential and pair force smoothly. */

void lj_s_period_single_p(int skip, int mol, int *mol_species, int *atom_rel, 
   int *atom_mol, int n_atoms, int *n_atoms_per_mol, int ***exclusions, 
   double ****comb_sigma2, double ****comb_four_eps, double **h, double r2_on, 
   double r2_off, double **scaled_atom_coords, double gamma, 
   double three_gamma_over_two, double two_over_gamma_cubed, 
   double *pe_vdw_s_single,
   int period_switch, int *first, int *last, phantom_cell *phantoms,
   int noffsets, int *nc, int *nc_p, int kc, int *phantom_skip, 
   atom_cell *atom_cells, int *offsets, double **atom_coords, 
   int *mol_first_atm)
{
   int i, j, k, i_species, j_species, i_mol, j_mol, i_rel, j_rel, inter_flag;
   double s_sep[NDIM], sep[NDIM], r2_sep, one_over_r2_sep,
      rho_2, rho_6, rho_12, four_epsilon, u_vdw, 
      sw1, sw2, sw3, sw4, s_i[NDIM], l_range[NDIM], shift[NDIM];
   int in_plane, icxy, icx, icy, icz, jcx, jcy, jcz, jc, i_off;
   int ic, j_real;

   /* Initialisation. */
   *pe_vdw_s_single = 0.0;

   update_cells_single(mol, period_switch, n_atoms, h, scaled_atom_coords, 
     atom_coords, atom_cells, first, last, phantoms, kc, phantom_skip, nc, 
     nc_p, n_atoms_per_mol, mol_species, mol_first_atm);

/*  printf(" I am here after update cells\n"); */
   update_phantoms_single(mol, n_atoms, atom_cells, phantom_skip, phantoms, 
     atom_coords, mol_first_atm, mol_species, n_atoms_per_mol);

/*  printf(" I am here after update phantom\n"); */

   i_species = mol_species[mol];
   /* Loop over atoms in the molecule i_mol. */
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {

      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k)
        s_i[k] = atom_coords[i][k];
      i_mol = atom_mol[i];
      i_rel = atom_rel[i];

     /* Get cell index. */
      ic = atom_cells[i].cell;

      for (i_off = 0; i_off < noffsets; ++i_off) {

            /* Compute index of neighboring cell and get index of the
               first atom in that cell. */
            jc = ic + offsets[i_off];
            j = first[jc];

/* printf(" atoms %d Cells index ic %d jc %d j %d i_off %d\n", i, ic, jc, j, 
i_off); */

       /* Loop over atoms in neighboring cell. */
         while (j != -1) {

         /* Get attributes of atom j. */
         j_real = atom_cells[j].real;
         j_mol = atom_mol[j_real];
         j_rel = atom_rel[j_real];

         /* Determine whether interaction should be computed. */
         inter_flag = i_mol != j_mol;
         if (inter_flag || exclusions[i_species][i_rel][j_rel]) {

               /* Calculate scaled pair separation. */
               for (k = 0; k < NDIM; ++k)
                 {
                 sep[k] = atom_coords[j][k] - s_i[k];
                 }

               r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

               /* If the pair separation is less than r_off, calculate
                  vdW interaction. */
               if (r2_sep < r2_off) {
/*               if( i ==0){
                   printf("atom %d and phanbors %d x %g y %g z %g\n", i, j_real,
  scaled_atom_coords[j_real][0], scaled_atom_coords[j_real][1],
  scaled_atom_coords[j_real][2]);
                 }
*/

                  /* Calculate pair interaction. The vdW pair potential
                     is switched off smoothly between r_on and r_off. */
                  j_species = mol_species[j_mol];
                  one_over_r2_sep = 1.0 / r2_sep;
                  rho_2 = comb_sigma2[i_species][i_rel][j_species][j_rel] 
                          * one_over_r2_sep;
                  rho_6 = CUBE(rho_2);
                  rho_12 = SQR(rho_6);
                  u_vdw = rho_12 - rho_6;
                  four_epsilon = 
                            comb_four_eps[i_species][i_rel][j_species][j_rel];
                  u_vdw *= four_epsilon;
                  if (r2_sep > r2_on) {
                     sw1 = r2_off - r2_sep;
                     sw2 = two_over_gamma_cubed * sw1;
                     sw3 = sw1 * sw2 * (three_gamma_over_two - sw1);
                     sw4 = 6.0 * sw2 * (gamma - sw1);
                     u_vdw *= sw3;
                  }
                  /* Add contribution to potential energy. */
                     *pe_vdw_s_single += u_vdw;
                  }
               }
              j = atom_cells[j].next;
            }
          }
	}
/* printf(" Pot. ener. for single mol. %g\n", *pe_vdw_s_single); */
}
/****************************************************************************/
