/* Routines for calculating ordering tensor and its eigenvalues
(order parameters) and eigenvectors (the largest eigenvector is
usually taken as the director for the system). */

#include "shared.h"

/* ordering_tensor calculates the instantaneous ordering
tensor for a system of molecules.

input: number of molecules (n_mol)
       array of molecular directors (u)

output: normalized ordering tensor (q) */

void ordering_tensor(int n_mol, double **u, double **q)
{
   int i, j, i_mol;
   double norm;

   /* Zero ordering tensor. */
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         q[i][j] = 0.0;

   /* Add contributions to q. */
   for (i_mol = 0; i_mol < n_mol; ++i_mol)
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            q[i][j] += u[i_mol][i] * u[i_mol][j];

   /* Normalize q. */
   norm = 1.5 / n_mol;
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         q[i][j] *= norm;
   for (i = 0; i < 3; ++i)
      q[i][i] -= 0.5;

   /* Fill in q symmetrically. */
   for (i = 1; i < 3; ++i)
      for (j = 0; j < i; ++j)
         q[i][j] = q[j][i];
}

/* update_ordering_tensor updates the instantaneous ordering
tensor for a system of molecules, taking into account the
reorientation of a single molecule.

input: number of molecules (n_mol)
       array of molecular directors (u)
       label of reoriented molecule (i_mol)
       new director for reoriented molecule (u_new)
       previous ordering tensor (q)

output: updated ordering tensor (q) - the output is written
        over the input ordering tensor */

void update_ordering_tensor(int n_mol, double **u,
                            int i_mol, double *u_new, double **q)
{
   int i, j;
   double norm;

   /* Subtract old contribution from q. */
   norm = 1.5 / n_mol;
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         q[i][j] -= norm * u[i_mol][i] * u[i_mol][j];

   /* Add new contribution to q. */
   for (i = 0; i < 3; ++i)
      for (j = i; j < 3; ++j)
         q[i][j] += norm * u_new[i] * u_new[j];

   /* Fill in q symmetrically. */
   for (i = 1; i < 3; ++i)
      for (j = 0; j < i; ++j)
         q[i][j] = q[j][i];
}

/* eigenvalues calculates the eigenvalues and eigenvectors
of the ordering tensor for a system of molecules.

input: ordering tensor (q)

output: eigenvalues of ordering tensor (lambda)
        eigenvectors of ordering tensor (v) */

void eigenvalues(double **q, double *lambda, double **v)
{
   int i, j, nrot;
   double **q_temp, *lambda_temp, **v_temp;

   /* Allocate memory for scratch arrays. */
   q_temp = dmatrix(1, 3, 1, 3);
   lambda_temp = dvector(1,3);
   v_temp = dmatrix(1, 3, 1, 3);

   /* Copy ordering tensor to scratch array. */
   for (i = 0; i < 3; ++i)
      for (j = 0; j < 3; ++j)
         q_temp[i+1][j+1] = q[i][j];

   /* Calculate eigenvalues and eigenvectors of q and sort them into
      descending order. */
   jacobi(q_temp, 3, lambda_temp, v_temp, &nrot);
   eigsrt(lambda_temp, v_temp, 3);

   /* Copy scratch arrays into passed arrays. */
   for (i = 0; i < 3; ++i) {
      lambda[i] = lambda_temp[i+1];
      for (j = 0; j < 3; ++j)
         v[i][j] = v_temp[i+1][j+1];
   }

   /* Free memory for scratch arrays. */
   free_dmatrix(q_temp, 1, 3, 1, 3);
   free_dvector(lambda_temp, 1, 3);
   free_dmatrix(v_temp, 1, 3, 1, 3);
}
