/* Initialization routines for builder module. */

#include "build.h"
#include "ff.h"

/*****************************************************************************/
/* Set up force field data structures. Lists of force field parameters are
   searched in reverse order, so that only the last entry for a given
   interaction term is used (the search terminates as soon as an appropriate
   entry is found). */
/*
input :  
        Number of species  (n_species) 
        Number of molecules (n_mols)
        Array of the number of atoms per molecule (n_atoms_per_mol)
        Array of the number of angles per molecule (n_angles_per_mol)
        Labels of the three atoms sharing ann angle (temp_angle_1,
                                                    temp_angle_2,
                                                    temp_angle_3) 
        Number of mass entries (n_mass)
        Array of force field mass (mass)
        Array of label of the molecular species (mol_species) 
        Array of the number of bonds per molecules (n_bonds_per_mol) 
        Array of template atom - type of atomic sites (temp_atm_type)
output :
        Array of template atom - mass of atomic sites (temp_atm_mass)
        Array of molecular masses (mol_mass) 
        Array of molecular true masses (mol_mass_true) 
        Array of label of first atom in template molecule (mol_first_atm) 
*/
void set_up_force_field(int n_species, int n_mols, int *n_atoms_per_mol, 
     int n_mass, mass_entry *mass, int *mol_species, int *n_bonds_per_mol,
     double ***temp_atm_mass, char ***temp_atm_type, double **mol_mass, 
     double **mol_mass_true, int **mol_first_atm, int **n_angles_per_mol,
     int **temp_atm_nbr, int ***temp_atm_br, 
     angle_prop ***angle_props, int n_bend, bend_entry *bend)
{
   int i_species, i, j, k, l, i_mass, i_lj, i_e6, i_cnt,
      i_search, br, found, i_mol, atom_count, vector_count;
   char line[MAX_LINE],
        *type_i, *type_j, *type_k, *type_l,
      *type_j_tmp, *type_k_tmp, *type_l_tmp,
      *type_1, *type_2, *type_3, *type_4; 

   int br_1, br_2, i_bend, i1, i2, i3;

   /* Allocate memory for force field data structures. */
   *mol_mass_true = allocate_1d_array(n_species, sizeof(double));
   *mol_mass = allocate_1d_array(n_species, sizeof(double));
   *temp_atm_mass = allocate_1d_array(n_species, sizeof(double*));
     for (i = 0; i < n_species; ++i)
     (*temp_atm_mass)[i] = allocate_1d_array(n_atoms_per_mol[i],
                                                       sizeof(double));

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Zero molecular mass accumulators. */
      (*mol_mass_true)[i_species] = 0.0;
      (*mol_mass)[i_species] = 0.0;

      /* Loop over atoms in template molecule. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

         /* Get template atom type. */
         type_i = temp_atm_type[i_species][i];

         /* Assign mass to template atom. */
         found = 0;
         for (i_mass = n_mass - 1; i_mass >= 0; --i_mass) {
            type_1 = mass[i_mass].type;
            if (strcmp(type_i, type_1) == 0) {
               found = 1;
               break;
            }
         }
         if (found) {
            (*temp_atm_mass)[i_species][i] = mass[i_mass].eff_mass;
            (*mol_mass)[i_species] += mass[i_mass].eff_mass;
            (*mol_mass_true)[i_species] += mass[i_mass].true_mass;
         }
         else {
            printf("mass: %s\n", type_i);
            error_exit("Atomic mass not found in set_up_force_field");
         }
      }
}

   /* Print force field parameters to standard output. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Print species label to standard output. */
      printf("Force field parameters for species %d\n", i_species);

      /* Print mass parameters to standard output. */
      printf("\nmass parameters\n");
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         printf("%d %s %g\n", i, temp_atm_type[i_species][i],
            (*temp_atm_mass)[i_species][i]);
      }
   }

   /* Allocate memory for molecular properties array. */
   *mol_first_atm = allocate_1d_array(n_mols, sizeof(int));

   /* Set up molecular properties array. */
   atom_count = 0;
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {
      i_species = mol_species[i_mol];
      (*mol_first_atm)[i_mol] = atom_count;
      atom_count += n_atoms_per_mol[i_species];
   }

/* count angle per molecule and assign bond angle and spring constant to them
          to them */

   *n_angles_per_mol = allocate_1d_array(n_species, sizeof(int));
   *angle_props = gcalloc(n_species, sizeof(angle_prop*));

   /* Count bond angles per molecule. */
     for (i_species = 0; i_species < n_species; ++i_species) {
      i_cnt = 0;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i)
         for (br_1 = 0; br_1 < temp_atm_nbr[i_species][i] - 1; ++br_1)
            for (br_2 = br_1 + 1; br_2 < temp_atm_nbr[i_species][i]; ++br_2)
               ++i_cnt;
      (*n_angles_per_mol)[i_species] = i_cnt;
  printf("i_species %d angles %d\n", i_species,(*n_angles_per_mol)[i_species]); 

if ((*n_angles_per_mol)[i_species] > 0 )
   (*angle_props)[i_species] = gcalloc((*n_angles_per_mol)[i_species],
         sizeof(angle_prop));
   }

printf(" I am here\n");

   /* Find labels of atom sharing an angle */
     for (i_species = 0; i_species < n_species; ++i_species) {
      i_cnt = 0;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i)
         if (temp_atm_nbr[i_species][i] > 1) {
            type_i = temp_atm_type[i_species][i];
            for (br_1 = 0; br_1 < temp_atm_nbr[i_species][i]- 1; ++br_1)
               for (br_2 = br_1 + 1; br_2 < temp_atm_nbr[i_species][i]; ++br_2) {
                  j = temp_atm_br[i_species][i][br_1];
                  k = temp_atm_br[i_species][i][br_2];

                  type_j = temp_atm_type[i_species][j];
                  type_k = temp_atm_type[i_species][k];

                  (*angle_props)[i_species][i_cnt].atom_1 = j;
                  (*angle_props)[i_species][i_cnt].atom_2 = i;
                  (*angle_props)[i_species][i_cnt].atom_3 = k;

                  found = 0;
                  for (i_search = 0; i_search < 4; ++i_search) {
                     if (i_search == 0) {
                        type_j_tmp = type_j;
                        type_k_tmp = type_k;
                     }
                     else if (i_search == 1) {
                        type_j_tmp = type_j;
                        type_k_tmp = "X";
                     }
                     else if (i_search == 2) {
                        type_j_tmp = "X";
                        type_k_tmp = type_k;
                     }
                     else {
                        type_j_tmp = "X";
                        type_k_tmp = "X";
                     }
                     for (i_bend = n_bend - 1; i_bend >= 0; --i_bend) {
                        type_1 = bend[i_bend].type_1;
                        type_2 = bend[i_bend].type_2;
                        type_3 = bend[i_bend].type_3;
                        if (strcmp(type_j_tmp, type_1) == 0
                           && strcmp(type_i, type_2) == 0
                           && strcmp(type_k_tmp, type_3) == 0
                           || strcmp(type_k_tmp, type_1) == 0
                           && strcmp(type_i, type_2) == 0
                           && strcmp(type_j_tmp, type_3) == 0) {
                           found = 1;
                           break;
                        }
                     }
                     if (found)
                        break;
                  }
                  if (found) {
                       (*angle_props)[i_species][i_cnt].theta_equil
                        = degrees_to_radians * bend[i_bend].theta_equil;
                     (*angle_props)[i_species][i_cnt].sin_theta_equil
                        = sin((*angle_props)[i_species][i_cnt].theta_equil);
                     (*angle_props)[i_species][i_cnt].cos_theta_equil
                        = cos((*angle_props)[i_species][i_cnt].theta_equil);

                        if (bend[i_bend].theta_equil == 0.0
                           || bend[i_bend].theta_equil > 180.0)
         error_exit("Cannot convert bond angle spring constant in set_up_force_field");
             (*angle_props)[i_species][i_cnt].spring_bend = bend[i_bend].spring_bend
                           / SQR(sin((*angle_props)[i_species][i_cnt].theta_equil));

                  }
                  else {
                     printf("bend: %s %s %s\n", type_j, type_i, type_k);
          error_exit("Bond angle bending parameters not found in set_up_force_field");
                  }
                  ++i_cnt;
               }
             }
        }
     /* Print bend parameters to standard output. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      printf("Force field parameters for species %d\n", i_species);

      printf("\nbend parameters\n");
      for (i = 0; i < (*n_angles_per_mol)[i_species]; ++i) {
           i1 = (*angle_props)[i_species][i].atom_1;
           i2 = (*angle_props)[i_species][i].atom_2;
           i3 = (*angle_props)[i_species][i].atom_3;

         printf("%d %s %s %s %g %g\n", i, temp_atm_type[i_species][i1], 
                temp_atm_type[i_species][i2], temp_atm_type[i_species][i3],
                (*angle_props)[i_species][i].theta_equil,
                (*angle_props)[i_species][i].spring_bend);
      }
   }
}

/*****************************************************************************/
void convert_type(int n_species, int *n_atoms_per_mol, 
     char ***temp_atm_type, int ***temp_atm_type_i, type_entry **atm_type, 
     int *n_type)
{
   int i_species, i, j, k, found;
   char *type_i, *type_1;

   /* Allocate memory for force field data structures. */
   *temp_atm_type_i = allocate_1d_array(n_species, sizeof(int*));
     for (i = 0; i < n_species; ++i)
     (*temp_atm_type_i)[i] = allocate_1d_array(n_atoms_per_mol[i],
                                                       sizeof(int));

   /* Initialize counter of atom types. */
   *n_type = 0;

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Loop over atoms in template molecule. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

         /* Get template atom type. */
         type_i = temp_atm_type[i_species][i];

         /* Check if the template atom type has already been assigned. */
         found = 0;
         for (k = 0; k < *n_type; ++k) {
            type_1 = (*atm_type)[k].type;
            if (strcmp(type_i, type_1) == 0) {
               found = 1;
               break;
            }
         }

         if (found) {
            (*temp_atm_type_i)[i_species][i] = (*atm_type)[k].id; 
         }
         else {
          if ((*atm_type = 
               realloc(*atm_type, (*n_type + 1) * sizeof(type_entry))) == NULL)
          error_exit("Out of memory in convert_type");
         
          strcpy((*atm_type)[*n_type].type, type_i); 
          (*atm_type)[*n_type].id = *n_type;
          (*temp_atm_type_i)[i_species][i] = (*atm_type)[*n_type].id; 
          ++(*n_type);
         }
      }
   }

   /* Print force field parameters to standard output. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Print species label to standard output. */
      printf("Force field parameters for species %d\n", i_species);

      /* Print type id parameters to standard output. */
      printf("\nTemplate atom type - id\n");
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         printf("%d %s %d\n", i, temp_atm_type[i_species][i],
            (*temp_atm_type_i)[i_species][i]);
      }
   }

   /* Print array of correspondance between type and integer id. */
   printf("\nAtom type - Id matching\n");
   for (i = 0; i < *n_type; ++i) {

      /* Print type id parameters to standard output. */
         printf("%s %d\n", &(*atm_type)[i].type, (*atm_type)[i].id);
      }
}

/*****************************************************************************/
void adjust_cutoff(int n_type, int **comb_pot, double ***comb_par, 
   double *r_on, double *r_off)
{
int i_type, j_type, found;
double max_min_wca;

/* Initialize counters. */
found = 0;
max_min_wca = 0.0;

/* Loop over the different atomic types. */
for (i_type = 0; i_type < n_type; ++i_type)
  for (j_type = i_type; j_type < n_type; ++j_type){

    /* Compute largest minimum cuttoff if WCA interactions. */
    if (comb_pot[i_type][j_type] == 1) {
      if (comb_par[i_type][j_type][4] > max_min_wca)
         max_min_wca = comb_par[i_type][j_type][4];
    } 
    else
     found = 1;
   }

max_min_wca = sqrt(max_min_wca);

/* If only WCA potential form is present, redefine the cutoff to be
the largest minimum computed pair interaction. */
if (found == 0) {
  *r_off = max_min_wca; 
  *r_on = max_min_wca; 
  printf("Only WCA interactions found :\n");
  printf("  Redefine r_on and r_off : r_on = %g, r_off = %g\n", *r_on, *r_off);
  }
else {  /* if both WCA and another potential form are present. */
  if (*r_off < max_min_wca) {
    printf("r_max WCA = %g, r_off = %g\n", max_min_wca, *r_off);
    error_exit("Conflict between r_off and the computed WCA cutoff."); 
  }
}
}
   
/*****************************************************************************/
/* Set up combination array for nonbonded interactions. */
/*
input :
     Combination rule switch between atomic sites (vdw_rad_switch)
     Number of molecular species (n_species)
     Array of number of atomic sites per template molecule (n_atoms_per_mol)
     Number of LJ parameters (n_lj)
     Array of LJ parameters (lj)
     Array of template atom - type of atomic sites (temp_atm_type)
     Array of template atom - sigma LJ parameter (temp_atm_sigma) 
     Array of template atom - epsilon LJ parameter (temp_atm_eps)
     Array of template atom - charge of atomic sites (temp_atm_chg) 
output :
     Array of sigma combination between 2 atomic sites (comb_sigma)
     Array of sigma2 combination between 2 atomic sites (comb_sigma2)
     Array of r_min2 combination between 2 atomic sites (comb_r_min2)
     Array of epsilon combination between 2 atomic sites (comb_eps)
     Array of 4*epsilon combination between 2 atomic sites (comb_four_eps)
     Array of pair interaction  combination between 2 atomic sites (comb_pot)
     Array of charge combination between 2 atomic sites (comb_chg)
*/

void combination_array(int vdw_switch, int vdw_rad_switch, int n_species, 
     int *n_atoms_per_mol, int n_lj, lj_entry *lj, int n_pot, pot_entry *pot,
     char ***temp_atm_type, double **temp_atm_sigma, double **temp_atm_eps, 
     double **temp_atm_chg, double *****comb_sigma, double *****comb_sigma2, 
     double *****comb_r_min2, double *****comb_eps, double *****comb_four_eps, 
     int *****comb_pot, double *****comb_chg, double *r_on, double *r_off)
{
   char *type_1, *type_2, *type_i, *type_j;
   int i_species, j_species, i_rel, j_rel,
      i_lj, i_e6, i_pot, test_1, test_2, test_3, test_4, found, pot_form;
   double cube_root_of_two, sigma, sigma2, r_min2, eps, four_eps, charge_prod,
      max_min;

   /* Allocate memory for combination array. */
   *comb_sigma = allocate_1d_array(n_species, sizeof(double***));
   *comb_sigma2 = allocate_1d_array(n_species, sizeof(double***));
   *comb_r_min2 = allocate_1d_array(n_species, sizeof(double***));
   *comb_eps = allocate_1d_array(n_species, sizeof(double***));
   *comb_four_eps = allocate_1d_array(n_species, sizeof(double***));
   *comb_pot = allocate_1d_array(n_species, sizeof(int***));
   *comb_chg = allocate_1d_array(n_species, sizeof(double***));
   for (i_species = 0; i_species < n_species; ++i_species) {
      (*comb_sigma)[i_species] = allocate_1d_array(n_atoms_per_mol[i_species],
                                                      sizeof(double**));
      (*comb_sigma2)[i_species] = allocate_1d_array(n_atoms_per_mol[i_species],
                                                      sizeof(double**));
      (*comb_r_min2)[i_species] = allocate_1d_array(n_atoms_per_mol[i_species],
                                                      sizeof(double**));
      (*comb_eps)[i_species] = allocate_1d_array(n_atoms_per_mol[i_species],
                                                      sizeof(double**));
      (*comb_four_eps)[i_species] =allocate_1d_array(n_atoms_per_mol[i_species],
                                                      sizeof(double**));
      (*comb_pot)[i_species] =allocate_1d_array(n_atoms_per_mol[i_species],
                                                      sizeof(int**));
      (*comb_chg)[i_species] = allocate_1d_array(n_atoms_per_mol[i_species],
                                                      sizeof(double**));
      for (i_rel = 0; i_rel < n_atoms_per_mol[i_species]; ++i_rel) {
        (*comb_sigma)[i_species][i_rel]
                      = allocate_1d_array(n_species, sizeof(double*));
        (*comb_sigma2)[i_species][i_rel]
                      = allocate_1d_array(n_species, sizeof(double*));
        (*comb_r_min2)[i_species][i_rel]
                      = allocate_1d_array(n_species, sizeof(double*));
        (*comb_eps)[i_species][i_rel]
                      = allocate_1d_array(n_species, sizeof(double*));
        (*comb_four_eps)[i_species][i_rel]
                      = allocate_1d_array(n_species, sizeof(double*));
        (*comb_pot)[i_species][i_rel]
                      = allocate_1d_array(n_species, sizeof(int*));
        (*comb_chg)[i_species][i_rel]
                      = allocate_1d_array(n_species, sizeof(double*));
         for (j_species = 0; j_species < n_species; ++j_species) {
           (*comb_sigma)[i_species][i_rel][j_species]
               = allocate_1d_array(n_atoms_per_mol[j_species], sizeof(double));
           (*comb_sigma2)[i_species][i_rel][j_species]
               = allocate_1d_array(n_atoms_per_mol[j_species], sizeof(double));
           (*comb_r_min2)[i_species][i_rel][j_species]
               = allocate_1d_array(n_atoms_per_mol[j_species], sizeof(double));
           (*comb_eps)[i_species][i_rel][j_species]
               = allocate_1d_array(n_atoms_per_mol[j_species], sizeof(double));
           (*comb_four_eps)[i_species][i_rel][j_species]
               = allocate_1d_array(n_atoms_per_mol[j_species], sizeof(double));
           (*comb_pot)[i_species][i_rel][j_species]
               = allocate_1d_array(n_atoms_per_mol[j_species], sizeof(int));
           (*comb_chg)[i_species][i_rel][j_species]
               = allocate_1d_array(n_atoms_per_mol[j_species], sizeof(double));
         }
      }
   }
 
   /* Set up combination array for non-bonded interactions. */
   cube_root_of_two = pow(2.0, 1.0 / 3.0);
   max_min = 0.0;
   for (i_species = 0; i_species < n_species; ++i_species)
      for (i_rel = 0; i_rel < n_atoms_per_mol[i_species]; ++i_rel)
         for (j_species = 0; j_species < n_species; ++j_species)
            for (j_rel = 0; j_rel < n_atoms_per_mol[j_species]; ++j_rel) {

               /* Use combination rules to assign vdW parameters. */
               if (vdw_rad_switch == 0) {
                  sigma = sqrt(temp_atm_sigma[i_species][i_rel]
                     * temp_atm_sigma[j_species][j_rel]);
               }
               else {
                  sigma = 0.5 * (temp_atm_sigma[i_species][i_rel]
                     + temp_atm_sigma[j_species][j_rel]);
               }
               sigma2 = SQR(sigma);
               r_min2 = cube_root_of_two * sigma2;
               eps = sqrt(temp_atm_eps[i_species][i_rel]
                     * temp_atm_eps[j_species][j_rel]);
               four_eps = 4.0 * eps;

               /* If atoms are of different types, search for specific 
                  off-diagonal LJ parameters. */
               type_i = temp_atm_type[i_species][i_rel];
               type_j = temp_atm_type[j_species][j_rel];
               if (strcmp(type_i, type_j) != 0) {
                  found = 0;
                  for (i_lj = n_lj - 1; i_lj >= 0; --i_lj) {
                     type_1 = lj[i_lj].type_1;
                     type_2 = lj[i_lj].type_2;
                     test_1 = strcmp(type_i, type_1) == 0;
                     test_2 = strcmp(type_j, type_2) == 0;
                     test_3 = strcmp(type_i, type_2) == 0;
                     test_4 = strcmp(type_j, type_1) == 0;
                     if (test_1 && test_2 || test_3 && test_4) {
                        found = 1;
                        break;
                     }
                  }
                  if (found) {
                     sigma = lj[i_lj].sigma;
                     sigma2 = SQR(lj[i_lj].sigma);
                     r_min2 = cube_root_of_two * sigma2;
                     eps = lj[i_lj].epsilon; 
                     four_eps = 4.0 * eps;
                  }
               }

               /* Define the form of the interaction potential between
                  sites type_i and type_j. */ 
               found = 0;
               for (i_pot = n_pot - 1; i_pot >= 0; --i_pot) {
                  type_1 = pot[i_pot].type_1;
                  type_2 = pot[i_pot].type_2;
                  test_1 = strcmp(type_i, type_1) == 0;
                  test_2 = strcmp(type_j, type_2) == 0;
                  test_3 = strcmp(type_i, type_2) == 0;
                  test_4 = strcmp(type_j, type_1) == 0;
                  if (test_1 && test_2 || test_3 && test_4) {
                    found = 1;
                    break;
                  }
               }
               if (found) 
                 pot_form = pot[i_pot].form;
               else {
                 printf(" Pair interaction %s %s\n", type_i, type_j);
                 error_exit("Entry not found in combination_array.\n");
               }   
              
               /* Define product of electrostatic charges. */ 
               charge_prod = temp_atm_chg[i_species][i_rel]
                  * temp_atm_chg[j_species][j_rel];

               /* Assign the computed quantities to the combinations
                  arrays. */ 
               (*comb_sigma)[i_species][i_rel][j_species][j_rel] = sigma;
               (*comb_sigma2)[i_species][i_rel][j_species][j_rel] = sigma2;
               (*comb_r_min2)[i_species][i_rel][j_species][j_rel] = r_min2;
               (*comb_eps)[i_species][i_rel][j_species][j_rel] = eps;
               (*comb_four_eps)[i_species][i_rel][j_species][j_rel] = four_eps;
               (*comb_pot)[i_species][i_rel][j_species][j_rel] = pot_form;
               (*comb_chg)[i_species][i_rel][j_species][j_rel] = charge_prod;

               /* Determine largest minimum position for LJ potential. */
               /* If we use WCA potential. */
               max_min = MAX(max_min, r_min2);
            }

   /* If params.vdw_switch == 2, reset r_on and r_off to largest minimum 
      position for LJ potential. */
   if (vdw_switch == 2) {
      max_min = sqrt(max_min);
      *r_on = max_min;
      *r_off = max_min;
   }
}

/*****************************************************************************/
/* Set up nonbonded exclusion array. */
void exclusion_array(int n_species, int *n_atoms_per_mol, int *excl_switch, 
       int **temp_atm_nbr, int ***temp_atm_br, int ****exclusions)
{
   char *type_1, *type_2, *type_i, *type_j;
   int i_species, i_rel, j_rel, i, j, k, l, i1, i2, i3;
   int n_parents, ind, parent[100], child[100], lev;

   /* Allocate memory for exclusion array. */
   *exclusions = allocate_1d_array(n_species, sizeof(int**));
   for (i_species = 0; i_species < n_species; ++i_species) {
     (*exclusions)[i_species] = allocate_1d_array(n_atoms_per_mol[i_species],
                                   sizeof(int*));
      for (i_rel = 0; i_rel < n_atoms_per_mol[i_species]; ++i_rel) {
        (*exclusions)[i_species][i_rel] =
               allocate_1d_array(n_atoms_per_mol[i_species], sizeof(int));
      }
   }

   /* Initialize exclusion array. */
   for (i_species = 0; i_species < n_species; ++i_species)
      for (i_rel = 0; i_rel < n_atoms_per_mol[i_species]; ++i_rel)
         for (j_rel = 0; j_rel < n_atoms_per_mol[i_species]; ++j_rel)
            (*exclusions)[i_species][i_rel][j_rel] = 1;

   /* Set up exclusion array to exclude 1-1, 1-2, and 1-3 nonbonded
      interactions. If params.one_four_switch == 0, 1-4 interactions are
      excluded as well. */

   for (i_species = 0; i_species < n_species; ++i_species)
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         
         if (excl_switch[i_species] == -1) {
           for (j = 0; j < n_atoms_per_mol[i_species]; ++j)
             (*exclusions)[i_species][i][j] = 0;
           }
         else {
         (*exclusions)[i_species][i][i] = 0;
         n_parents = 1;
         parent[0] = i;
         for (lev = 1; lev < excl_switch[i_species]; ++lev) {
           ind = 0;
           for (j = 0; j < n_parents; ++j) {
             for (k = 0; k < temp_atm_nbr[i_species][parent[j]]; ++k) {
               child[ind] = temp_atm_br[i_species][parent[j]][k];
               (*exclusions)[i_species][i][child[ind]] = 0;
               ++ind;
             }
           }

         for (k = 0; k < ind; ++k)
           parent[k] = child[k];
         n_parents = ind;
         }
         }
      }

 /* Print exclusion array tp standard output. */
 printf("\nExclusion arrays\n");
  for (i_species = 0; i_species < n_species; ++i_species) {
    printf("\ni_species = %d\n", i_species);
    for (i_rel = 0; i_rel < n_atoms_per_mol[i_species]; ++i_rel)
      for (j_rel = 0; j_rel < n_atoms_per_mol[i_species]; ++j_rel)
        printf("excl[ %d ][ %d ] = %d\n", i_rel, j_rel, (*exclusions)[i_species][i_rel][j_rel]);
  }
printf("\n");

}
/*****************************************************************************/
/* Calculate scaled atomic coordinates. */
/*
input :
       Total number of atoms (n_atoms)
       Array of the inverse box matrix (h_inv)
       Array of atomic coordinates (atom_coords)
output :
       Array of scaled atomic coordinates (scaled_atom_coords)
*/

void scaled_atomic_coords(int n_atoms, double **h_inv, double **atom_coords, 
   double **scaled_atom_coords)
{
   int i;

   for (i = 0; i < n_atoms; ++i) {
      scaled_atom_coords[i][0] = h_inv[0][0] * atom_coords[i][0]
         + h_inv[0][1] * atom_coords[i][1]
         + h_inv[0][2] * atom_coords[i][2];
      scaled_atom_coords[i][1] = h_inv[1][1] * atom_coords[i][1]
         + h_inv[1][2] * atom_coords[i][2];
      scaled_atom_coords[i][2] = h_inv[2][2] * atom_coords[i][2];
   }
}
/*****************************************************************************/
/* Calculate scaled atomic coordinates for a single molecule. */
/*
input :
       Label of the molecule inserted (i_mol)
       Array of number of atoms per template molecule (n_atoms_per_mol)
       Array of first atomic site in each molecule (mol_first_atm)
       Array of the label of molecules (mol_species)
       Array of the inverse box matrix (h_inv)
       Array of atomic coordinates (atom_coords)
output :
       Array of scaled atomic coordinates (scaled_atom_coords)
*/

void scaled_atomic_coords_single(int i_mol, int *n_atoms_per_mol,
       int *mol_first_atm, int *mol_species, double **h_inv,
       double **atom_coords, double **scaled_atom_coords)
{
   int i_species, skip, i;

   i_species = mol_species[i_mol];
   skip = mol_first_atm[i_mol];
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {
      scaled_atom_coords[i][0] = h_inv[0][0] * atom_coords[i][0]
         + h_inv[0][1] * atom_coords[i][1]
         + h_inv[0][2] * atom_coords[i][2];
      scaled_atom_coords[i][1] = h_inv[1][1] * atom_coords[i][1]
         + h_inv[1][2] * atom_coords[i][2];
      scaled_atom_coords[i][2] = h_inv[2][2] * atom_coords[i][2];
   }
}
/*****************************************************************************/

/* Apply periodic boundary conditions. */
/*
input :
      Total number of atoms (n_atoms)
      Array of scaled atomic positions (scaled_atom_coords)
      Array of the box matrix (h)
output :
       Array of the atomic positions (atom_coords)
*/

void periodic_boundary_conditions(int n_atoms, double **h, 
   double **scaled_atom_coords, double **atom_coords)
{
   int i, j;

   for (i = 0; i < n_atoms; ++i) {
      for (j = 0; j < NDIM; ++j)
         scaled_atom_coords[i][j] -= NINT(scaled_atom_coords[i][j]);
      atom_coords[i][0] = h[0][0] * scaled_atom_coords[i][0]
         + h[0][1] * scaled_atom_coords[i][1]
         + h[0][2] * scaled_atom_coords[i][2];
      atom_coords[i][1] = h[1][1] * scaled_atom_coords[i][1]
         + h[1][2] * scaled_atom_coords[i][2];
      atom_coords[i][2] = h[2][2] * scaled_atom_coords[i][2];
   }
}
/*****************************************************************************/
/* Apply periodic boundary conditions to a single molecule. */
void periodic_boundary_conditions_single(int i_mol, int *n_atoms_per_mol,
       int *mol_first_atm, int *mol_species, double **h,
       double **atom_coords, double **scaled_atom_coords)

{
   int i_species, skip, i;

   i_species = mol_species[i_mol];
   skip = mol_first_atm[i_mol];
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {
      scaled_atom_coords[i][0] -= NINT(scaled_atom_coords[i][0]);
      scaled_atom_coords[i][1] -= NINT(scaled_atom_coords[i][1]);
      scaled_atom_coords[i][2] -= NINT(scaled_atom_coords[i][2]);
      atom_coords[i][0] = h[0][0] * scaled_atom_coords[i][0]
         + h[0][1] * scaled_atom_coords[i][1]
         + h[0][2] * scaled_atom_coords[i][2];
      atom_coords[i][1] = h[1][1] * scaled_atom_coords[i][1]
         + h[1][2] * scaled_atom_coords[i][2];
      atom_coords[i][2] = h[2][2] * scaled_atom_coords[i][2];
   }
}

/*****************************************************************************/
/* Calculate inverse box matrix */
/*
input :
	Array of the box matrix (h)
output :
        Array of the inverse box matrix (h_inv) 
*/

void box_dimensions(double **h, double **h_inv, int period_switch, double r_off,
  double mol_max_length)
{
double min_dimension, half_min_dimension;

   /* Calculate inverse of box matrix. */
   h_inv[0][0] = 1.0 / h[0][0];
   h_inv[0][1] = 0.0;
   h_inv[0][2] = 0.0;
   h_inv[1][1] = 1.0 / h[1][1];
   h_inv[1][2] = 0.0;
   h_inv[2][2] = 1.0 / h[2][2];

   /* Exit if system is too small. */
   min_dimension = MIN(h[0][0], h[1][1]);
   min_dimension = MIN(min_dimension, h[2][2]);
   half_min_dimension = 0.5 * min_dimension;
   if (period_switch && r_off > half_min_dimension) 
      error_exit("System too small in box_dimensions");

   /* Exit if box length is smaller than molecular length. */
   if (min_dimension < mol_max_length + r_off)
/*      error_exit("System too small with respect to molecular length"); */
      printf("System too small with respect to molecular length");
}
/* Sample atomic velocities from a Maxwellian distribution. */
void sample_velocities(long *idum, double ***atom_vels, int n_atoms, 
                       double *sqrt_kT_over_m)
{
   int i, k;
   double sqrt_kT_by_m;

   *atom_vels = allocate_2d_array(n_atoms, 3, sizeof(double));

   /* Choose Gaussian-distributed cartesian velocity components for each
      atom, with zero mean and standard deviation equal to sqrt(kT/m). */
   for (i = 0; i < n_atoms; ++i) {
      sqrt_kT_by_m = sqrt_kT_over_m[i];
       for (k = 0; k < NDIM; ++k)
      (*atom_vels)[i][k] = sqrt_kT_by_m * gasdev(idum);  
   }
}
