/* Routines for calculating phase modulated molecular directors. */

#include "shared.h"

/* modulated_directors calculates the phase modulated director 
for all molecules.

input: number of molecules (n_mol)
       number of smectic layers (n_layers)
       phase rotation angle (dphi)
       array of scaled molecule positions (s)
       array of molecular directors (u)

output: array of modulated molecular directors (u_mod) */

void modulated_directors(int n_mol, int n_layers, double dphi,
                         double **s, double **u, double **u_mod)
{
   int i_mol;

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mol; ++i_mol) {

      /* Calculate phase modulated director for molecule i_mol. */
      single_modulated_director(n_layers, dphi, s[i_mol], u[i_mol], u_mod[i_mol]);
   }
}

/* single_modulated_director calculates the phase modulated director
for a single molecule.

input: number of smectic layers (n_layers)
       phase rotation angle (dphi)
       scaled molecule position (s)
       molecular director (u)

output: modulated molecular director (u_mod) */

void single_modulated_director(int n_layers, double dphi,
                               double **scaled_mol_coords, double **u, double *u_mod)
{
   int layer;
   double phi, c_phi, s_phi;

   /* Assign molecule to a smectic layer. */
   layer = layer_assignment(n_layers, scaled_mol_coords);

   /* Compute rotation angle. */
   phi = layer * dphi;

   /* Rotate director by phi about layer normal (assumed to be z-axis). */
   c_phi = cos(phi);
   s_phi = sin(phi);
   u_mod[0] = c_phi * u[0] + s_phi * u[1];
   u_mod[1] = - s_phi * u[0] + c_phi * u[1];
   u_mod[2] = u[2];
}
