/* Initialization routines for builder module. */

#include "build.h"
#include "ff.h"

/*****************************************************************************/
/* Carry out various initializations. */
/*
input :  
        Molecular structures files names (struct_file) 
        Number of species  (n_species) 
        Number of unit cells (n_unit_cells) 
        Array of the number of molecules of each specie (n_mol_per_unit_cell)
output :
        Array of number of atoms per molecule (n_atoms_per_mol)
        Array of the number of molecules per species (n_mols_per_species)
        Array of the number of bonds per template molecule (n_bonds_per_mol)
        Array of template atom - label of atomic sites (temp_atm_lab)
        Array of template atom - type of atomic sites (temp_atm_type)
        Array of template atom - relative atomic site positions (temp_atm_pos)
        Array of template atom - charge of atomic sites (temp_atm_chg)
        Array of template atom - branching number of atomic sites (temp_atm_nbr)
        Array of template atom - label of branched atoms (temp_atm_br)
        Array of template atom - order of branched bonds (temp_atm_br)
        Arrays of template molecule - labels of atoms sharing a bond 
                                                               (temp_bonds_1,
                                                                temp_bonds_2)
        Array of template molecule - order of a given bond (temp_bonds_ord)
        Total number of molecules (n_mols)
        Total number of atoms (n_atoms)
        Array of molecular species label (mol_species)
*/

void start_new_build(char **struct_file, long *i_ran, int trial_switch,
    int n_species, int n_unit_cells,
    int *n_mols_per_unit_cell, int **n_atoms_per_mol, int **n_mols_per_species, 
    int **n_bonds_per_mol, int ***temp_atm_lab, char ****temp_atm_type, 
    double ****temp_atm_pos, double ***temp_atm_chg, int ***temp_atm_nbr, 
    int ****temp_atm_br, double ****temp_atm_ord, int ***temp_bonds_1, 
    int ***temp_bonds_2, double ***temp_bonds_ord, int *n_mols, int *n_atoms, 
    int **mol_species)
{

   int i_species, i, i_count, found, *acc_species; 

   /* Allocate memory for molecular structure arrays. */
   *n_atoms_per_mol = allocate_1d_array(n_species, sizeof(int));
   *n_mols_per_species = allocate_1d_array(n_species, sizeof(int));
   *n_bonds_per_mol = allocate_1d_array(n_species, sizeof(int));
   *temp_atm_lab = allocate_1d_array(n_species, sizeof(int));
   *temp_atm_type = allocate_1d_array(n_species, sizeof(char));
   *temp_atm_pos = allocate_1d_array(n_species, sizeof(double));
   *temp_atm_chg = allocate_1d_array(n_species, sizeof(double));
   *temp_atm_nbr = allocate_1d_array(n_species, sizeof(int));
   *temp_atm_br = allocate_1d_array(n_species, sizeof(int));
   *temp_atm_ord = allocate_1d_array(n_species, sizeof(double));
   *temp_bonds_1 = allocate_1d_array(n_species, sizeof(int));
   *temp_bonds_2 = allocate_1d_array(n_species, sizeof(int));
   *temp_bonds_ord = allocate_1d_array(n_species, sizeof(double));

   /* Read molecular structures from struct files. */
   for (i_species = 0; i_species < n_species; ++i_species)
      read_str(struct_file[i_species], &(*n_atoms_per_mol)[i_species], 
         &(*temp_atm_lab)[i_species], &(*temp_atm_type)[i_species], 
         &(*temp_atm_pos)[i_species], &(*temp_atm_chg)[i_species], 
         &(*temp_atm_nbr)[i_species], &(*temp_atm_br)[i_species], 
         &(*temp_atm_ord)[i_species], &(*n_bonds_per_mol)[i_species], 
         &(*temp_bonds_1)[i_species], &(*temp_bonds_2)[i_species], 
         &(*temp_bonds_ord)[i_species]);

   /* Calculate number of unit cells. For a homogeneous fluid, the unit cell
      is the entire system; for a smectic, the unit cell is one smectic layer;
      for a columnar phase, the unit cell is one column; and for a crystal,
      the unit cell has its usual meaning. */
   /* Calculate total number of molecules per species. */
   for (i_species = 0; i_species < n_species; ++i_species)
      (*n_mols_per_species)[i_species] = n_unit_cells
         * n_mols_per_unit_cell[i_species];

   /* Count molecules and atoms. */
   *n_mols = 0;
   *n_atoms = 0;
   for (i_species = 0; i_species < n_species; ++i_species) {
      *n_mols += (*n_mols_per_species)[i_species];
      *n_atoms += (*n_mols_per_species)[i_species]
         * (*n_atoms_per_mol)[i_species];
   }

/* Allocate memory for molecular species array. */
*mol_species = allocate_1d_array(*n_mols, sizeof(int));
acc_species = allocate_1d_array(n_species, sizeof(int));

/* Define the order in which the molecules will be inserted 
in the box. */
/* If more than one specie is present, pick up molecules in a 
random order if trial_switch = 1. */
if ((trial_switch) && (n_species > 1)) {

/* Initialize species accumulators. */
for (i_species = 0; i_species < n_species; ++i_species)
  acc_species[i_species] = 0;

  for (i = 0; i < *n_mols; ++i) {
    found = 0;
    do {
    i_count = (int) (n_species * ran3(i_ran)); 
    if (i_count == n_species) 
      i_count = 0;
    ++acc_species[i_count];
    if (acc_species[i_count] <= (*n_mols_per_species)[i_count])
     {
     (*mol_species)[i] = i_count;
     found = 1;
     }
    } while (found == 0);
  }
}
else {
/* Pick up molecular specie in the order they are read in build.params. */
  i_count = 0;
  for (i_species = 0; i_species < n_species; ++i_species) {
    for (i = 0; i < (*n_mols_per_species)[i_species]; ++i) 
      (*mol_species)[i + i_count] = i_species;
    i_count += (*n_mols_per_species)[i_species];
    }
  }  
}

/*****************************************************************************/
/* Set up force field data structures. Lists of force field parameters are
   searched in reverse order, so that only the last entry for a given
   interaction term is used (the search terminates as soon as an appropriate
   entry is found). */
/*
input :  
        Number of species  (n_species) 
        Number of molecules (n_mols)
        Array of the number of atoms per molecule (n_atoms_per_mol)
        Number of mass entries (n_mass)
        Array of force field mass (mass)
        Number of LJ entries (n_lj)
        Array of force field LJ parameters (lj) 
        Array of label of the molecular species (mol_species) 
        Array of the number of bonds per molecules (n_bonds_per_mol) 
        Array of template atom - type of atomic sites (temp_atm_type)
output :
        Array of template atom - mass of atomic sites (temp_atm_mass)
        Arrays of template atom - LJ atomic site parameters (temp_atm_sigma,
                                                             temp_atm_eps)
        Array of molecular masses (mol_mass) 
        Array of molecular true masses (mol_mass_true) 
        Array of label of first atom in template molecule (mol_first_atm) 
*/
void set_up_force_field_old(int n_species, int n_mols, int *n_atoms_per_mol, 
     int n_mass, mass_entry *mass, int n_lj, lj_entry *lj, int *mol_species, 
     int *n_bonds_per_mol, double ***temp_atm_mass, char ***temp_atm_type, 
     double ***temp_atm_sigma, double ***temp_atm_eps, double **mol_mass, 
     double **mol_mass_true, int **mol_first_atm)
{
   int i_species, i, j, k, l, i_mass, i_lj, i_e6, i_cnt,
      i_search, br, found, i_mol, atom_count, vector_count;
   char line[MAX_LINE],
      *type_i, *type_1, *type_2;

   /* Allocate memory for force field data structures. */
   *mol_mass_true = allocate_1d_array(n_species, sizeof(double));
   *mol_mass = allocate_1d_array(n_species, sizeof(double));
   *temp_atm_mass = allocate_1d_array(n_species, sizeof(double*));
   *temp_atm_sigma = allocate_1d_array(n_species, sizeof(double*));
   *temp_atm_eps = allocate_1d_array(n_species, sizeof(double*));
     for (i = 0; i < n_species; ++i) {
     (*temp_atm_mass)[i] = allocate_1d_array(n_atoms_per_mol[i],
                                                          sizeof(double));
     (*temp_atm_sigma)[i] = allocate_1d_array(n_atoms_per_mol[i],
                                                          sizeof(double));
     (*temp_atm_eps)[i] = allocate_1d_array(n_atoms_per_mol[i],
                                                          sizeof(double));
     }

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Zero molecular mass accumulators. */
      (*mol_mass_true)[i_species] = 0.0;
      (*mol_mass)[i_species] = 0.0;

      /* Loop over atoms in template molecule. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

         /* Get template atom type. */
         type_i = temp_atm_type[i_species][i];

         /* Assign mass to template atom. */
         found = 0;
         for (i_mass = n_mass - 1; i_mass >= 0; --i_mass) {
            type_1 = mass[i_mass].type;
            if (strcmp(type_i, type_1) == 0) {
               found = 1;
               break;
            }
         }
         if (found) {
            (*temp_atm_mass)[i_species][i] = mass[i_mass].eff_mass;
            (*mol_mass)[i_species] += mass[i_mass].eff_mass;
            (*mol_mass_true)[i_species] += mass[i_mass].true_mass;
         }
         else {
            printf("mass: %s\n", type_i);
            error_exit("Atomic mass not found in set_up_force_field");
         }

         /* Assign van der Waals (LJ) parameters to template atom. */
         found = 0;
         for (i_lj = n_lj - 1; i_lj >= 0; --i_lj) {
            type_1 = lj[i_lj].type_1;
            type_2 = lj[i_lj].type_2;
            if (strcmp(type_i, type_1) == 0 && strcmp(type_i, type_2) == 0) {
               found = 1;
               break;
            }
         }
         if (found) {
            (*temp_atm_sigma)[i_species][i] = lj[i_lj].sigma;
            (*temp_atm_eps)[i_species][i] = lj[i_lj].epsilon;
         }
         else {
            printf("lj: %s\n", type_i);
            error_exit("LJ parameters not found in set_up_force_field");
         }

      }
     }

   /* Print force field parameters to standard output. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Print species label to standard output. */
      printf("Force field parameters for species %d\n", i_species);

      /* Print mass parameters to standard output. */
      printf("\nmass parameters\n");
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         printf("%d %s %g\n", i, temp_atm_type[i_species][i],
            (*temp_atm_mass)[i_species][i]);
      }

      /* Print van der Waals (LJ, exp-6) parameters to standard output. */
      printf("\nLJ parameters\n");
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         printf("%d %s %g %g\n", i, temp_atm_type[i_species][i],
            (*temp_atm_sigma)[i_species][i], (*temp_atm_eps)[i_species][i]); 
      }
   }

   /* Allocate memory for molecular properties array. */
   *mol_first_atm = allocate_1d_array(n_mols, sizeof(int));

   /* Set up molecular properties array. */
   atom_count = 0;
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {
      i_species = mol_species[i_mol];
      (*mol_first_atm)[i_mol] = atom_count;
      atom_count += n_atoms_per_mol[i_species];
   }
}


/*****************************************************************************/
/* Set up force field data structures. Lists of force field parameters are
   searched in reverse order, so that only the last entry for a given
   interaction term is used (the search terminates as soon as an appropriate
   entry is found). */
/*
input :  
        Number of species  (n_species) 
        Number of molecules (n_mols)
        Array of the number of atoms per molecule (n_atoms_per_mol)
        Number of mass entries (n_mass)
        Array of force field mass (mass)
        Array of label of the molecular species (mol_species) 
        Array of the number of bonds per molecules (n_bonds_per_mol) 
        Array of template atom - type of atomic sites (temp_atm_type)
output :
        Array of template atom - mass of atomic sites (temp_atm_mass)
        Array of molecular masses (mol_mass) 
        Array of molecular true masses (mol_mass_true) 
        Array of label of first atom in template molecule (mol_first_atm) 
*/
void set_up_force_field(int n_species, int n_mols, int *n_atoms_per_mol, 
     int n_mass, mass_entry *mass, int *mol_species, int *n_bonds_per_mol,
     double ***temp_atm_mass, char ***temp_atm_type, double **mol_mass, 
     double **mol_mass_true, int **mol_first_atm)
{
   int i_species, i, j, k, l, i_mass, i_lj, i_e6, i_cnt,
      i_search, br, found, i_mol, atom_count, vector_count;
   char line[MAX_LINE],
      *type_i, *type_1, *type_2;

   /* Allocate memory for force field data structures. */
   *mol_mass_true = allocate_1d_array(n_species, sizeof(double));
   *mol_mass = allocate_1d_array(n_species, sizeof(double));
   *temp_atm_mass = allocate_1d_array(n_species, sizeof(double*));
     for (i = 0; i < n_species; ++i)
     (*temp_atm_mass)[i] = allocate_1d_array(n_atoms_per_mol[i],
                                                       sizeof(double));

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Zero molecular mass accumulators. */
      (*mol_mass_true)[i_species] = 0.0;
      (*mol_mass)[i_species] = 0.0;

      /* Loop over atoms in template molecule. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

         /* Get template atom type. */
         type_i = temp_atm_type[i_species][i];

         /* Assign mass to template atom. */
         found = 0;
         for (i_mass = n_mass - 1; i_mass >= 0; --i_mass) {
            type_1 = mass[i_mass].type;
            if (strcmp(type_i, type_1) == 0) {
               found = 1;
               break;
            }
         }
         if (found) {
            (*temp_atm_mass)[i_species][i] = mass[i_mass].eff_mass;
            (*mol_mass)[i_species] += mass[i_mass].eff_mass;
            (*mol_mass_true)[i_species] += mass[i_mass].true_mass;
         }
         else {
            printf("mass: %s\n", type_i);
            error_exit("Atomic mass not found in set_up_force_field");
         }
      }
}

   /* Print force field parameters to standard output. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Print species label to standard output. */
      printf("Force field parameters for species %d\n", i_species);

      /* Print mass parameters to standard output. */
      printf("\nmass parameters\n");
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         printf("%d %s %g\n", i, temp_atm_type[i_species][i],
            (*temp_atm_mass)[i_species][i]);
      }
   }

   /* Allocate memory for molecular properties array. */
   *mol_first_atm = allocate_1d_array(n_mols, sizeof(int));

   /* Set up molecular properties array. */
   atom_count = 0;
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {
      i_species = mol_species[i_mol];
      (*mol_first_atm)[i_mol] = atom_count;
      atom_count += n_atoms_per_mol[i_species];
   }
}

/*****************************************************************************/
void convert_type(int n_species, int *n_atoms_per_mol, 
     char ***temp_atm_type, int ***temp_atm_type_i, type_entry **atm_type, 
     int *n_type)
{
   int i_species, i, j, k, found;
   char *type_i, *type_1;

   /* Allocate memory for force field data structures. */
   *temp_atm_type_i = allocate_1d_array(n_species, sizeof(int*));
     for (i = 0; i < n_species; ++i)
     (*temp_atm_type_i)[i] = allocate_1d_array(n_atoms_per_mol[i],
                                                       sizeof(int));

   /* Initialize counter of atom types. */
   *n_type = 0;

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Loop over atoms in template molecule. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

         /* Get template atom type. */
         type_i = temp_atm_type[i_species][i];

         /* Check if the template atom type has already been assigned. */
         found = 0;
         for (k = 0; k < *n_type; ++k) {
            type_1 = (*atm_type)[k].type;
            if (strcmp(type_i, type_1) == 0) {
               found = 1;
               break;
            }
         }

         if (found) {
            (*temp_atm_type_i)[i_species][i] = (*atm_type)[k].id; 
         }
         else {
          if ((*atm_type = 
               realloc(*atm_type, (*n_type + 1) * sizeof(type_entry))) == NULL)
          error_exit("Out of memory in convert_type");
         
          strcpy((*atm_type)[*n_type].type, type_i); 
          (*atm_type)[*n_type].id = *n_type;
          (*temp_atm_type_i)[i_species][i] = (*atm_type)[*n_type].id; 
          ++(*n_type);
         }
      }
   }

   /* Print force field parameters to standard output. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Print species label to standard output. */
      printf("Force field parameters for species %d\n", i_species);

      /* Print type id parameters to standard output. */
      printf("\nTemplate atom type - id\n");
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         printf("%d %s %d\n", i, temp_atm_type[i_species][i],
            (*temp_atm_type_i)[i_species][i]);
      }
   }

   /* Print array of correspondance between type and integer id. */
   printf("\nAtom type - Id matching\n");
   for (i = 0; i < *n_type; ++i) {

      /* Print type id parameters to standard output. */
         printf("%s %d\n", &(*atm_type)[i].type, (*atm_type)[i].id);
      }
}
/*****************************************************************************/
/* Add an entry type if we are dealing with a pore (or restricted geometry
   in general) */

void add_type(type_entry **atm_type, int *n_type)
{

    /* Reallocate memory. */
    if ((*atm_type =
         realloc(*atm_type, (*n_type + 1) * sizeof(type_entry))) == NULL)
    error_exit("Out of memory in convert_type");

    /* For now the label for the wall is hardwired. */
    strcpy((*atm_type)[*n_type].type, "Wall");
    (*atm_type)[*n_type].id = *n_type;
    ++(*n_type);

    /* Print type id parameters to standard output. */
    printf("\nRestricted geometry - Id matching\n");
    printf("%s %d\n", &(*atm_type)[*n_type - 1].type, 
            (*atm_type)[*n_type - 1].id);

}

/*****************************************************************************/
/* Set up combination array for nonbonded interactions. */
/*
input :
     Combination rule switch between atomic sites (vdw_rad_switch)
     Overlap criterion between atomic sites (overlap_criterion)
     Number of molecular species (n_species)
     Array of number of atomic sites per template molecule (n_atoms_per_mol)
     Number of LJ parameters (n_lj)
     Array of LJ parameters (lj)
     Array of template atom - type of atomic sites (temp_atm_type)
     Array of template atom - sigma LJ parameter (temp_atm_sigma) 
output :
     Array of overlaping criterion between 2 atomic sites (combine)
*/

void combination_array(double overlap_criterion, int n_type, int **comb_pot,
   double ***comb_par, double ***combine)  
{
int i_type, j_type, pot_form;

   /* Allocate memory for combination array. */
   *combine = allocate_2d_array(n_type, n_type, sizeof(double));
 
   /* Set up combination array for non-bonded interactions. */
   for (i_type = 0; i_type < n_type; ++i_type)
     for (j_type = 0; j_type < n_type; ++j_type) {

     pot_form = comb_pot[i_type][j_type];
     /* Set overlap criterion based on the zero-crossing point
     of the vdW potential. */

     switch (pot_form) {
     /*******************************************************************/
     /*******************    0 = LJ potential    ************************/
     /*******************************************************************/
     case 0 :

          (*combine)[i_type][j_type] = (*combine)[j_type][i_type] =
                      SQR(overlap_criterion) * comb_par[i_type][j_type][2];

          break;

     /*******************************************************************/
     /*******************    1 = WCA potential    ***********************/
     /*******************************************************************/
     case 1 :

          (*combine)[i_type][j_type] = (*combine)[j_type][i_type] =
                      SQR(overlap_criterion) * comb_par[i_type][j_type][2];

          break;
      }

    /* Print to standard output the overlap criterion between type. */
    printf("types %d, %d : r2_overlap = %g\n", i_type, j_type, (*combine)[i_type][j_type]);
    }
}

/*****************************************************************************/
/* Set up nonbonded exclusion array. */
void exclusion_array(int n_species, int *n_atoms_per_mol, int one_four_switch, 
       int **temp_atm_nbr, int ***temp_atm_br, int ****exclusions)
{
   char *type_1, *type_2, *type_i, *type_j;
   int i_species, i_rel, j_rel, i, j, k, l, i1, i2, i3;

   /* Allocate memory for exclusion array. */
   *exclusions = allocate_1d_array(n_species, sizeof(int**));
   for (i_species = 0; i_species < n_species; ++i_species) {
     (*exclusions)[i_species] = allocate_1d_array(n_atoms_per_mol[i_species],
                                   sizeof(int*));
      for (i_rel = 0; i_rel < n_atoms_per_mol[i_species]; ++i_rel) {
        (*exclusions)[i_species][i_rel] =
               allocate_1d_array(n_atoms_per_mol[i_species], sizeof(int));
      }
   }

   /* Initialize exclusion array. */
   for (i_species = 0; i_species < n_species; ++i_species)
      for (i_rel = 0; i_rel < n_atoms_per_mol[i_species]; ++i_rel)
         for (j_rel = 0; j_rel < n_atoms_per_mol[i_species]; ++j_rel)
            (*exclusions)[i_species][i_rel][j_rel] = 1;

   /* Set up exclusion array to exclude 1-1, 1-2, and 1-3 nonbonded
      interactions. If params.one_four_switch == 0, 1-4 interactions are
      excluded as well. */
   for (i_species = 0; i_species < n_species; ++i_species)
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         (*exclusions)[i_species][i][i] = 0;
         for (j = 0; j < temp_atm_nbr[i_species][i]; ++j) {
            i1 = temp_atm_br[i_species][i][j];
            (*exclusions)[i_species][i][i1] = 0;
            for (k = 0; k < temp_atm_nbr[i_species][i1]; ++k) {
               i2 = temp_atm_br[i_species][i1][k];
               (*exclusions)[i_species][i][i2] = 0;
               if (!one_four_switch)
                  for (l = 0; l < temp_atm_nbr[i_species][i2]; ++l) {
                     i3 = temp_atm_br[i_species][i2][l];
                     (*exclusions)[i_species][i][i3] = 0;
                  }
            }
         }
      }
}

/*****************************************************************************/
/* Calculate scaled atomic coordinates for a single molecule. */
/*
input :
       Label of the molecule inserted (i_mol)
       Array of number of atoms per template molecule (n_atoms_per_mol)
       Array of first atomic site in each molecule (mol_first_atm)
       Array of the label of molecules (mol_species)
       Array of the inverse box matrix (h_inv)
       Array of atomic coordinates (atom_coords)
output :
       Array of scaled atomic coordinates (scaled_atom_coords)
*/

void scaled_atomic_coords_single(int i_mol, int *n_atoms_per_mol,
       int *mol_first_atm, int *mol_species, double **h_inv, 
       double **atom_coords, double **scaled_atom_coords)
{
   int i_species, skip, i;

   i_species = mol_species[i_mol];
   skip = mol_first_atm[i_mol];
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {
      scaled_atom_coords[i][0] = h_inv[0][0] * atom_coords[i][0]
         + h_inv[0][1] * atom_coords[i][1]
         + h_inv[0][2] * atom_coords[i][2];
      scaled_atom_coords[i][1] = h_inv[1][1] * atom_coords[i][1]
         + h_inv[1][2] * atom_coords[i][2];
      scaled_atom_coords[i][2] = h_inv[2][2] * atom_coords[i][2];
   }
}

/*****************************************************************************/
/* Calculate inverse box matrix */
/*
input :
	Array of the box matrix (h)
output :
        Array of the inverse box matrix (h_inv) 
*/

void box_inverse(double **h, double **h_inv)
{
   /* Calculate inverse of box matrix. */
   h_inv[0][0] = 1.0 / h[0][0];
   h_inv[0][1] = 0.0;
   h_inv[0][2] = 0.0;
   h_inv[1][1] = 1.0 / h[1][1];
   h_inv[1][2] = 0.0;
   h_inv[2][2] = 1.0 / h[2][2];
}

/*****************************************************************************/
/* Center and orient template molecules. */
/*
input :
       Number of molecular species (n_species)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of principal molecular axis aligned along the frame-z axis
                                                              (principal)
       Array of template atom - mass of atomic sites (temp_atm_mass)
       Array of template atom - position of atomic sites (temp_atm_pos)
output :
       Array of template atom - position of atomic sites (temp_atm_pos)
*/

void adjust_template_molecules(int n_species, int *n_atoms_per_mol, 
      int *principal, double **temp_atm_mass, double ***temp_atm_pos)
{
   int i_species, i, j, k, nrot;
   double **inertia, *d, **v, x_cm, y_cm, z_cm, x_tmp, y_tmp, z_tmp, test,
      x_min, y_min, z_min, x_max, y_max, z_max, x_cg, y_cg, z_cg, mol_mass,
      x_range, y_range, z_range;

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Calculate center of mass position of template molecule. */
      x_cm = y_cm = z_cm = 0.0;
      mol_mass = 0.0;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         x_cm += temp_atm_mass[i_species][i] 
                        * temp_atm_pos[i_species][i][0];
         y_cm += temp_atm_mass[i_species][i] 
                        * temp_atm_pos[i_species][i][1];
         z_cm += temp_atm_mass[i_species][i] 
                        * temp_atm_pos[i_species][i][2];
      mol_mass +=  temp_atm_mass[i_species][i];
      }
      x_cm /= mol_mass;
      y_cm /= mol_mass;
      z_cm /= mol_mass;

      /* Translate center of mass of template molecule to origin. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         temp_atm_pos[i_species][i][0] -= x_cm;
         temp_atm_pos[i_species][i][1] -= y_cm;
         temp_atm_pos[i_species][i][2] -= z_cm;
      }

      /* Calculate inertia tensor of template molecule. */
      inertia = dmatrix(1, 3, 1, 3);
      for (j = 1; j <= 3; ++j)
         for (k = 1; k <= 3; ++k)
            inertia[j][k] = 0.0;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         inertia[1][1] += temp_atm_mass[i_species][i]
            * (SQR(temp_atm_pos[i_species][i][1])
            + SQR(temp_atm_pos[i_species][i][2]));
         inertia[2][2] += temp_atm_mass[i_species][i]
            * (SQR(temp_atm_pos[i_species][i][0])
            + SQR(temp_atm_pos[i_species][i][2]));
         inertia[3][3] += temp_atm_mass[i_species][i]
            * (SQR(temp_atm_pos[i_species][i][0])
            + SQR(temp_atm_pos[i_species][i][1]));
         inertia[1][2] -= temp_atm_mass[i_species][i]
            * temp_atm_pos[i_species][i][0] * temp_atm_pos[i_species][i][1];
         inertia[1][3] -= temp_atm_mass[i_species][i]
            * temp_atm_pos[i_species][i][0] * temp_atm_pos[i_species][i][2];
         inertia[2][3] -= temp_atm_mass[i_species][i]
            * temp_atm_pos[i_species][i][1] * temp_atm_pos[i_species][i][2];
      }
      inertia[2][1] = inertia[1][2];
      inertia[3][1] = inertia[1][3];
      inertia[3][2] = inertia[2][3];

      /* Calculate eigenvalues and eigenvectors of inertia tensor. */
      d = dvector(1, 3);
      v = dmatrix(1, 3, 1, 3);
      jacobi(inertia, 3, d, v, &nrot);
      eigsrt(d, v, 3);

      /* Make sure that eigenvectors constitute a right-handed 
         coordinate system, and make it so if not. */
      test = (v[2][1] * v[3][2] - v[3][1] * v[2][2]) * v[1][3]
         + (v[3][1] * v[1][2] - v[1][1] * v[3][2]) * v[2][3]
         + (v[1][1] * v[2][2] - v[2][1] * v[1][2]) * v[3][3];
      if (test < 0.0) {
         v[1][3] = - v[1][3];
         v[2][3] = - v[2][3];
         v[3][3] = - v[3][3];
      }

      /* Rotate template molecule so that the principal axes of the inertia
         tensor coincide with the space-fixed cartesian axes. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         if (principal[i_species] == 0) {
            x_tmp = v[1][1] * temp_atm_pos[i_species][i][0]
               + v[2][1] * temp_atm_pos[i_species][i][1]
               + v[3][1] * temp_atm_pos[i_species][i][2];
            y_tmp = v[1][2] * temp_atm_pos[i_species][i][0]
               + v[2][2] * temp_atm_pos[i_species][i][1]
               + v[3][2] * temp_atm_pos[i_species][i][2];
            z_tmp = v[1][3] * temp_atm_pos[i_species][i][0]
               + v[2][3] * temp_atm_pos[i_species][i][1]
               + v[3][3] * temp_atm_pos[i_species][i][2];
         }
         else if (principal[i_species] == 1) {
            x_tmp = v[1][2] * temp_atm_pos[i_species][i][0]
               + v[2][2] * temp_atm_pos[i_species][i][1]
               + v[3][2] * temp_atm_pos[i_species][i][2];
            y_tmp = v[1][3] * temp_atm_pos[i_species][i][0]
               + v[2][3] * temp_atm_pos[i_species][i][1]
               + v[3][3] * temp_atm_pos[i_species][i][2];
            z_tmp = v[1][1] * temp_atm_pos[i_species][i][0]
               + v[2][1] * temp_atm_pos[i_species][i][1]
               + v[3][1] * temp_atm_pos[i_species][i][2];
         }
         else {
            x_tmp = v[1][3] * temp_atm_pos[i_species][i][0]
               + v[2][3] * temp_atm_pos[i_species][i][1]
               + v[3][3] * temp_atm_pos[i_species][i][2];
            y_tmp = v[1][1] * temp_atm_pos[i_species][i][0]
               + v[2][1] * temp_atm_pos[i_species][i][1]
               + v[3][1] * temp_atm_pos[i_species][i][2];
            z_tmp = v[1][2] * temp_atm_pos[i_species][i][0]
               + v[2][2] * temp_atm_pos[i_species][i][1]
               + v[3][2] * temp_atm_pos[i_species][i][2];
         }
         temp_atm_pos[i_species][i][0] = x_tmp;
         temp_atm_pos[i_species][i][1] = y_tmp;
         temp_atm_pos[i_species][i][2] = z_tmp;
      }
      free_dmatrix(inertia, 1, 3, 1, 3);
      free_dvector(d, 1, 3);
      free_dmatrix(v, 1, 3, 1, 3);

      /* Measure geometrical dimensions of molecule. */
      x_min = x_max = temp_atm_pos[i_species][0][0];
      y_min = y_max = temp_atm_pos[i_species][0][1];
      z_min = z_max = temp_atm_pos[i_species][0][2];
      for (i = 1; i < n_atoms_per_mol[i_species]; ++i) {
         x_min = MIN(x_min, temp_atm_pos[i_species][i][0]);
         y_min = MIN(y_min, temp_atm_pos[i_species][i][1]);
         z_min = MIN(z_min, temp_atm_pos[i_species][i][2]);
         x_max = MAX(x_max, temp_atm_pos[i_species][i][0]);
         y_max = MAX(y_max, temp_atm_pos[i_species][i][1]);
         z_max = MAX(z_max, temp_atm_pos[i_species][i][2]);
      }
      x_range = x_max - x_min;
      y_range = y_max - y_min;
      z_range = z_max - z_min;

      /* Shift geometrical center of template molecule to the origin. */
      x_cg = x_min + 0.5 * x_range;
      y_cg = y_min + 0.5 * y_range;
      z_cg = z_min + 0.5 * z_range;
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
         temp_atm_pos[i_species][i][0] -= x_cg;
         temp_atm_pos[i_species][i][1] -= y_cg;
         temp_atm_pos[i_species][i][2] -= z_cg;
      }
   }
}

