/* Molecular inertia tensor routines. */

#include "build.h"

/* Calculate molecular inertia tensors. */
void inertia_tensors(int n_mols, int *mol_species, int *mol_first_atm, 
   int *n_atoms_per_mol, double *atom_mass, double *mol_mass, 
   double **rel_atom_coords, double **mol_inert_mt, double ***mol_inert_axis)
{
   int i_mol, i_species, i_atom, skip, abs, j, k, nrot, i;
   double **inertia, *d, **v;

   /* Allocate memory for temporary arrays. */
   inertia = dmatrix(1, 3, 1, 3);
   d = dvector(1, 3);
   v = dmatrix(1, 3, 1, 3);

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Calculate inertia tensor of molecule i_mol. */
      for (j = 1; j <= 3; ++j)
         for (k = 1; k <= 3; ++k)
            inertia[j][k] = 0.0;
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {
         abs = skip + i_atom;
         inertia[1][1] += atom_mass[abs]
            * (SQR(rel_atom_coords[abs][1]) + SQR(rel_atom_coords[abs][2]));
         inertia[2][2] += atom_mass[abs]
            * (SQR(rel_atom_coords[abs][0]) + SQR(rel_atom_coords[abs][2]));
         inertia[3][3] += atom_mass[abs]
            * (SQR(rel_atom_coords[abs][0]) + SQR(rel_atom_coords[abs][1]));
         inertia[1][2] -= atom_mass[abs]
            * rel_atom_coords[abs][0] * rel_atom_coords[abs][1];
         inertia[1][3] -= atom_mass[abs]
            * rel_atom_coords[abs][0] * rel_atom_coords[abs][2];
         inertia[2][3] -= atom_mass[abs]
            * rel_atom_coords[abs][1] * rel_atom_coords[abs][2];
      }
      inertia[2][1] = inertia[1][2];
      inertia[3][1] = inertia[1][3];
      inertia[3][2] = inertia[2][3];
      for (j = 1; j <= 3; ++j)
         for (k = 1; k <= 3; ++k)
            inertia[j][k] /= mol_mass[i_species];

      /* Calculate eigenvalues and eigenvectors of inertia tensor. */
      jacobi(inertia, 3, d, v, &nrot);
      eigsrt(d, v, 3);

      /* Save eigenvalues and eigenvectors of inertia tensor. */
      mol_inert_mt[i_mol][0] = d[1];
      mol_inert_mt[i_mol][1] = d[2];
      mol_inert_mt[i_mol][2] = d[3];

/*       printf (" d1 %g d2 %g d3 %g\n", d[1], d[2], d[3]);  
      for (i = 1; i < 4; ++i)
        for (j = 1; j < 4; ++j)
        printf(" v[%d][%d] = %g\n", i, j, v[i][j]);

exit(1); */

      mol_inert_axis[i_mol][0][0] = v[1][1];
      mol_inert_axis[i_mol][0][1] = v[2][1];
      mol_inert_axis[i_mol][0][2] = v[3][1];
      mol_inert_axis[i_mol][1][0] = v[1][2];
      mol_inert_axis[i_mol][1][1] = v[2][2];
      mol_inert_axis[i_mol][1][2] = v[3][2];
      mol_inert_axis[i_mol][2][0] = v[1][3];
      mol_inert_axis[i_mol][2][1] = v[2][3];
      mol_inert_axis[i_mol][2][2] = v[3][3];
/*
      for (k = 0; k <NDIM; ++k)
       printf("inertia0[%d] = %g, inertia1[%d] = %g, inertia2[%d] = %g\n", 
                 k, mol_inert_axis[0][0][k],
                 k, mol_inert_axis[0][1][k], k, mol_inert_axis[0][2][k]); */
   }

   /* Free up memory. */
   free_dmatrix(v, 1, 3, 1, 3);
   free_dvector(d, 1, 3);
   free_dmatrix(inertia, 1, 3, 1, 3);
}
